% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RCM_data.R, R/define_S4_generic_RCM.R
\name{check_RCMdata}
\alias{check_RCMdata}
\alias{RCM}
\alias{RCM,OM,list-method}
\alias{RCM,OM,Data-method}
\alias{RCM,OM,RCMdata-method}
\title{Rapid Conditioning Model (RCM)}
\usage{
check_RCMdata(RCMdata, OM, condition = c("catch", "catch2", "effort"))

RCM(OM, data, ...)

\S4method{RCM}{OM,RCMdata}(
  OM,
  data,
  condition = c("catch", "catch2", "effort"),
  selectivity = "logistic",
  s_selectivity = NULL,
  LWT = list(),
  comp_like = c("multinomial", "lognormal"),
  prior = list(),
  max_F = 3,
  cores = 1L,
  integrate = FALSE,
  mean_fit = FALSE,
  drop_nonconv = FALSE,
  drop_highF = FALSE,
  control = list(iter.max = 2e+05, eval.max = 4e+05),
  ...
)

\S4method{RCM}{OM,list}(
  OM,
  data,
  condition = c("catch", "catch2", "effort"),
  selectivity = "logistic",
  s_selectivity = NULL,
  LWT = list(),
  comp_like = c("multinomial", "lognormal"),
  ESS = c(30, 30),
  prior = list(),
  max_F = 3,
  cores = 1L,
  integrate = FALSE,
  mean_fit = FALSE,
  drop_nonconv = FALSE,
  drop_highF = FALSE,
  control = list(iter.max = 2e+05, eval.max = 4e+05),
  ...
)

\S4method{RCM}{OM,Data}(
  OM,
  data,
  condition = c("catch", "catch2", "effort"),
  selectivity = "logistic",
  s_selectivity = NULL,
  LWT = list(),
  comp_like = c("multinomial", "lognormal"),
  ESS = c(30, 30),
  prior = list(),
  max_F = 3,
  cores = 1L,
  integrate = FALSE,
  mean_fit = FALSE,
  drop_nonconv = FALSE,
  drop_highF = FALSE,
  control = list(iter.max = 2e+05, eval.max = 4e+05),
  ...
)
}
\arguments{
\item{RCMdata}{An \linkS4class{RCMdata} object.}

\item{OM}{An object of class \linkS4class{OM} that specifies natural mortality (M), growth (Linf, K, t0, a, b), stock-recruitment relationship,
steepness, maturity parameters (L50 and L50_95), standard deviation of recruitment variability (Perr), as well as index uncertainty (Iobs).}

\item{condition}{String to indicate whether the RCM is conditioned on "catch" (where F are estimated parameters), "catch2" (where F is solved internally using Newton's method),
or "effort".}

\item{data}{Data inputs formatted in a \linkS4class{RCMdata} (preferred) or \linkS4class{Data} object. 
Use of a list is deprecated. See Data section below.}

\item{...}{Other arguments to pass in for starting values of parameters and fixing parameters. See details.}

\item{selectivity}{A character vector of length nfleet to indicate \code{"logistic"}, \code{"dome"}, or \code{"free"} selectivity for each fleet in \code{Chist}.
If there is time-varying selectivity, this is a character vector of length nsel_block (see Data section below). "free" indicates independent selectivity parameters for each age,
and additional modifications for fixing selectivity parameters will likely be needed. See Additional arguments section.}

\item{s_selectivity}{A vector of length nsurvey to indicate the selectivity of the corresponding columns in \code{data$Index}. Use \code{"B"} for
total biomass, or \code{"SSB"} for spawning biomass (by default, "B" is used). Use numbers if the survey selectivity follows a fleet (corresponding to the columns in data$Chist, e.g., 1 = first fleet/column and so on).
If the survey selectivity is otherwise independent of anything else in the model, use \code{"logistic"}, \code{"dome"}, or \code{"free"} to specify the functional form of selectivity, and
see Additional arguments section for setup of survey selectivity parameters and Articles section for more information.}

\item{LWT}{A named list of likelihood weights for the RCM. See below.}

\item{comp_like}{A string indicating either \code{"multinomial"} (default) or \code{"lognormal"} distributions for the composition data.}

\item{prior}{A named list (R0, h, M, and q) to provide the mean and standard deviations of prior distributions for those parameters. R0, index q, and M priors are
lognormal (provide the mean in normal space, SD in lognormal space). Beverton-Holt steepness uses a beta prior, while Ricker steepness uses a normal prior.
For index q, provide a matrix for nsurvey rows and 2 columns (for mean and SD), with NA in rows corresponding to indices without priors. For all others, provide a length-2 vector for the mean and SD.
See Articles section for full description.}

\item{max_F}{The maximum F for any fleet in the scoping model (higher F's in the model are penalized in the objective function). See also \code{drop_highF}.}

\item{cores}{Integer for the number of CPU cores for the stock reduction analysis.}

\item{integrate}{Logical, whether to treat recruitment deviations as penalized parameters in the likelihood (FALSE) or random effects to be marginalized out of the likelihood (TRUE).}

\item{mean_fit}{Logical, whether to run an additional with mean values of life history parameters from the OM.}

\item{drop_nonconv}{Logical, whether to drop non-converged fits of the RCM, including fits where F = NA.}

\item{drop_highF}{Logical, whether to drop fits of the RCM where F = \code{max_F}.}

\item{control}{A named list of arguments (e.g, max. iterations, etc.) for optimization, to be passed to the control argument of \code{\link[stats]{nlminb}}.}

\item{ESS}{A vector of length two. A shortcut method to setting the maximum multinomial sample size of the age and length compositions. 
Not used when data are provided in a \linkS4class{RCMdata} object.}
}
\value{
An object of class \linkS4class{RCModel} (see link for description of output).

\code{check_RCMdata} returns a list of updated RCMdata object, OM, and StockPars, ObsPars, and FleetPars from the Hist object generated
from the OM.
}
\description{
Intended for conditioning operating models for MSEtool. For data-limited stocks, this function can generate a range of potential depletion scenarios inferred from sparse data.
From a historical time series of total catch or effort, and potentially age/length compositions and multiple indices of abundance, the RCM returns a range of values for depletion, selectivity,
unfished recruitment (R0), historical fishing effort, and recruitment deviations for the operating model. This is done by sampling life history parameters
provided by the user and fitting a statistical catch-at-age model (with the predicted catch equal to the observed catch).
Alternatively one can do a single model fit and sample the covariance matrix to generate an operating model with uncertainty based on the model fit.
Either a full catch (conditioned on catch) or effort (conditioned on effort) time series is needed but missing data (as NAs) are allowed for all other data types.
\code{check_RCMdata} evaluates whether the inputs in the S4 RCMdata object are correctly formatted.
}
\details{
Fleet selectivity is fixed to values sampled from \code{OM} if no age or length compositions are provided.

Survey selectivity is estimable only if \code{IAA} or \code{IAL} is provided. Otherwise, the selectivity should
be mirrored to a fleet (vulnerable biomass selectivity) or indexed to total or spawning biomass (see \code{s_selectivity}).

Parameters that were used in the fitting model are placed in the \code{RCM@OM@cpars} list.

If the operating model \code{OM} uses time-varying growth or M, then those trends will be used in the RCM as well.
Non-stationary productivity creates ambiguity in the calculation and interpretation of depletion and MSY reference points.

The easiest way to turn off time-varying growth/M is by setting: \code{OM@Msd <- OM@Linfsd <- OM@Ksd <- c(0, 0)}.

To play with alternative fits by excluding indices, for example, or other optional data, set the corresponding likelihood weight to zero. The model will still generate the inferred
index but the data won't enter the likelihood. See section on likelihood weights.
}
\section{Online Documentation}{

Several articles are available for the RCM:

\itemize{
\item \href{https://openmse.com/tutorial-rcm/}{General overview of approach}
\item \href{https://openmse.com/tutorial-rcm-eq/}{Mathematical description}
\item \href{https://openmse.com/tutorial-rcm-select/}{Setup of selectivity settings} (useful for more data-rich cases)
\item \href{https://openmse.com/features-assessment-models/5-priors/}{Description of priors}
}
}

\section{Data}{

One of indices, age compositions, or length compositions should be provided in addition to the historical catch or effort. Not all arguments
are needed to run the model (some have defaults, while others are ignored if not applicable depending on the data provided).

The \code{data} variable can be an object of class \linkS4class{RCMdata}. See help file for description of inputs.

Alternatively, the \code{data} input can be a \linkS4class{Data} S4 object which will retrieve data from the following slots:

\itemize{
\item Data@Cat - catch series (single fleet with the Data S4 object)
\item Data@Effort - effort series
\item Data@CAA - fishery age composition
\item Data@CAL, Data@CAL_mids - fishery length composition and corresponding length bins
\item Data@Ind, Data@SpInd, Data@VInd, Data@AddInd - indices of abundance
\item Data@CV_Ind, Data@CV_SpInd, Data@CV_VInd, Data@CV_AddInd - annual coefficients of variation for the corresponding indices of abundance. CVs will be converted to lognormal standard deviations.
\item Data@ML - fishery mean lengths
\item Data@AddIndV, Data@AddIndType, Data@AddIunits - Additional information for indices in Data@AddInd: selectivity and units (i.e., biomass or abundance).
}

There is no slot in the Data S4 object for the equilibrium catch/effort. These can be passed directly in the function call, i.e., \code{RCM(OM, Data, C_eq = C_eq, ...)}.

Use of a list is deprecated. For backwards compatibility, here is the list of supported entries:

\itemize{
\item Chist - A vector of historical catch, should be of length OM@nyears. If there are multiple fleets: a matrix of OM@nyears rows and nfleet columns.
Ideally, the first year of the catch series represents unfished conditions (see also \code{C_eq}).
\item C_sd - A vector or matrix of standard deviations (lognormal distribution) for the catches in \code{Chist}.
If not provided, the default is 0.01. Only used if \code{condition = "catch"}.
\item Ehist - A vector of historical effort, should be of length OM@nyears (see also \code{E_eq}).
\item Index - A vector of values of an index (of length OM@nyears). If there are multiple indices: a matrix of historical indices of abundances, with rows
indexing years and columns indexing the index.
\item I_sd - A vector or matrix of standard deviations (lognormal distribution) for the indices corresponding to the entries in \code{Index}.
If not provided, this function will use values from \code{OM@Iobs}.
\item I_type - Obsolete as of version 2.0. See \code{s_selectivity} argument.
\item CAA - Fishery age composition matrix with nyears rows and OM@maxage+1 columns. If multiple fleets: an array with dimension: nyears, OM@maxage, and nfleets.
\item CAL - Fishery length composition matrix with nyears rows and columns indexing the length bin. If multiple fleets: an array with dimension: nyears,
length bins, and nfleets.
\item MS - A vector of fishery mean size (MS, either mean length or mean weight) observations (length OM@nyears), or if multiple fleets: matrix of dimension: nyears and nfleets.
Generally, mean lengths should not be used if \code{CAL} is also provided, unless mean length and length comps are independently sampled.
\item MS_type - A character (either \code{"length"} (default) or \code{"weight"}) to denote the type of mean size data.
\item MS_cv - The coefficient of variation of the observed mean size. If there are multiple fleets, a vector of length nfleet.
Default is 0.2.
\item s_CAA - Survey age composition data, an array of dimension nyears, maxage+1, nsurvey.
\item s_CAL - Survey length composition data, an array of dimension nyears, length(length_bin), nsurvey.
\item length_bin - A vector for the midpoints of the length bins for \code{CAL} and \code{s_CAL}. All bin widths should be equal in size.
\item C_eq - A numeric vector of length nfleet for the equilibrium catch for each fleet in \code{Chist} prior to the first year of the operating model.
Zero (default) implies unfished conditions in year one. Otherwise, this is used to estimate depletion in the first year of the data. Alternatively,
if one has a full CAA matrix, one could instead estimate "artificial" rec devs to generate the initial numbers-at-age (and hence initial depletion) in the first year of the model (see additional arguments).
\item C_eq_sd - A vector of standard deviations (lognormal distribution) for the equilibrium catches in \code{C_eq}.
If not provided, the default is 0.01. Only used if \code{condition = "catch"}.
\item E_eq - The equilibrium effort for each fleet in \code{Ehist} prior to the first year of the operating model.
Zero (default) implies unfished conditions in year one. Otherwise, this is used to estimate depletion in the first year of the data.
\item abs_I - Optional, an integer vector to indicate which indices are in absolute magnitude. Use 1 to set q = 1, otherwise use 0 to estimate q.
\item I_units - Optional, an integer vector to indicate whether indices are biomass based (1) or abundance-based (0). By default, all are biomass-based.
\item age_error - Optional, a square matrix of maxage + 1 rows and columns to specify ageing error. The aa-th column assigns a proportion of the true age in the
a-th row to observed age. Thus, all rows should sum to 1. Default is an identity matrix (no ageing error).
\item sel_block - Optional, for time-varying fleet selectivity (in time blocks), a integer matrix of nyears rows and nfleet columns to assigns a selectivity function to a fleet for certain years.
}
}

\section{Additional arguments}{

For \code{RCM}, additional arguments can be passed to the model via \code{...}:

\itemize{
\item vul_par: A matrix of 3 rows and nfleet columns for starting values for fleet selectivity. The three rows correspond
to LFS (length of full selectivity), L5 (length of 5 percent selectivity), and Vmaxlen (selectivity at length Linf). By default,
the starting values are values from the OM object. If any selectivity = "free", then this matrix needs to be of maxage+1 rows where
the row specifies the selectivity at age. See Articles section.
\item ivul_par: A matrix of 3 rows and nsurvey columns for starting values for fleet selectivity. Same setup as vul_par. Values in the column are ignored
if \code{s_selectivity} is mapped to a fishing fleet (add NA placeholders in that case). 
If any \code{s_selectivity = "free"}, then this matrix needs to be of maxage+1 rows where
the row specifies the selectivity at age. 
\item log_rec_dev: A numeric vector of length nyears for the starting values of the log-recruitment deviations.
\item log_early_rec_dev: A numeric vector of length OM@maxage for the starting values of the recruitment deviations controlling the abundance-at-age in the first year of the model.
\item map_vul_par: An integer matrix of the same dimension as vul_par. This is the 'map' argument for vul_par in TMB, see \link[TMB]{MakeADFun}, which indicates whether selectivity parameters are fixed
or estimated. If an entry is \code{NA}, the corresponding parameter is fixed in the model to the starting
value. Otherwise, an integer for each independent parameter. By default, selectivity is fixed if there are no age or length composition for that fleet
or survey, otherwise estimated. Unused cells in the vul_par matrix should be given NA in the map matrix.
\item map_ivul_par: The map argument for the survey selectivity parameters (same dimension as ivul_par). Placeholder parameters should have a map value of NA.
\item map_log_early_rec_dev: A vector of length OM@maxage that indexes which recruitment deviates for the cohorts in the first year of the model are fixed (using NA) or estimated (a separate integer).
By default, no deviates are estimated (all are NA).
\item map_log_rec_dev: A vector of length OM@nyears that indexes which recruitment deviates are fixed (using NA) or estimated (a separate integer).
By default, all deviates are estimated.
\item plusgroup: Logical for whether the maximum age is a plusgroup or not. By default, TRUE.
\item fix_dome: Logical for whether the dome selectivity parameter for fleets is fixed. Used primarily for backwards compatibility, this is overridden by map_vul_par.
\item resample: Logical, whether the OM conditioning parameters (recruitment, fishing mortality, SSB, selectivity, etc.) are obtained by sampling the Hessian matrix from
a single model fit. By default FALSE. This feature requires identical biological parameters among simulations.
}
}

\section{Likelihood weights}{

\code{LWT} is an optional named list containing the likelihood weights (values >= 0) with the possible options:
\itemize{
\item Chist, CAA, CAL, MS, C_eq: A vector of length nfleet for each.
\item Index, IAA, IAL: A vector of length nsurvey for each.
}

By default, all likelihood weights are equal to one if not specified by the user.

Annual multinomial sample sizes for the age and length comps can now be provided directly in the 
\linkS4class{RCMdata} object. For a list or \linkS4class{Data} object, use the \code{ESS} argument.
}

\examples{
\donttest{ 
# An example that conditions a Pacific cod operating model. There are 48 simulations, 
# where values of natural mortality and steepness are sampled from distributions. 
# The model is fitted with priors on the index catchability. Maturity and selectivity 
# are knife-edge at the age of 2 years. See online tutorial for more information.

data(pcod) 
mat_ogive <- pcod$OM@cpars$Mat_age[1, , 1]
out <- RCM(OM = pcod$OM, data = pcod$data, 
           condition = "catch", mean_fit = TRUE,
           selectivity = "free", s_selectivity = rep("SSB", ncol(pcod$data@Index)),
           vul_par = matrix(mat_ogive, length(mat_ogive), 1),
           map_vul_par = matrix(NA, length(mat_ogive), 1),
           map_log_early_rec_dev = rep(1, pcod$OM@maxage),
           prior = pcod$prior)
plot(out, s_name = colnames(pcod$data@Index))

# Alternative OM with age-3 maturity and selectivity instead.
out_age3 <- local({
  pcod$OM@cpars$Mat_age[, 2, ] <- 0
  mat_ogive_age3 <- pcod$OM@cpars$Mat_age[1, , 1]
  RCM(OM = pcod$OM, data = pcod$data, 
      condition = "catch", mean_fit = TRUE,
      selectivity = "free", s_selectivity = rep("SSB", ncol(pcod$data@Index)),
      vul_par = matrix(mat_ogive_age3, length(mat_ogive_age3), 1),
      map_vul_par = matrix(NA, length(mat_ogive_age3), 1),
      map_log_early_rec_dev = rep(1, pcod$OM@maxage),
      prior = pcod$prior)
})
  
compare_RCM(out, out_age3, scenario = list(names = c("Age-2 maturity", "Age-3 maturity")),
            s_name = colnames(pcod$data@Index))
} 
}
\seealso{
\link{plot.RCModel} \linkS4class{RCModel} \link{compare_RCM} \link{pcod}
}
\author{
Q. Huynh
}
