#' Calculates the  amount of the addon for each heding/nettting set
#' 
#' @title Calculates the Addon amount
#' @param trades_tree A tree structure with the input trades
#' @param MF (Optional) The Maturity Factor based on the collateral agreement  
#' @return The aggregate amount of the addon summed up for all the asset classes
#' @export
#' @author Tasos Grivas <tasos@@openriskcalculator.com>
#' @references Basel Committee: The standardised approach for measuring counterparty credit risk exposures
#' http://www.bis.org/publ/bcbs279.htm
#' 
CalcAddon <- function(trades_tree, MF)  {
  ## function which will calculate the Add-On for all the trade classes
  
  superv <- LoadSupervisoryData()
  
  asset_classes = trades_tree$children
  asset_class_names = names(asset_classes)
  trades_tree$addon = 0
  # going through each trade class
  for (i in 1:length(asset_class_names))
  { 
    #picking up the trades belonging to this specific trade class
    asset_class_node = asset_classes[[asset_class_names[i]]]
    asset_class_node$addon = 0
    
    if(asset_class_names[i]=='FX')
    {
      ccypairs   <- asset_class_node$children
      # for the FX case the Hedging sets will be created based on the ccy pair
      ccypairs_names  <- names(ccypairs)
      ccypairs_addon <- array(data<-0,dim<-length(ccypairs))
      # going through all the ccy pairs found
      for (j  in 1:length(ccypairs_names))
      {
        
        ccypairs_trades  <- ccypairs[[ccypairs_names[j]]]$children
        ccypairs[[ccypairs_names[j]]]$add_on = 0
        ccypairs_trades_names = names(ccypairs_trades)
        
        # for each of the trade calculate the Adjusted Notional and their contribution to the addon of the hedging set
        for (l in 1:length(ccypairs_trades_names))
        {
          ccypairs_trade = ccypairs_trades[[ccypairs_trades_names[l]]]
          ccypairs_trade$exposure_details = SingleTradeAddon(ccypairs_trade$trade,MF)
          # aggregating the add-on contribution for a specific hedging set
          ccypairs[[ccypairs_names[j]]]$add_on <- ccypairs[[ccypairs_names[j]]]$add_on + ccypairs_trade$exposure_details$addon
        }
        factor_mult = CalculateFactorMult(ccypairs_names[j])
        # getting the supervisory factor
        supervisory_factor <- factor_mult*superv$Supervisory_factor[superv$Asset_Class==ccypairs_trade$trade$TradeGroup]
        ccypairs_addon[j] = supervisory_factor*ccypairs[[ccypairs_names[j]]]$add_on
      }
      asset_class_node$addon = sum(abs(ccypairs_addon))
      trades_tree$addon = trades_tree$addon + asset_class_node$addon
    }  else if(asset_class_names[i]=='IRD')
    {
      
      # picking up the currencies found in the IRD trades which will be the first-level grouping applied 
      currencies   <-  asset_class_node$children
      currencies_names  <- names(currencies)
      for (j  in 1:length(currencies_names))
      {
        
        currencies_buckets  <- currencies[[currencies_names[j]]]$children
        
        currencies_buckets_names = names(currencies_buckets)
        
        for (k in 1:length(currencies_buckets_names))
        {
          currencies_buckets[[currencies_buckets_names[k]]]$addon = 0
          #picking up all the trades belonging to a specific timebucket
          timebuckets_trades  <- currencies_buckets[[currencies_buckets_names[k]]]$children
          
          timebuckets_trades_names = names(timebuckets_trades)
          
          for (l in 1:length(timebuckets_trades_names))
          {
            timebuckets_trade = timebuckets_trades[[timebuckets_trades_names[l]]]
            timebuckets_trade$exposure_details = SingleTradeAddon(timebuckets_trade$trade,MF)
            currencies_buckets[[currencies_buckets_names[k]]]$addon = currencies_buckets[[currencies_buckets_names[k]]]$addon + timebuckets_trade$exposure_details$addon
          }
          factor_mult = CalculateFactorMult(currencies_names[j])
          
          supervisory_factor <- factor_mult*superv$Supervisory_factor[superv$Asset_Class==timebuckets_trade$trade$TradeGroup&superv$SubClass==timebuckets_trade$trade$SubClass]
          
          # aggregating the add-on timebuckets recognizing correlation between each time bucket  
          currencies[[currencies_names[j]]]$addon <- supervisory_factor*(sum(currencies_buckets[["1"]]$addon^2,currencies_buckets[["2"]]$addon^2,currencies_buckets[["3"]]$addon^2,1.4*currencies_buckets[["2"]]$addon*currencies_buckets[["3"]]$addon,1.4*currencies_buckets[["2"]]$addon*currencies_buckets[["1"]]$addon,0.6*currencies_buckets[["2"]]$addon*currencies_buckets[["1"]]$addon,na.rm = TRUE))^0.5
          
        }
        # adding up the addon of each currency after multiplying with the supervisory factor
        asset_class_node$addon <- asset_class_node$addon + currencies[[currencies_names[j]]]$addon
        
       
      }
      trades_tree$addon = trades_tree$addon + asset_class_node$addon
    }else  if(asset_class_names[i]=='Credit')
    {
      asset_class_node$addon = 0
      refEntities   <- asset_class_node$children
      # for the FX case the Hedging sets will be created based on the ccy pair
      refEntities_names  <- names(refEntities)
      refEntities_addon <- array(data<-0,dim<-length(refEntities))
      
      # for the Credit case the Hedging sets will be created based on the reference entity
      supervisory_corel <- array(data<-0,dim<-length(refEntities_names))
      for (j  in 1:length(refEntities_names))
      {  
        refEntities_trades  <- refEntities[[refEntities_names[j]]]$children
        refEntities[[refEntities_names[j]]]$add_on = 0
        
        refEntities_trades_names = names(refEntities_trades)        
        for (k in 1:length(refEntities_trades_names))
        {
          refEntities_trade = refEntities_trades[[refEntities_trades_names[k]]]
          refEntities_trade$exposure_details = SingleTradeAddon(refEntities_trade$trade,MF)
          refEntities[[refEntities_names[j]]]$add_on <- refEntities[[refEntities_names[j]]]$add_on  + refEntities_trade$exposure_details$addon
        }
        factor_mult = CalculateFactorMult(refEntities_names[j])
        AssetClass<-paste(refEntities_trade$trade$TradeGroup,refEntities_trade$trade$TradeType,sep="")
        supervisory_factor <- factor_mult*superv$Supervisory_factor[superv$Asset_Class==AssetClass&superv$SubClass==refEntities_trade$trade$SubClass]
        refEntities[[refEntities_names[j]]]$add_on <- refEntities[[refEntities_names[j]]]$add_on*supervisory_factor
        supervisory_corel[j]  <- superv$Correlation[superv$Asset_Class==AssetClass&superv$SubClass==refEntities_trade$trade$SubClass]
        refEntities_addon[j] = refEntities[[refEntities_names[j]]]$add_on
      }
      systematic_component     <- (sum(refEntities_addon*supervisory_corel))^2
      idiosynchratic_component <-  sum((rep(1,length(refEntities))-supervisory_corel^2)*refEntities_addon^2)
      asset_class_node$addon   <- sqrt(systematic_component + idiosynchratic_component)
      trades_tree$addon = trades_tree$addon + asset_class_node$addon
    }else   if(asset_class_names[i]=='Commodity')
    {
      AssetClass <-'Commodity'
      
      HedgingSets   <- asset_class_node$children
      # for the FX case the Hedging sets will be created based on the ccy pair
      HedgingSets_names  <- names(HedgingSets)
      
      # for the commodities case the Hedging sets will be created based on the commodity sector on a first
      # level (energy,metals etc) and on a second level based on the actual commodities (oil, silver etc)
      HedgingSets_addon = 0
      
      for (j  in 1:length(HedgingSets_names))
      {
        
        com_types  <- HedgingSets[[HedgingSets_names[j]]]$children
        
        com_types_names = names(com_types)
        com_types_addon = 0
        com_types_addon_sq = 0
        for (k in 1:length(com_types_names))
        {
          com_types[[com_types_names[k]]]$addon = 0
          com_types_trades  <- com_types[[com_types_names[k]]]$children
          com_types_trades_names  <- names(com_types_trades)
          
          for (l in 1:length(com_types_trades_names))
          {
            com_types_trade = com_types_trades[[com_types_trades_names[l]]]
            com_types_trade$exposure_details = SingleTradeAddon(com_types_trade$trade,MF)
            # aggregating the add-on contribution for a specific hedging set
            com_types[[com_types_names[k]]]$addon <- com_types[[com_types_names[k]]]$addon + com_types_trade$exposure_details$addon
          }
          factor_mult = CalculateFactorMult(HedgingSets_names[j])
          supervisory_factor <- factor_mult*superv$Supervisory_factor[superv$Asset_Class==AssetClass&(superv$SubClass==com_types_trade$trade$SubClass|superv$SubClass==com_types_trade$trade$commodity_type)]
          com_types[[com_types_names[k]]]$addon <- com_types[[com_types_names[k]]]$addon*supervisory_factor
          com_types_addon = com_types_addon + com_types[[com_types_names[k]]]$addon 
          com_types_addon_sq = com_types_addon_sq + com_types[[com_types_names[k]]]$addon^2
        }
        supervisory_corel     <- superv$Correlation[superv$Asset_Class==AssetClass&(superv$SubClass==com_types_trade$trade$SubClass|superv$SubClass==com_types_trade$trade$commodity_type)]
        HedgingSets[[HedgingSets_names[j]]]$addon  <- sqrt((com_types_addon*supervisory_corel)^2 + (1-supervisory_corel^2)*com_types_addon_sq)
        HedgingSets_addon = HedgingSets_addon + HedgingSets[[HedgingSets_names[j]]]$addon
      }
      asset_class_node$addon <- asset_class_node$addon + HedgingSets_addon
      trades_tree$addon = trades_tree$addon + asset_class_node$addon
    }else  if(asset_class_names[i]=='EQ')
    {
      asset_class_node$addon = 0
      refEntities   <- asset_class_node$children
      # for the FX case the Hedging sets will be created based on the ccy pair
      refEntities_names  <- names(refEntities)
      refEntities_addon <- array(data<-0,dim<-length(refEntities))
      
      # for the Equity case the Hedging sets will be created based on the reference entity
      supervisory_corel <- array(data<-0,dim<-length(refEntities_names))
      for (j  in 1:length(refEntities_names))
      {  
        refEntities_trades  <- refEntities[[refEntities_names[j]]]$children
        refEntities[[refEntities_names[j]]]$add_on = 0
        
        refEntities_trades_names = names(refEntities_trades)        
        for (k in 1:length(refEntities_trades_names))
        {
          refEntities_trade = refEntities_trades[[refEntities_trades_names[k]]]
          refEntities_trade$exposure_details = SingleTradeAddon(refEntities_trade$trade,MF)
          refEntities[[refEntities_names[j]]]$add_on <- refEntities[[refEntities_names[j]]]$add_on  + refEntities_trade$exposure_details$addon
        }
        factor_mult = CalculateFactorMult(refEntities_names[j])
        AssetClass<-paste(refEntities_trade$trade$TradeGroup,refEntities_trade$trade$TradeType,sep="")
        supervisory_factor <- factor_mult*superv$Supervisory_factor[superv$Asset_Class==AssetClass&superv$SubClass==refEntities_trade$trade$SubClass]
        refEntities[[refEntities_names[j]]]$add_on <- refEntities[[refEntities_names[j]]]$add_on*supervisory_factor
        supervisory_corel[j]  <- superv$Correlation[superv$Asset_Class==AssetClass&superv$SubClass==refEntities_trade$trade$SubClass]
        refEntities_addon[j] = refEntities[[refEntities_names[j]]]$add_on
      }
      systematic_component     <- (sum(refEntities_addon*supervisory_corel))^2
      idiosynchratic_component <-  sum((rep(1,length(refEntities))-supervisory_corel^2)*refEntities_addon^2)
      asset_class_node$addon   <- sqrt(systematic_component + idiosynchratic_component)
      trades_tree$addon = trades_tree$addon + asset_class_node$addon
    }  
  }
  return(trades_tree)
}

