% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/covEstimation.R
\name{covEstimation}
\alias{covEstimation}
\title{Covariance matrix estimation}
\usage{
covEstimation(rets, control = list())
}
\arguments{
\item{rets}{a matrix \eqn{(T \times N)}{(T x N)} of returns.}

\item{control}{control parameters (see *Details*).}
}
\value{
A \eqn{(N \times N)}{(N x N)} covariance matrix.
}
\description{
Function which performs various estimations of covariance matrices.
}
\details{
The argument \code{control} is a list that can supply any of the following
components: 
\itemize{
\item \code{type} method used to compute the
covariance matrix, among \code{'naive'}, \code{'ewma'}, \code{'lw'},
\code{'factor'},\code{'const'}, \code{'cor'}, \code{'oneparm'},
\code{'diag'} and \code{'large'} where: 

\code{'naive'} is used to compute 
the naive (standard) covariance matrix. 

\code{'ewma'} is used to compute the exponential weighting moving average covariance matrix. The following formula is used 
to compute the ewma covariance matrix:
\deqn{\Sigma_t := \lambda \Sigma_{t-1} + (1-\lambda)r_{t-1}r_{t-1}}{Sigma[t]
:= lambda * Sigma[t-1] + (1-lambda) r[t-1]'r[t-1]}
where \eqn{r_t} is the \eqn{(N \times 1)}{(N x 1)} vector of returns at time
\eqn{t}. Note that the data must be sorted from the oldest to the latest. See RiskMetrics (1996)

\code{'factor'} is used to compute the covariance matrix estimation using a
K-factor approach. See Harman (1976).

\code{'lw'} is a weighted average of the sample covariance matrix and a
'prior' or 'shrinkage target'. The prior is given by a one-factor model and
the factor is equal to the cross-sectional average of all the random
variables. See Ledoit and Wolf (2003).

\code{'const'} is a weighted average of the sample covariance matrix and a
'prior' or 'shrinkage target'.  The prior is given by constant correlation
matrix. See Ledoit and Wolf (2002).

\code{'cor'} is a weighted average of the sample covariance matrix and a
'prior' or 'shrinkage target'.  The prior is given by the constant
correlation covariance matrix given by Ledoit and Wolf (2003).

\code{'oneparm'} is a weighted average of the sample covariance matrix and a
'prior' or 'shrinkage target'.  The prior is given by the one-parameter
matrix. All variances are the same and all covariances are zero. 
See Ledoit and Wolf (2004).

\code{'diag'} is a weighted average of the sample covariance matrix and a
'prior' or 'shrinkage target'.  The prior is given by a diagonal matrix. 
See Ledoit and Wolf (2002).

\code{'large'} This estimator is a weighted average of the sample covariance
matrix and a 'prior' or 'shrinkage target'. Here, the prior is given by a
one-factor model. The factor is equal to the cross-sectional average of all
the random variables. The weight, or 'shrinkage intensity' is chosen to
minimize quadratic loss measured by the Frobenius norm. The estimator is
valid as the number of variables and/or the number of observations go to
infinity, but Monte-Carlo simulations show that it works well for values as
low as 10. The main advantage is that this estimator is guaranteed to be
invertible and well-conditioned even if variables outnumber observations. 
See Ledoit and Wolf (2004).

\code{'bs'} is the Bayes-Stein estimator for the covariance matrix given by
Jorion (1986).

Default: \code{type = 'naive'}.

\item \code{lambda} decay parameter. Default: \code{lambda = 0.94}.

\item \code{K} number of factors to use when the K-factor approach is
chosen to estimate the covariance matrix. Default: \code{K = 1}.}
}
\note{
Part of the code is adapted from the Matlab code by Ledoit and Wolf (2014).
}
\examples{
# Load returns of assets or portfolios
data("Industry_10")
rets = Industry_10

# Naive covariance estimation
covEstimation(rets)

# Ewma estimation of the covariance with default lambda = 0.94
covEstimation(rets, control = list(type = 'ewma'))

# Ewma estimation of the covariance with default lambda = 0.90
covEstimation(rets, control = list(type = 'ewma', lambda = 0.9))

# Factor estimation of the covariance with dafault K = 1
covEstimation(rets, control = list(type = 'factor'))

# Factor estimation of the covariance with K = 3
covEstimation(rets, control = list(type = 'factor', K = 3))

# Ledot-Wolf's estimation of the covariance
covEstimation(rets, control = list(type = 'lw'))

# Shrinkage of the covariance matrix using constant correlation matrix
covEstimation(rets, control = list(type = 'const'))

# Shrinkage of the covariance matrix towards constant correlation matrix by
# Ledoit-Wolf.
covEstimation(rets, control = list(type = 'cor'))

# Shrinkage of the covariance matrix towards one-parameter matrix
covEstimation(rets, control = list(type = 'oneparm'))

# Shrinkage of the covaraince matrix towards diagonal matrix
covEstimation(rets, control = list(type = 'diag'))

# Shrinkage of the covariance matrix for large data set
covEstimation(rets, control = list(type = 'large'))
}
\author{
David Ardia, Kris Boudt and Jean-Philippe Gagnon Fleury.
}
\references{
Jorion, P. (1986). 
Bayes-Stein estimation for portfolio analysis.
\emph{Journal of Financial and Quantitative Analysis} \bold{21}(3), pp.279--292. 
\doi{10.2307/2331042}

Harman, H.H. (1976)
\emph{Modern Factor Analysis}. 
3rd Ed. Chicago: University of Chicago Press.

Ledoit, O., Wolf, M. (2002).  
Improved estimation of the covariance matrix of stock returns with an application to portfolio selection. 
\emph{Journal of Empirical Finance} \bold{10}(5), pp.603--621. 
\doi{10.1016/S0927-5398(03)00007-0}

Ledoit, O., Wolf, M. (2003).  
Honey, I Shrunk the Sample Covariance Matrix.
\emph{Journal of Portfolio Management} \bold{30}(4), pp.110--119. 
\doi{10.3905/jpm.2004.110}

Ledoit, O., Wolf, M. (2004).  
A well-conditioned estimator for large-dimensional covariance matrices.
\emph{Journal of Multivariate Analysis} \bold{88}(2), pp.365--411. 
\doi{10.1016/S0047-259X(03)00096-4}

RiskMetrics (1996)
\emph{RiskMetrics Technical Document}.
J. P. Morgan/Reuters. 
\url{https://www.msci.com/documents/10199/5915b101-4206-4ba0-aee2-3449d5c7e95a}
}
\keyword{htest}

