## ----output='hide'------------------------------------------------------------
require(Rdistance)

## ----setup--------------------------------------------------------------------
data("sparrowDetectionData")
data("sparrowSiteData")

## ----detectionDataDisplay-----------------------------------------------------
head(sparrowDetectionData)

## ----siteDataDisplay----------------------------------------------------------
head(sparrowSiteData)

## ----rawDistanceHistogram, fig.width=6, fig.height=4, fig.cap="Histogram of sparrow off-transect detection distances."----
hist(sparrowDetectionData$dist
     , col="grey"
     , main=""
     , xlab = "Distance")
rug(sparrowDetectionData$dist,quiet = TRUE)

## ----rawDistanceSummary-------------------------------------------------------
summary(sparrowDetectionData$dist)

## ----dfuncEstimation----------------------------------------------------------
rightTruncDistance <- units::set_units(150, "m")
dfuncSparrow<- dfuncEstim(formula = dist ~ 1 + offset(groupsize)
                          , detectionData = sparrowDetectionData
                          , likelihood = "halfnorm"
                          , w.hi = rightTruncDistance)
dfuncSparrow

## ----fig-dfuncPlot, fig.width=6, fig.height=4, fig.cap="The half-normal distance function fitted to sparrow off-transect detection distances."----
# Figure 2
plot(dfuncSparrow, nbins =40, col="grey")

## ----studyAreaSize, include=FALSE---------------------------------------------
# NOT INCLUDED IN VIGNETTE
saSize <- units::set_units(4105, "km^2")
set.seed(8927342)

## ----abundEstim, include=FALSE------------------------------------------------
# To save vignette build time, we run it with ci=NULL and insert values from 
# a separate run with R=500
fit <- abundEstim(dfuncSparrow
                , detectionData = sparrowDetectionData
                , siteData = sparrowSiteData
                , area = saSize
                , ci = NULL
                )
fit$n.hat.ci <-   c( `2.5%` = 268646.3 
                   , `97.5%` = 412088 )
fit$density.ci <- units::set_units(c( 6.544368e-05, 0.0001003868 ), "1/m^2")
fit$effDistance.ci <- units::set_units(c(56.66586, 68.72937 ), "m")
fit$nItersConverged <- 500
fit$alpha <- 0.95
B.d <- c(
7.41186193094976e-05 ,  8.69775946939212e-05 ,  8.20995278909548e-05 ,  8.24824091338752e-05 ,  9.8337763002546e-05 ,  9.26031826522007e-05 ,  6.04364659611598e-05 ,  8.71601415352838e-05 ,  8.40262756115299e-05 ,  7.7231071824023e-05 ,  9.11710914168955e-05 ,  9.2903836882483e-05 ,  6.80190749696935e-05 ,  7.24194771707422e-05 ,  8.03752068184811e-05 ,  7.10616504662341e-05 ,  8.87211875789235e-05 ,  7.77797903678722e-05 ,  9.70539888565341e-05 ,  6.91494234014299e-05 ,  5.99323466448954e-05 ,  9.1536426090801e-05 ,  8.78047152332687e-05 ,  8.240447497912e-05 ,  7.72538426316594e-05 ,  9.00281157467167e-05 ,  6.79047145642239e-05 ,  7.3269216589717e-05 ,  9.50687068382639e-05 ,  8.69559685519084e-05 ,  7.86417398636142e-05 ,  8.38757670358632e-05 ,  9.32955821610548e-05 ,  8.34866913495973e-05 ,  8.45745109724636e-05 ,  9.40384432934679e-05 ,  7.74584770226755e-05 ,  7.93460986097706e-05 ,  8.49367942080276e-05 ,  8.7762782032957e-05 ,  0.000107336477382257 ,  7.49378886521039e-05 ,  8.64341528033551e-05 ,  7.97889053472375e-05 ,  9.82845151606708e-05 ,  6.75369876955574e-05 ,  7.94871587715466e-05 ,  9.28890954855497e-05 ,  8.17756108416586e-05 ,  8.13518485015131e-05 ,  9.23380101515964e-05 ,  6.38674815032795e-05 ,  7.4019104504842e-05 ,  8.76412066853059e-05 ,  7.3134988616935e-05 ,  7.80253387361734e-05 ,  6.98590861109206e-05 ,  8.37681709997682e-05 ,  7.78681183406401e-05 ,  7.75360888613941e-05 ,  8.57628854489922e-05 ,  7.50104807254164e-05 ,  9.70816294988619e-05 ,  8.65005709839268e-05 ,  9.10233045157741e-05 ,  8.65089098348133e-05 ,  7.86196622774781e-05 ,  7.30487925477049e-05 ,  7.18769800268041e-05 ,  7.10210407671115e-05 ,  9.54275285749659e-05 ,  8.00591983079024e-05 ,  7.82397156022772e-05 ,  8.54669971830502e-05 ,  9.10098078983285e-05 ,  7.61447451365422e-05 ,  7.41766138615937e-05 ,  8.32873458731774e-05 ,  9.72387260595565e-05 ,  8.80288813357399e-05 ,  8.76201986956681e-05 ,  8.67431742854634e-05 ,  6.20300640936345e-05 ,  7.46761669713478e-05 ,  8.65846621379941e-05 ,  7.39459571082965e-05 ,  7.59675056903756e-05 ,  8.06379952300787e-05 ,  8.44384485429816e-05 ,  7.93605156635565e-05 ,  8.78797654717914e-05 ,  8.13040086387347e-05 ,  8.10778220209004e-05 ,  8.70121599421275e-05 ,  0.000109439576130608 ,  9.02348628721857e-05 ,  7.85842207560927e-05 ,  7.6337523898567e-05 ,  7.31013121262569e-05 ,  9.6084063609991e-05 ,  8.05140795975154e-05 ,  7.5520113016103e-05 ,  7.62595381278356e-05 ,  8.11812033705672e-05 ,  9.55471854262335e-05 ,  8.24751175371162e-05 ,  7.99379602452841e-05 ,  7.9791817578694e-05 ,  7.46026214166558e-05 ,  9.25099706143335e-05 ,  7.03852253456436e-05 ,  8.97392059110786e-05 ,  9.10088741706802e-05 ,  7.4827800563782e-05 ,  8.67447212282118e-05 ,  7.652474453691e-05 ,  7.36150525872533e-05 ,  7.78963015251251e-05 ,  7.80880365777775e-05 ,  7.33254920828694e-05 ,  7.14432856399823e-05 ,  9.07093463579799e-05 ,  7.20861284165556e-05 ,  7.71881472063593e-05 ,  9.91233069543969e-05 ,  7.07914575676011e-05 ,  9.50389888737708e-05 ,  8.9389729492371e-05 ,  7.57369453485519e-05 ,  9.00023130347773e-05 ,  7.77713365398204e-05 ,  8.26355817317967e-05 ,  7.83727221099925e-05 ,  7.4344935479553e-05 ,  8.38312644685292e-05 ,  7.38907412191759e-05 ,  8.28064211688874e-05 ,  8.56200556855738e-05 ,  8.54118422450951e-05 ,  8.0846577222901e-05 ,  6.66306956925327e-05 ,  8.42745451068001e-05 ,  9.22676510649875e-05 ,  9.05812587755955e-05 ,  8.34443403975103e-05 ,  7.2468669576973e-05 ,  8.60217034370583e-05 ,  8.87803189671566e-05 ,  8.40572435161317e-05 ,  8.2375515608265e-05 ,  8.72089493672299e-05 ,  8.92386292037965e-05 ,  7.7321706664353e-05 ,  9.48539369313877e-05 ,  7.82325655422112e-05 ,  7.88587311194897e-05 ,  6.12971568684862e-05 ,  9.86676651983264e-05 ,  8.44782180790178e-05 ,  7.25005571717218e-05 ,  9.31506533512914e-05 ,  7.68451826075576e-05 ,  9.61352779273209e-05 ,  8.22535433620533e-05 ,  8.88401988898716e-05 ,  5.7368655141976e-05 ,  9.43683368766725e-05 ,  9.46919138935022e-05 ,  7.83062823497553e-05 ,  8.66341331744823e-05 ,  8.03358088558252e-05 ,  8.02954426552289e-05 ,  9.29002348362239e-05 ,  9.76140481236503e-05 ,  6.99897680000816e-05 ,  6.84325902659195e-05 ,  7.10052585166883e-05 ,  9.17103117701564e-05 ,  8.47553335038473e-05 ,  0.000100195318601671 ,  7.39411305811077e-05 ,  8.73737054151313e-05 ,  6.54262607944402e-05 ,  8.26005372279447e-05 ,  9.40766966415021e-05 ,  8.40570139394394e-05 ,  8.32389988846452e-05 ,  0.000105555849210801 ,  8.43379151549714e-05 ,  8.6542776584437e-05 ,  8.13502321093254e-05 ,  7.49586983063665e-05 ,  8.60321222243637e-05 ,  5.90080020631661e-05 ,  7.39450631003699e-05 ,  6.93419810588636e-05 ,  8.59061475332039e-05 ,  6.12425878456712e-05 ,  6.84278169812998e-05 ,  7.48341249018789e-05 ,  8.54687808287217e-05 ,  7.95429438968889e-05 ,  7.67924558644659e-05 ,  7.73521486583513e-05 ,  8.28847804526009e-05 ,  0.000103560867467733 ,  7.35567566801547e-05 ,  7.17223867623591e-05 ,  8.38685287383189e-05 ,  9.35723341360076e-05 ,  0.000103487040907377 ,  7.78562335695492e-05 ,  8.70121124031759e-05 ,  8.29740661742719e-05 ,  8.79164504169475e-05 ,  8.58233488548016e-05 ,  9.58092656886269e-05 ,  0.000100469782260996 ,  8.83972477686073e-05 ,  7.61138460936409e-05 ,  7.41559243482071e-05 ,  9.50748630110072e-05 ,  8.37040929238613e-05 ,  8.37502369717202e-05 ,  6.65219215914592e-05 ,  7.9934785649973e-05 ,  9.7495805701392e-05 ,  8.00119680015466e-05 ,  8.76766403314432e-05 ,  7.66475173007073e-05 ,  7.39334220092162e-05 ,  7.77286392114712e-05 ,  8.01467116403014e-05 ,  8.16367333916679e-05 ,  8.12739502821282e-05 ,  7.63775631356292e-05 ,  7.71471577903636e-05 ,  8.26474083140042e-05 ,  8.81594593144686e-05 ,  9.07093226056759e-05 ,  8.68499670774804e-05 ,  7.60290476847027e-05 ,  8.54993686045404e-05 ,  8.19956934672603e-05 ,  7.67235332473808e-05 ,  8.49506178730528e-05 ,  9.09212568982495e-05 ,  7.06884108073003e-05 ,  7.53803529942619e-05 ,  7.83857379355414e-05 ,  9.56518644168316e-05 ,  7.90241666126352e-05 ,  7.88646530360321e-05 ,  8.41433018135625e-05 ,  8.00842194741539e-05 ,  7.40798429278931e-05 ,  9.94632749485314e-05 ,  8.21821050635686e-05 ,  9.59236458638392e-05 ,  8.53279562468714e-05 ,  9.41072707955564e-05 ,  0.000107585906111944 ,  9.58321352931763e-05 ,  7.7439721280841e-05 ,  7.18625161107672e-05 ,  9.56393489464636e-05 ,  9.24861423375279e-05 ,  8.44078039797996e-05 ,  8.5148962189532e-05 ,  7.66263043773153e-05 ,  8.76612230454225e-05 ,  9.14827373707543e-05 ,  7.94684501067572e-05 ,  7.28743986739702e-05 ,  7.72482274192602e-05 ,  7.37027389458352e-05 ,  9.44379855967609e-05 ,  8.21969343771601e-05 ,  8.67810817102775e-05 ,  9.72492676478023e-05 ,  8.93332637979939e-05 ,  9.52469081127755e-05 ,  6.56700770340861e-05 ,  8.45271456349873e-05 ,  8.83605343954807e-05 ,  9.02043486970886e-05 ,  8.73651069539185e-05 ,  8.508876846241e-05 ,  7.03088135389129e-05 ,  7.85725264832678e-05 ,  9.4713714398389e-05 ,  8.36368956120976e-05 ,  8.03359790618387e-05 ,  6.78381960132318e-05 ,  8.47645824967194e-05 ,  9.84713650120266e-05 ,  8.71327376693591e-05 ,  9.99008375568115e-05 ,  9.18158734348995e-05 ,  7.44317326166889e-05 ,  8.80393977060951e-05 ,  7.83806074474559e-05 ,  8.82399550332299e-05 ,  7.72577952274759e-05 ,  8.92415967635025e-05 ,  6.28835202470605e-05 ,  8.50915904311932e-05 ,  9.69985789857186e-05 ,  8.55295649855376e-05 ,  8.75082603362225e-05 ,  9.09316582475909e-05 ,  6.99588860380832e-05 ,  6.84776195851645e-05 ,  7.52833852763376e-05 ,  7.225872427367e-05 ,  8.67932301139964e-05 ,  9.5003097267605e-05 ,  8.01043145281e-05 ,  9.3510847981782e-05 ,  9.56071309916164e-05 ,  8.57934788672115e-05 ,  7.47832403418488e-05 ,  6.74771838987688e-05 ,  9.13139661027117e-05 ,  7.45271960196292e-05 ,  8.83462068848155e-05 ,  7.02653252639893e-05 ,  8.61167518591516e-05 ,  7.71260663165087e-05 ,  8.23212807087248e-05 ,  9.2101312466067e-05 ,  7.69018658603984e-05 ,  9.47705827026236e-05 ,  7.97018966409744e-05 ,  8.07979021064315e-05 ,  0.000107806118230997 ,  7.27795558608282e-05 ,  8.02626336352446e-05 ,  6.96139498431645e-05 ,  8.13489238821347e-05 ,  0.000100074515147317 ,  6.37307062497307e-05 ,  9.09590806243367e-05 ,  8.81045593797417e-05 ,  8.60951138243968e-05 ,  6.69747263733447e-05 ,  9.59602406644013e-05 ,  6.089626189937e-05 ,  7.99371274447994e-05 ,  8.37424323477484e-05 ,  7.47609312801187e-05 ,  8.99404598079024e-05 ,  7.32368161103617e-05 ,  9.88843652882384e-05 ,  8.23776785471916e-05 ,  6.67366287584427e-05 ,  0.000107893158484432 ,  9.3846129534546e-05 ,  8.5569952656581e-05 ,  8.77292828376391e-05 ,  7.33752225301726e-05 ,  7.86228631706859e-05 ,  7.77390006408133e-05 ,  9.42718031524649e-05 ,  6.89460012523336e-05 ,  9.18648659179194e-05 ,  7.96042544183538e-05 ,  7.51937979451023e-05 ,  7.4181318838139e-05 ,  8.11692946743056e-05 ,  8.12358230606766e-05 ,  8.40251333599078e-05 ,  0.000100377778993875 ,  7.79467471730603e-05 ,  7.38607825234405e-05 ,  8.2059461695045e-05 ,  6.88004436658658e-05 ,  7.85852939328257e-05 ,  8.99553099266676e-05 ,  7.415469248303e-05 ,  7.09710694617005e-05 ,  8.59555409409972e-05 ,  6.71360512752293e-05 ,  8.32934693406917e-05 ,  8.58784043875092e-05 ,  9.17829779917562e-05 ,  9.52600072349062e-05 ,  8.17004142369388e-05 ,  8.67766066848941e-05 ,  9.71279798704004e-05 ,  8.0995587348777e-05 ,  7.7433104020976e-05 ,  8.61724602351863e-05 ,  8.25470306643349e-05 ,  8.84510552946986e-05 ,  7.25418231680128e-05 ,  8.47028778539682e-05 ,  6.94509323024331e-05 ,  8.5718140561333e-05 ,  7.8770814729392e-05 ,  9.67530233591282e-05 ,  8.97585617552819e-05 ,  9.21691678591068e-05 ,  6.65727821013571e-05 ,  7.60021325659079e-05 ,  8.12478621728794e-05 ,  8.59958792746685e-05 ,  8.36960043581442e-05 ,  7.01575042429371e-05 ,  8.57942585361146e-05 ,  6.54113045312806e-05 ,  8.19137305843735e-05 ,  7.6855594786111e-05 ,  9.29754243337834e-05 ,  8.69577157086979e-05 ,  7.74688755886869e-05 ,  7.7990562557043e-05 ,  8.22744847206398e-05 ,  7.79709778190657e-05 ,  7.61529124722675e-05 ,  9.50882820408906e-05 ,  0.000102034551740795 ,  7.22040282241486e-05 ,  8.34854827996266e-05 ,  8.4329434061583e-05 ,  8.41214660876611e-05 ,  8.13096858712433e-05 ,  8.45455525380393e-05 ,  8.24171965801847e-05 ,  8.32524653216603e-05 ,  7.35089833442168e-05 ,  9.4929948703978e-05 ,  7.61339751256264e-05 ,  8.92103422396434e-05 ,  6.96588631016521e-05 ,  8.51463730701273e-05 ,  9.26805074753872e-05 ,  0.000104939030722566 ,  8.00873055152132e-05 ,  8.45061695488167e-05 ,  8.42962960891549e-05 ,  5.46176632336652e-05 ,  7.34705577252939e-05 ,  7.48161779714619e-05 ,  8.44453907619565e-05 ,  7.98805790409065e-05 ,  8.55012090624576e-05 ,  7.66960317244663e-05 ,  9.72888791164028e-05 ,  7.93768454192467e-05 ,  9.19351756179612e-05 ,  7.56428783277542e-05 ,  8.34896510123919e-05 ,  7.38515691032997e-05 ,  8.9201060352664e-05 ,  8.00259956724171e-05 ,  8.52203065104761e-05 ,  7.3152670548734e-05 ,  8.53891607188066e-05 ,  6.83593784638123e-05 ,  8.82942822728837e-05 ,  8.00505925062488e-05 ,  7.83255614211715e-05 ,  8.56377699002543e-05 ,  6.62659830025762e-05 ,  6.70084218709181e-05 ,  7.81130079504255e-05 ,  9.3839040986604e-05 ,  7.46036927417697e-05 ,  8.43842805078824e-05 ,  8.89102273215777e-05 ,  7.05097591709063e-05 ,  9.37405117887987e-05 ,  9.06518598865084e-05 ,  7.49248035781908e-05 ,  9.41993064846115e-05 ,  7.35818587150744e-05 ,  8.44600353996145e-05 ,  7.29637812162819e-05 ,  0.00010070491332943 ,  7.78664551284871e-05 ,  9.41628045066642e-05 ,  7.71782442862443e-05 ,  9.06020842144899e-05 ,  8.4283060805051e-05 ,  8.34190774938153e-05 ,  7.24684523401352e-05 ,  8.95130897223415e-05 ,  6.74433633021476e-05 ,  8.74464124698647e-05 ,  8.17601182430774e-05 ,  8.82544272239838e-05 ,  9.52179677767985e-05 ,  7.95008933367293e-05 ,  8.87977007919396e-05 ,  7.26011471845029e-05 ,  8.19604156226147e-05 ,  8.59438605003949e-05 ,  9.21585005854874e-05 ,  8.58517787408752e-05)
  
 
B.esw <- c(
65.960334048781 ,  64.5121439706043 ,  62.5934036516555 ,  60.7873720171166 ,  57.0595765023233 ,  58.7932756576273 ,  62.9678703913832 ,  56.8876237004058 ,  61.653994751669 ,  64.9206177057004 ,  55.7559776276991 ,  60.3969792787867 ,  65.13695691289 ,  60.9872763407773 ,  60.9986334324486 ,  63.5207437383364 ,  63.0877739011678 ,  59.2841802388783 ,  58.816061045893 ,  59.0508659722084 ,  65.3514652091511 ,  62.0578691800862 ,  59.9499568434843 ,  59.8335898239129 ,  60.5867021770309 ,  60.7834805475399 ,  60.5423516981703 ,  70.3266394484822 ,  62.9661569005563 ,  62.2920629471591 ,  64.1092971163947 ,  61.9302168911723 ,  60.7388531740376 ,  64.0487979075723 ,  59.7763498414029 ,  63.8036933605511 ,  65.2679441925455 ,  62.4899449400625 ,  63.2823507187653 ,  63.3019535942845 ,  60.1690449588103 ,  60.4204076097981 ,  59.1330040885495 ,  65.6245512872223 ,  61.4711957096196 ,  66.2188583592455 ,  60.2822737750428 ,  59.8084794185536 ,  61.8222903323166 ,  64.705215503382 ,  64.2265903913138 ,  61.5423602597157 ,  64.9231787888112 ,  60.2202773944994 ,  64.378670488274 ,  59.6316254840006 ,  60.8367534790242 ,  57.864725520098 ,  61.7140321091783 ,  65.202612484014 ,  62.0250171924102 ,  66.4721924575218 ,  58.9423792303497 ,  59.4087279475158 ,  59.9663279900744 ,  60.8479392347003 ,  59.8874784874536 ,  66.3559526881532 ,  63.9595906854718 ,  63.5570647815568 ,  60.6917810107264 ,  65.402992058119 ,  67.634022260079 ,  60.9397019317095 ,  60.7382644291825 ,  66.7588199844643 ,  65.9087633470446 ,  64.5356139477007 ,  57.2759914710656 ,  61.2172824091199 ,  65.6241378768327 ,  64.0460255382488 ,  59.558933211284 ,  60.260189863877 ,  63.3612348381938 ,  64.9874008462376 ,  62.7095604310782 ,  65.622497825849 ,  61.1885551643773 ,  61.6035423662694 ,  58.7924494214213 ,  66.2807280860384 ,  65.7804218268036 ,  57.6228528544018 ,  56.7284117212236 ,  64.1843571577246 ,  61.5051378868397 ,  63.1333608730649 ,  64.5983236807875 ,  58.1088384853945 ,  63.3084579455283 ,  62.5293320365571 ,  67.0225815234186 ,  62.6171222189141 ,  57.8539048860362 ,  62.4767557980002 ,  62.5485061747616 ,  67.0146687277624 ,  61.6227973618611 ,  61.8551944641481 ,  64.5258524692322 ,  59.8957829571839 ,  61.6545492100902 ,  63.479080825731 ,  61.0027514267424 ,  64.0678751356436 ,  63.2041629293548 ,  65.0793984462705 ,  61.7180896800263 ,  61.3701984422319 ,  69.7911786453547 ,  59.2551949254251 ,  63.1960081034024 ,  60.9981390114962 ,  56.7475013983125 ,  63.1746029236868 ,  61.8167351065053 ,  63.5481904667847 ,  63.4506122928476 ,  59.4120533341852 ,  60.5407280215967 ,  62.6915845085916 ,  62.5571965064191 ,  65.3859079943389 ,  61.9630930939193 ,  64.2841027390763 ,  62.5622443577121 ,  65.0483627912759 ,  59.1903350011852 ,  61.3301577339852 ,  67.1195486665678 ,  61.8019750416081 ,  61.7165862435756 ,  61.4856154543192 ,  63.7484150405383 ,  62.8624146427422 ,  65.7133903644897 ,  57.7267581331392 ,  61.3008178978519 ,  63.3953206078416 ,  60.5187634534327 ,  63.0332407634147 ,  65.2037685685355 ,  59.8873988716417 ,  62.6692700641708 ,  61.4671597350144 ,  66.8419618712963 ,  60.1063736902548 ,  62.310605130961 ,  65.708307283837 ,  61.5788607459251 ,  62.8970245020666 ,  59.0891884647185 ,  61.294218590376 ,  59.7202124922351 ,  70.9349806854175 ,  58.4294380020159 ,  58.3764500102379 ,  66.8670629480787 ,  58.8361888720737 ,  65.5235686782682 ,  63.6538132438862 ,  59.5023014475995 ,  52.9295400199151 ,  62.9060204596284 ,  66.5699710891156 ,  68.2656795212284 ,  61.1829902527606 ,  65.548152852273 ,  63.20987269387 ,  63.6767830885228 ,  59.1329696058868 ,  65.8077582804707 ,  64.399634953522 ,  57.8724023994122 ,  63.1185347528292 ,  63.0713976663221 ,  59.4735187553718 ,  63.5670338928824 ,  60.0216985108676 ,  63.3406647304095 ,  64.2946656403599 ,  61.0237184003028 ,  64.9629406064262 ,  61.6072982366928 ,  69.101958056247 ,  61.2748800876482 ,  68.9425834333136 ,  69.619185574644 ,  64.5872900855152 ,  61.7509425851572 ,  63.7321702729023 ,  62.7593477795589 ,  61.0483652248539 ,  62.670666629985 ,  59.9486416552824 ,  65.3312594579374 ,  60.4181417956872 ,  62.7635773284271 ,  59.8170631726109 ,  60.7966621859234 ,  61.5450612876907 ,  62.7307300395353 ,  60.5945689972316 ,  60.6636563241552 ,  61.9813199487709 ,  58.8553606831187 ,  51.7015596883704 ,  65.2043930595748 ,  64.9611693299721 ,  66.676324082042 ,  60.1864892675127 ,  62.3890904232419 ,  63.3497378323488 ,  65.1414335254214 ,  61.1609682710213 ,  55.4154893014094 ,  58.4981931136237 ,  60.3543503339387 ,  65.4148896854435 ,  63.1198521567816 ,  64.6837231268006 ,  63.7719378188564 ,  60.396286704666 ,  58.7860902686847 ,  62.7365743282193 ,  61.2105793327369 ,  60.8340646167104 ,  59.5511819244833 ,  62.164380979911 ,  61.4085820962416 ,  62.2934425109737 ,  60.9166291908353 ,  61.3175834629083 ,  64.4449523525201 ,  65.2339771671577 ,  62.936021975873 ,  64.2490984702916 ,  66.8830333953258 ,  61.4836904183717 ,  58.5166034802066 ,  62.7445191246161 ,  58.2925562370993 ,  62.3935582137304 ,  62.7811298004899 ,  63.9325209655356 ,  62.8372633339707 ,  62.8685121009033 ,  58.495598875338 ,  64.6199572967161 ,  56.3777432997884 ,  56.2857698967995 ,  63.1891957434735 ,  63.131540351998 ,  65.9051668022725 ,  62.5904627860022 ,  61.4206129911267 ,  62.1980403761769 ,  56.7632676787637 ,  65.2517440405258 ,  59.5727739221283 ,  60.1206321331424 ,  61.8694168574026 ,  65.9429874276553 ,  64.0072220377144 ,  62.9405223637353 ,  60.1511724298193 ,  63.0261407534326 ,  62.7376881820967 ,  62.4112097833531 ,  60.1679572813358 ,  58.1821161071722 ,  67.2553903716938 ,  65.7251053944972 ,  60.2015875168419 ,  57.123385426584 ,  61.6823933121413 ,  62.3531830514084 ,  66.1763091081414 ,  63.9890048444408 ,  57.0432467155958 ,  63.7677103424424 ,  68.8082455996804 ,  66.3343111176228 ,  57.8399330636416 ,  57.9694750106913 ,  60.5716969183874 ,  58.25220875786 ,  58.3898068008068 ,  65.6828575261825 ,  64.8383960143578 ,  66.8036556698192 ,  62.6448390153428 ,  61.1229223966103 ,  63.6536745371051 ,  65.376605745975 ,  60.2292185870492 ,  60.7110841258399 ,  56.8354476248428 ,  62.5337791106458 ,  60.6377250250462 ,  65.3161407109455 ,  62.8753683559459 ,  65.6777644402574 ,  60.161901084189 ,  57.9282251757904 ,  57.6004612439955 ,  58.7775247946437 ,  59.1137594950985 ,  58.8345235513159 ,  61.5172370611822 ,  65.0026809334554 ,  65.8658851429266 ,  60.3838506224421 ,  60.7533627937006 ,  58.4820429631223 ,  61.473343049593 ,  65.1570221817977 ,  61.5874791345772 ,  60.0627735851102 ,  59.113646686092 ,  66.2821657861373 ,  59.9400720514753 ,  66.2189734524397 ,  62.7422781072545 ,  58.7474388027113 ,  65.0747459927851 ,  64.718091211045 ,  62.846599135038 ,  61.6343603531061 ,  62.8698191283254 ,  67.1227110055109 ,  63.3679325838164 ,  63.214032096108 ,  59.6885079839767 ,  70.7149004941069 ,  59.6311783151354 ,  69.7908191314444 ,  59.0742046051524 ,  62.6922320359513 ,  64.0931912513632 ,  60.8427201051669 ,  61.8237932538574 ,  59.1319184299574 ,  60.0216530939498 ,  63.6831688844088 ,  55.3531132660648 ,  58.6065725595575 ,  65.0864499925151 ,  61.1097108936014 ,  58.6786030364006 ,  63.9480371868771 ,  64.3177807636361 ,  60.1098788521317 ,  61.6424436922098 ,  59.2657964505061 ,  66.6491457563641 ,  63.1701035166211 ,  62.3472334064535 ,  60.0596570360885 ,  64.4556910219296 ,  63.9689553002799 ,  58.667256318236 ,  64.5027273122112 ,  58.4808919817987 ,  64.6550129133484 ,  62.7822178796132 ,  58.3230411691422 ,  56.9727346187562 ,  60.4966585511462 ,  64.1889095107121 ,  61.7244158721247 ,  68.0624546372509 ,  60.8624479754722 ,  59.1922191567021 ,  62.3451357476784 ,  59.6321134172036 ,  62.2191867790437 ,  55.6985749729104 ,  61.7741664966769 ,  56.5874448640879 ,  60.805173612282 ,  62.2137408689423 ,  62.9270902010622 ,  62.4953287370116 ,  68.1599142193147 ,  69.3600990763122 ,  70.7933861169264 ,  60.5991265142728 ,  61.7120836925566 ,  60.1474170253452 ,  58.644978105157 ,  58.7687487311026 ,  65.091666542279 ,  61.2190424228288 ,  61.8819701013137 ,  62.6645013033349 ,  64.3864534539696 ,  66.9129339063831 ,  58.7646161024214 ,  68.583116378084 ,  65.9569249164249 ,  67.5870697364398 ,  61.0973877910174 ,  56.8603303818143 ,  65.4384668155007 ,  62.1513945187524 ,  62.1226754377826 ,  63.057650477027 ,  61.0978310182447 ,  57.4027968134507 ,  56.625698640353 ,  61.9386803231913 ,  62.5524587880275 ,  67.526178822521 ,  59.7680712380648 ,  63.8846945328263 ,  63.0823641602404 ,  64.2058561348703 ,  60.7255957649127 ,  63.862187053547 ,  56.1817783128102 ,  67.1331176741713 ,  61.6520446163685 ,  67.5913031549558 ,  63.9421768436379 ,  57.8450772136184 ,  57.8378168985629 ,  62.7787106262767 ,  64.590944808831 ,  64.7517576283588 ,  71.2020371917314 ,  65.029828623894 ,  63.1176618514712 ,  63.6506024958971 ,  64.6798850071034 ,  61.4026404721942 ,  62.838250377262 ,  62.2430396007574 ,  60.5410246549095 ,  61.9397788296815 ,  64.2639626965056 ,  62.7156904792182 ,  60.5569018576533 ,  53.2504880684206 ,  65.0830182064016 ,  61.6050354073136 ,  61.325322472301 ,  61.9711755229982 ,  64.2031713499597 ,  64.6512230054837 ,  63.154505135748 ,  63.4814756767033 ,  62.1156348494387 ,  65.6026218165845 ,  67.9848208383591 ,  67.5659268060363 ,  61.1271284403894 ,  61.6219124452039 ,  61.8861971778548 ,  63.7347000156799 ,  67.3665611094576 ,  61.9321939337787 ,  60.3652504104513 ,  62.6554120965478 ,  61.7779966925246 ,  67.0077603590823 ,  67.5862057874461 ,  65.1007927607243 ,  56.6837619397975 ,  63.1422948245646 ,  56.3444938089122 ,  57.5867524008523 ,  60.7050052731612 ,  56.5224950706037 ,  58.2733741148302 ,  62.2876402507214 ,  61.4435276121104 ,  66.104848795276 ,  62.2603522622575 ,  63.3628677022393 ,  60.7460869640565 ,  59.2208489694621 ,  66.2116948371067 ,  62.4077720171072 ,  64.2781395011178 ,  64.224770566396 ,  59.9551584927244 ,  60.2826165818762 ,  61.1519071240909 )

fit$B <- data.frame(
  density = B.d
  , effDistance = B.esw
  )
units(fit$B[,1]) <- "1/m^2"
units(fit$B[,2]) <- "m"

d <- fit$density
d.ci <- fit$density.ci
units(d) <- "1/ha"
units(d.ci) <- "1/ha"

## -----------------------------------------------------------------------------
fit

## ----bootstrapPrintout--------------------------------------------------------
# Convert to hectares for readability
units(fit$B$density) <- "1/ha"

head(fit$B)

## ----bootPlot, fig., fig.width=6, fig.height=4, fig.cap="Bootstrap distribution of sparrow density computed in the `abundEstim` routine."----
# Figure 3
hist(fit$B$density
     , n = 30
     , xlab = "Density"
     , main = NULL)
# Show final density estimates, after converting to 1/ha
d <- fit$density
d.ci <- fit$density.ci
units(d) <- "1/ha"
units(d.ci) <- "1/ha"
abline(v = c(d, d.ci), col="blue")

## ----changeUnits--------------------------------------------------------------
dfuncSparrow <- dfuncEstim(formula = dist~1
                          , detectionData = sparrowDetectionData
                          , likelihood = "halfnorm"
                          , w.hi = rightTruncDistance
                          , outputUnits = "km")

## ----include=FALSE------------------------------------------------------------
fit <- abundEstim(dfuncSparrow
                , detectionData = sparrowDetectionData
                , siteData = sparrowSiteData
                , area = saSize
                , ci = NULL
                )
fit$B <- matrix(1, nrow = 100, ncol = 2)
fit$n.hat.ci <-  c( `2.5%`= 250119.1
                   , `97.5%` = 390627.4 )
fit$density.ci <- units::set_units(c(6.093035e-05, 9.515893e-05), "1/m^2")
fit$effDistance.ci <- units::set_units(c(56.74188, 68.64928 ), "m")
fit$nItersConverged <- 500
fit$alpha <- 0.95
units(fit$density.ci) <- "1/km^2"
units(fit$effDistance.ci) <- "km"

## -----------------------------------------------------------------------------
fit

