#' Estimate discrete choice model with random parameters
#' 
#' Estimation of discrete choice models such as Binary (logit and probit), 
#' Poisson and Ordered (logit and probit) model with random coefficients for cross-section data by simulated maximum likelihood
#' 
#' @param x,object,obj and object of class \code{Rchoice},
#' @param formula a symbolic description of the model to be estimated,
#' @param new an updated formula for the update method,
#' @param data the data,
#' @param subset an optional vector specifying a subset of observations,
#' @param weights an optional vector of weigths,
#' @param na.action a function wich indicated what should happen when the data
#' contains \code{NA}'s
#' @param start a vector of starting values,
#' @param family the distribution to be used,
#' @param ranp a named vector whose names are the random parameters and values the distribution:
#' "\code{n}" for normal, "\code{ln}" for log-normal, "\code{cn}" for truncated normal, "\code{u}" for uniform, "\code{t}" for triangular,
#' @param R the number of draws of pseudo-random numbers if \code{ranp} is not \code{NULL}.
#' @param haltons only relevant if \code{ranp} is not \code{NULL}. If not \code{NULL}, halton sequence is used
#' instead of pseudo-random numbers. If \code{haltons=NA}, some default values are used for
#' the prime of the sequence and for the number of element droped. Otherwise, \code{haltons} should
#' be a list with elements \code{prime} and \code{drop}.
#' @param seed ,
#' @param correlation only relevant if \code{ranp} is not \code{NULL}. If true, the correlation between random
#' parameters is taken into account,
#' @param alpha significance value for \code{getSummary},
#' @param digits number of digits,
#' @param width width,
#' @param ... further arguments passed to \code{maxLik}.
#' @export
#' @aliases ordinal
#' @details
#' The models are estimated using the \code{maxLik} function of \code{\link[maxLik]{maxLik}} package.
#' 
#' 
#'  If \code{ranp} is not NULL, the random parameter (random coefficient) model is estimated.   
#'  A random parameter model or random coefficient models permits regression parameter to 
#'  vary across individuals according to some distribution. A fully parametric 
#'  random parameter model specifies the latent variable  \eqn{y^{*}} conditional on regressors
#'  \eqn{x} and given parameters \eqn{\beta_i} to have conditional density \eqn{f(y|x, \beta_i)} where
#'  \eqn{\beta_i} are iid with density \eqn{g(\beta_i|\theta_i)}. The density is assumed a priori by the user by the argument
#'  \code{ranp}. If the parameters are assumed to be normally distributed \eqn{\beta_i ~ N(\beta, \Sigma)}, then the random parameter 
#'  are constructed as: \deqn{\beta_{ir}=\beta+L\omega_{ir}}
#'  where \eqn{LL'=\Sigma} and \eqn{\omega_{ir}} is the {r}-th draw from standard normal distribution for individual \eqn{i}. 
#'  
#'  
#'  Once the model is specified by the argument \code{family}, the model is estimated using 
#'  Simulated Maximum Likelihood (SMLE). The probabilities, given by \eqn{f(y|x, \beta_i)}, are simulated using \code{R} pseudo-draws if \code{halton=NULL} or \code{R} halton
#'  draws if \code{halton = NA}. The user can also specified the primes and the number of dropped elements for the halton draws.
#'  For example, if the model consists of two random parameters, the user can specify \code{haltons = list("prime" = c(2, 3), "drop" = c(11, 11))}. 
#'  
#'  
#'  A random parameter hierarchical model can be estimated by including heterogeneity in the mean of the 
#'  random parameters: \deqn{\beta_{ir}=\beta+\pi's_i+L\omega_{ir}} \pkg{Rchoice} manages the variables in the hierarchical model by 
#'  the \code{formula} object: all the hierarchical variables (\eqn{s_i}) are included after the \code{|} symbol. See examples below
#' 
#' @return An object of class ``\code{Rchoice}'', a list elements:
#' \item{coefficients}{the named vector of coefficients,}
#' \item{family}{type of model,}
#' \item{link}{distribution of the errors,}
#' \item{logLik}{a set of values of the maximum likelihood procedure,}   
#' \item{mf}{the model framed used,} 
#' \item{formula}{the formula (a Formula object),}
#' \item{time}{\code{proc.time()} minus the start time,}
#' \item{freq}{frequency of dependent variable,}
#' \item{draws}{type of draws used,}
#' \item{R.model}{\code{TRUE} if a random parameter model is fitted,}
#' \item{R}{number of draws used,}
#' \item{b.random}{matrix of conditional expectation of random parameters,}
#' \item{sd.random}{matrix of standard deviation of conditional expectation of random parameters,}
#' \item{ranp}{vector indicating the variables with random parameters and their distribution,}
#' \item{probabilities}{the fitted probabilities for each individuals,}
#' \item{residuals}{the residuals,}
#' \item{call}{the matched call.}   
#' @author Mauricio Sarrias \email{msarrias86@@gmail.com}
#' @keywords package
#' @seealso \code{\link[mlogit]{mlogit}}, \code{\link[maxLik]{maxLik}}
#' @examples
#' ## Probit model
#' data("Workmroz")
#' probit <- Rchoice(lfp ~ k5 + k618 + age + wc + hc + lwg + inc,  
#'                  data = Workmroz, family = binomial('probit'))
#' summary(probit)
#' 
#' ## Poisson model
#' data("Articles")
#' poisson <- Rchoice(art ~ fem + mar + kid5 + phd + ment, data = Articles, family = poisson)
#' summary(poisson)
#' 
#' ## Ordered probit model
#' data("Health")
#' oprobit <- Rchoice(newhsat ~ age + educ + hhinc + married + hhkids, 
#' data = Health, family = ordinal('probit'), subset = year == 1988)
#' summary(oprobit)
#' 
#' \dontrun{
#' ## Hierarchical Logit Random Parameter Model 
#' Hran.logit<-Rchoice(lfp ~ k618 + lwg + wc + inc + k5 | age + wc + hc, 
#' ranp = c(inc = "t", k5 = "n"), 
#' family = binomial('logit'), data = Workmroz)
#' summary(Hran.logit)
#' }
#' 
#' \dontrun{
#' ## Hierarchical Poisson model with correlated random parameters
#' poissonH.ran <- Rchoice(art ~ fem + mar + kid5 + phd + ment | fem, data = Articles,
#' ranp = c(kid5 = "n", phd = "n", ment = "n"), family = poisson, correlation =  TRUE)
#' summary(poissonH.ran)
#' }
#' 
#' \dontrun{
#' ## Ordered Probit model with random parameters
#' oprobit.ran <- Rchoice(newhsat ~ age + educ + hhinc + married + hhkids, 
#'                       data = Health, family = ordinal('probit'), 
#'                       subset = year == 1988, 
#'                       ranp = c(age = "n", hhinc = "n"), 
#'                       start = rep(0, 11))
#' summary(oprobit.ran)
#' }
#' @references
#' Greene, W. H. (2012). Econometric analysis. 7 edition. Prentice Hall.
#' 
#' Train, K. (2009). Discretechoice methods with simulation. Cambridge university press.
#' @import maxLik Formula sandwich car lmtest
Rchoice <- function(formula, data, subset, weights, na.action, family,
                  start = NULL, ranp = NULL, R = 40, haltons = NA, 
                  seed = 10, correlation = FALSE,
                  ...)
{
  ####################
  # 1) Check arguments
  ####################
  start.time <- proc.time()
  callT <- match.call(expand.dots = TRUE)
  callF <- match.call(expand.dots = FALSE)
  
  ## I use Formula package
  formula <- callF$formula <- Formula::as.Formula(formula)
  nframe  <- length(sys.calls())
  
  ## family
  if (is.character(family)){
    if (family %in% c("ordprobit", "ordlogit")){
      if (family == "ordprobit") family <- list(family = "ordinal", link = "probit")
      if (family == "ordlogit")  family <- list(family = "ordinal", link = "logit")
    }
    else  family <- get(family, mode = "function")
  }
  if (is.function(family)) family <- family()
  link   <- family$link
  family <- family$family
  
  ## check if random and hierarchical model is estimated
  R.model <- !is.null(ranp)
  Hier <- is.hierarchical(formula)
  if (Hier) if (!R.model) stop('Hierarchical model needs ranp to be specified')
  
  if (R.model){
    if (is.null(callT$method))      callT$method      <- 'bfgs'
    if (is.null(callT$iterlim))     callT$iterlim     <- 2000
  }else{
    if (is.null(callT$method) && (family == "ordinal")) callT$method <- 'bfgs'
    if (is.null(callT$method)) callT$method <- 'nr'
  }
  
  ####################
  # 2) Model Frame
  ####################
  mf <- callT
  m  <- match(c("formula", "data", "subset", "na.action", "weights"), names(mf), 0L)
  mf <- mf[c(1L, m)]
  mf$formula <- formula
  mf[[1L]] <- as.name("model.frame")
  mf <- eval(mf, parent.frame())

  ##########################################
  # 3) Extract the elements of the model
  ##########################################
  
  if (Hier){
    cons <- has.intercept.Formula(formula)[2]
    if (cons == TRUE){
      warning("Model assumes no constant in S variables...updating formula", call. = FALSE, immediate. = TRUE)
      formula <- update(formula, ~. | ~. -1)
    }
    S <- model.matrix(formula, mf , rhs = 2) 
  }
  y <- model.response(mf)
  X <- model.matrix(formula, mf, rhs = 1)
  weights <- model.weights(mf)
  freq <- table(y)
  
  # Other warnings
  if (family == "ordinal") {
    y <- as.factor(y)
    J <- length(levels(y))
    if (J < 3) stop("The alternatives must be >=3 in y")
  }
  
  if (family == "binomial"){
    if (!all(y %in% c( 0, 1, TRUE, FALSE))){
      stop( "all dependent variables must be either 0, 1, TRUE, or FALSE")
    }
    if (!is.numeric(y)) y <- as.numeric(y)  
  }
  
  ########################################################
  #   4) Initial Values
  ########################################################
  
  ## Names of thresholds if ordered model
  names.kappa <- c()
  if (family == "ordinal")  names.kappa <- paste('kappa', 1:(J - 2) , sep = '.')
  
  ## Names for models with random parameters
  names.random <- c()
  if (R.model){
    ndist <- ranp[! (ranp %in% c("cn", "ln", "n", "u", "t"))]
    if (length(ndist) > 0){
      udstr <- paste("unknown distribution", paste(unique(ndist), collapse = ", "))
      stop(udstr)
    }
    Vara <- sort(match(names(ranp), colnames(X))) 
    Varc <- (1:ncol(X))[- Vara]
    Xa   <- X[ , Vara, drop = F]                        
    Xc   <- X[ , Varc, drop = F]  
    colnamesX <- colnames(X)
    names.f <- colnamesX[Varc] 
    names.b <- paste('mean', colnamesX[Vara], sep = '.')
    if (!correlation) {
      names.sd <- paste('sd', colnamesX[Vara], sep = '.')
    } else {
      names.sd <- c()
      Ka <- length(ranp)
      for (i in 1:Ka){
        names.sd <- c(names.sd,
                      paste('sd', names(ranp)[i], names(ranp)[i:Ka], sep = '.')
        )
      }
    }
    names.phi <- c()
    if (Hier) names.phi <- c(outer(names(ranp), colnames(S) , FUN = paste, sep = "."))
    names.random <- c(names.b, names.phi, names.sd)
  }  else {
    names.f <- colnames(X) 
  }
  
  all.names   <- c(names.kappa, names.f, names.random)
  if (is.null(start)){
    if (family == "ordinal"){
      if (R.model)  theta <- coef(lm(unclass(y)~Xc-1+Xa)) else theta <- coef(lm(unclass(y)~X-1))
      z <- as.integer(table(unclass(y)))
      z <- (cumsum(z)/sum(z))[1:(J-1)]
      start.kappa <- switch(link,
                            "probit" = qnorm(z),
                            "logit"  = qlogis(z))
      theta       <- c(log(diff(start.kappa)), theta)
    } else {
      if (R.model)  theta <- coef(lm(y~Xc-1+Xa)) else theta <- coef(lm(y~X-1))
    }
    names(theta) <- c(names.kappa, names.f)
  }
  
  
  ## Initial value if random parameters and start is null
  if (R.model & is.null(start)) {
    callst        <- callT
    callst$start  <- theta
    callst$method <- 'nr'
    callst$iterlim <-  callst$tol <- callst$steptol <- callst$ftol <- NULL
    callst$haltons <- callst$ranp <- callst$R <- callst$data <- callst$formula <- NULL 
    callst$print.level <- 0
    callst[[1]]   <- as.name('maxLik')
    Xst <- cbind(Xc, Xa) 
    callst$X <- as.name('Xst') ; callst$y <- as.name('y')
    callst$link  <- link
    if (family == "poisson")  callst$logLik <- as.name('lnpoisson')
    if (family == "binomial") callst$logLik <- as.name('lnbinary')
    if (family == "ordinal")  callst$logLik <- as.name('lnordered')
    start.fixed  <- coef(eval(callst, sys.frame(which = nframe)))
    if (is.null(start.fixed)) stop("attempt to find suitable starting values failed")
    start.random <- rep(0, length(c(names.phi, names.sd)))
    theta        <- c(start.fixed, start.random)
    names(theta) <- all.names
  }
  
  ## If initial value start is not null
  if (!is.null(start)){
    theta <- start
    if (length(start) != length(all.names)) stop('Incorrect Number of Initial Parameters')
    names(theta) <- all.names
  }
  
  cat("\nStarting values Parameters:\n")
  print(theta)
   
 #######################################################################
 # 5) Estimate the model using maxLik and passing the correct arguments
 #######################################################################
 opt <- callT
 opt$start <- theta
 
 ## Maximization control arguments
 m <- match(c('method', 'print.level', 'iterlim',
              'start','tol', 'ftol', 'steptol'),
               names(opt), 0L)
 opt <- opt[c(1L, m)]
 
 ## Optimization code name
 opt[[1]] <- as.name('maxLik')
 
 #Variables
 opt[c('X', 'y')] <- list(as.name('X'), as.name('y'))
 
 ## Weights
 if (is.null(weights)) weights <- 1
 opt$weights <- as.name('weights')
 
 ## Link
 opt$link  <- link
 
 ## loglik for standard Models
 if (family == "poisson")  opt$logLik <- as.name('lnpoisson')
 if (family == "binomial") opt$logLik <- as.name('lnbinary')
 if (family == "ordinal")  opt$logLik <- as.name('lnordered')
 
 ## Arguments for random parameters
 if (R.model) {
   opt[c('R', 'seed', 'ranp', 'correlation','haltons')] <-
     list(as.name('R'), as.name('seed'), as.name('ranp'), as.name('correlation'), as.name('haltons'))
   if (Hier) {
     if (family == "binomial") opt$logLik <- as.name('lnlbinaryH.ran')
     if (family == "poisson")  opt$logLik <- as.name('lnlpoissonH.ran')
     if (family == "ordinal")  opt$logLik <- as.name('lnorderedH.ran')
     opt$S <- as.name('S')
   } else {
     if (family == "binomial") opt$logLik <- as.name('lnlbinary.ran')
     if (family == "poisson")  opt$logLik <- as.name('lnlpoisson.ran')
     if (family == "ordinal")  opt$logLik <- as.name('lnordered.ran')
   }
 }
  
  ## Optimizing the ML
  x <- eval(opt, sys.frame(which = nframe))
  
 ###################################
 # 6) Extract predicted probabilities, 
 # conditional means of ranp, etc.
 ##################################
 
 ## Get probability, and conditional beta
 if (!is.null(ranp)){
   opt$steptol <- opt$logLik <- opt$iterlim <- opt$method <- opt$print.level <- opt$tol<-opt$ftol <- NULL
   names(opt)[[2]] <- 'theta'
   betahat <- coef(x)
   if (Hier) {
     if (family == "binomial") opt[[1]] <- as.name('lnlbinaryH.ran')
     if (family == "poisson")  opt[[1]] <- as.name('lnlpoissonH.ran')
     if (family == "ordinal")  opt[[1]] <- as.name('lnorderedH.ran')
   }else{
     if (family == "binomial") opt[[1]]  <- as.name('lnlbinary.ran')
     if (family == "poisson")  opt[[1]]  <- as.name('lnlpoisson.ran')
     if (family == "ordinal")  opt[[1]]  <- as.name('lnordered.ran')
   }
   if (correlation)  diag.sd <- paste('sd',  names(ranp),  names(ranp),  sep = '.')
   else              diag.sd <- paste('sd',  names(ranp),  sep = '.')
   betahat <- ifelse(names(betahat) %in% diag.sd, abs(betahat), betahat)
   names(betahat) <- names(coef(x))
   opt[[2]] <- betahat
   opt$make.estb <- TRUE
   again <- eval(opt, sys.frame(which = nframe))
   x$probabilities <- attr(again, 'probabilities')
   x$b.random      <- attr(again, 'b.random')
   x$sd.random     <- attr(again, 'sd.random')
   x$estimate      <- betahat
 } else {
   opt$steptol <- opt$logLik <- opt$iterlim <- opt$method <- opt$print.level <- opt$tol<-opt$ftol <- NULL
   names(opt)[[2]] <- 'theta'
   betahat  <- coef(x)
   opt[[2]] <- betahat
   if (family == "poisson")  opt[[1]] <- as.name('lnpoisson')
   if (family == "binomial") opt[[1]] <- as.name('lnbinary')
   if (family == "ordinal")  opt[[1]] <- as.name('lnordered')
   again <- eval(opt, sys.frame(which = nframe))
   x$probabilities <- drop(attr(again, 'probabilities'))
 }
 
 ## Ordered Model
 if (family == "ordinal"){
   J <- length(levels(y))
   attr(x$estimate, "alphas") <- x$estimate[1:(J - 2)]
   kappas <- cumsum(c(exp(x$estimate[1:(J - 2)])))
   names(kappas) <- names.kappa
   x$estimate[names.kappa]  <- kappas
   attr(x$estimate, "fixed" ) <- x$estimate[-c(1:(J - 2))]
 }
 
 resid <- drop(unclass(y) - x$probabilities)
 
 
###########################
# 7) Put results in form
###########################

logLik<-structure(list(
                    maximum     = logLik(x),
                    gradient    = x$gradient,
                    nobs        = nObs(x),
                    gradientObs = x$gradientObs,
                    hessian     = hessian(x),
                    iterations  = nIter(x),
                    type        = maximType(x),
                    code        = returnCode(x),
                    nparam      = nParam(x),
                    message     = returnMessage(x)),
                    class = "logLik"
                   )
 
 
  result <- structure(
                      list(
                        coefficients  = x$estimate,
                        family        = family,
                        link          = link,
                        logLik        = logLik,
                        mf            = mf,
                        formula       = formula,
                        time          = proc.time()-start.time,
                        freq          = freq,
                        draws         = haltons,
                        R.model       = R.model,
                        R             = R,
                        b.random      = x$b.random,
                        sd.random     = x$sd.random,
                        ranp          = ranp,
                        probabilities = x$probabilities,
                        residuals     = resid,
                        correlation   = correlation,
                        call          = callT),
                    class = 'Rchoice'
                 )
 result
}