% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RaScreen.R
\name{RaScreen}
\alias{RaScreen}
\title{Variable screening via RaSE.}
\usage{
RaScreen(
  xtrain,
  ytrain,
  xval = NULL,
  yval = NULL,
  B1 = 200,
  B2 = NULL,
  D = NULL,
  dist = NULL,
  model = NULL,
  criterion = NULL,
  k = 5,
  cores = 1,
  seed = NULL,
  iteration = 0,
  cv = 5,
  scale = FALSE,
  C0 = 0.1,
  kl.k = NULL,
  classification = NULL,
  ...
)
}
\arguments{
\item{xtrain}{n * p observation matrix. n observations, p features.}

\item{ytrain}{n 0/1 observatons.}

\item{xval}{observation matrix for validation. Default = \code{NULL}. Useful only when \code{criterion} = 'validation'.}

\item{yval}{0/1 observation for validation. Default = \code{NULL}. Useful only when \code{criterion} = 'validation'.}

\item{B1}{the number of weak learners. Default = 200.}

\item{B2}{the number of subspace candidates generated for each weak learner. Default = \code{NULL}, which will set B2 = \eqn{20*floor(p/D)}.}

\item{D}{the maximal subspace size when generating random subspaces. Default = \code{NULL}. It means that \code{D} = \eqn{min(\sqrt n0, \sqrt n1, p)} when \code{model} = 'qda', and \code{D} = \eqn{min(\sqrt n, p)} otherwise.}

\item{dist}{the distribution for features when generating random subspaces. Default = \code{NULL}, which represents the hierarchical uniform distribution. First generate an integer \eqn{d} from \eqn{1,...,D} uniformly, then uniformly generate a subset with cardinality \eqn{d}.}

\item{model}{the model to use. Default = 'lda' when \code{classification} = TRUE and 'lm' when \code{classification} = FALSE.
\itemize{
\item lm: linear regression. Only available for regression.
\item lda: linear discriminant analysis. \code{\link[MASS]{lda}} in \code{MASS} package. Only available for classification.
\item qda: quadratic discriminant analysis. \code{\link[MASS]{qda}} in \code{MASS} package. Only available for classification.
\item knn: k-nearest neighbor. \code{\link[class]{knn}}, \code{\link[class]{knn.cv}} in \code{class} package, \code{\link[caret]{knn3}} in \code{caret} package and \code{\link[caret]{knnreg}} in \code{caret} package.
\item logistic: logistic regression. \code{\link[glmnet]{glmnet}} in \code{glmnet} package. Only available for classification.
\item tree: decision tree. \code{\link[rpart]{rpart}} in \code{rpart} package. Only available for classification.
\item svm: support vector machine. If kernel is not identified by user, it will use RBF kernel. \code{\link[e1071]{svm}} in \code{e1071} package.
\item randomforest: random forest. \code{\link[randomForest]{randomForest}} in \code{randomForest} package and \code{\link[ranger]{ranger}} in \code{ranger} package.
\item kernelknn: k-nearest neighbor with different kernels. It relies on function \code{\link[KernelKnn]{KernelKnn}} in \code{KernelKnn} package. Arguments \code{method} and \code{weights_function} are required. Different choices of multiple arguments are available. See documentation of function \code{\link[KernelKnn]{KernelKnn}} for details.
}}

\item{criterion}{the criterion to choose the best subspace. Default = 'ric' when \code{model} = 'lda', 'qda'; default = 'bic' when \code{model} = 'lm' or 'logistic'; default = 'loo' when \code{model} = 'knn'; default = 'cv' and set \code{cv} = 5 when \code{model} = 'tree', 'svm', 'randomforest'.
\itemize{
\item ric: minimizing ratio information criterion (RIC) with parametric estimation (Tian, Y. and Feng, Y., 2020). Available for binary classification and \code{model} = 'lda', 'qda', or 'logistic'.
\item nric: minimizing ratio information criterion (RIC) with non-parametric estimation (Tian, Y. and Feng, Y., 2020; ). Available for binary classification and \code{model} = 'lda', 'qda', or 'logistic'.
\item training: minimizing training error/MSE. Not available when \code{model} = 'knn'.
\item loo: minimizing leave-one-out error/MSE. Only available when  \code{model} = 'knn'.
\item validation: minimizing validation error/MSE based on the validation data.
\item cv: minimizing k-fold cross-validation error/MSE. k equals to the value of \code{cv}. Default = 5.
\item aic: minimizing Akaike information criterion (Akaike, H., 1973). Available when \code{base} = 'lm' or 'logistic'.

AIC = -2 * log-likelihood + |S| * 2.

\item bic: minimizing Bayesian information criterion (Schwarz, G., 1978). Available when \code{model} = 'lm' or 'logistic'.

BIC = -2 * log-likelihood + |S| * log(n).

\item ebic: minimizing extended Bayesian information criterion (Chen, J. and Chen, Z., 2008; 2012). \code{gam} value is needed. When \code{gam} = 0, it represents BIC. Available when \code{model} = 'lm' or 'logistic'.

eBIC = -2 * log-likelihood + |S| * log(n) + 2 * |S| * gam * log(p).
}}

\item{k}{the number of nearest neightbors considered when \code{model} = 'knn' or 'kernel'. Only useful when \code{model} = 'knn' or 'kernel'. \code{k} is required to be a positive integer. Default = 5.}

\item{cores}{the number of cores used for parallel computing. Default = 1.}

\item{seed}{the random seed assigned at the start of the algorithm, which can be a real number or \code{NULL}. Default = \code{NULL}, in which case no random seed will be set.}

\item{iteration}{the number of iterations. Default = 0.}

\item{cv}{the number of cross-validations used. Default = 5. Only useful when \code{criterion} = 'cv'.}

\item{scale}{whether to normalize the data. Logistic, default = FALSE.}

\item{C0}{a positive constant used when \code{iteration} > 1. See Tian, Y. and Feng, Y., 2021 for details. Default = 0.1.}

\item{kl.k}{the number of nearest neighbors used to estimate RIC in a non-parametric way. Default = \code{NULL}, which means that \eqn{k0 = floor(\sqrt n0)} and \eqn{k1 = floor(\sqrt n1)}. See Tian, Y. and Feng, Y., 2020 for details. Only available when \code{criterion} = 'nric'.}

\item{classification}{the indicator of the problem type, which can be TRUE, FALSE or \code{NULL}. Default = \code{NULL}, which will automatically set \code{classification} = TRUE if the number of unique response value \eqn{\le} 10. Otherwise, it will be set as FALSE.}

\item{...}{additional arguments.}
}
\value{
A list including the following items.
\item{model}{the model used in RaSE screening.}
\item{criterion}{the criterion to choose the best subspace for each weak learner.}
\item{B1}{the number of selected subspaces.}
\item{B2}{the number of subspace candidates generated for each of B1 subspaces.}
\item{n}{the sample size.}
\item{p}{the dimension of data.}
\item{D}{the maximal subspace size when generating random subspaces.}
\item{iteration}{the number of iterations.}
\item{selected.perc}{A list of length (\code{iteration}+1) recording the selected percentages of each feature in B1 subspaces. When it is of length 1, the result will be automatically transformed to a vector.}
\item{scale}{a list of scaling parameters, including the scaling center and the scale parameter for each feature. Equals to \code{NULL} when the data is not scaled by \code{RaScreen}.}
}
\description{
\code{RaSE} is a general framework for variable screening. In RaSE screening, to select each of the B1 subspaces, B2 random subspaces are generated and the optimal one is chosen according to some criterion. Then the selected proportions (equivalently, percentages) of variables in the B1 subspaces are used as importance measure to rank these variables.
}
\examples{
set.seed(0, kind = "L'Ecuyer-CMRG")
train.data <- RaModel("screening", 1, n = 100, p = 100)
xtrain <- train.data$x
ytrain <- train.data$y

# test RaSE screening with linear regression model and BIC
fit <- RaScreen(xtrain, ytrain, B1 = 100, B2 = 50, iteration = 0, model = 'lm',
cores = 2, criterion = 'bic')

# Select D variables
RaRank(fit, selected.num = "D")


\dontrun{
# test RaSE screening with knn model and 5-fold cross-validation MSE
fit <- RaScreen(xtrain, ytrain, B1 = 100, B2 = 50, iteration = 0, model = 'knn',
cores = 2, criterion = 'cv', cv = 5)

# Select n/logn variables
RaRank(fit, selected.num = "n/logn")


# test RaSE screening with SVM and 5-fold cross-validation MSE
fit <- RaScreen(xtrain, ytrain, B1 = 100, B2 = 50, iteration = 0, model = 'svm',
cores = 2, criterion = 'cv', cv = 5)

# Select n/logn variables
RaRank(fit, selected.num = "n/logn")


# test RaSE screening with logistic regression model and eBIC (gam = 0.5). Set iteration number = 1
train.data <- RaModel("screening", 6, n = 100, p = 100)
xtrain <- train.data$x
ytrain <- train.data$y

fit <- RaScreen(xtrain, ytrain, B1 = 100, B2 = 100, iteration = 1, model = 'logistic',
cores = 2, criterion = 'ebic', gam = 0.5)

# Select n/logn variables from the selected percentage after one iteration round
RaRank(fit, selected.num = "n/logn", iteration = 1)
}
}
\references{
Tian, Y. and Feng, Y., 2021(a). RaSE: A variable screening framework via random subspace ensembles. Journal of the American Statistical Association, (just-accepted), pp.1-30.

Tian, Y. and Feng, Y., 2021(b). RaSE: Random subspace ensemble classification. Journal of Machine Learning Research, 22(45), pp.1-93.

Chen, J. and Chen, Z., 2008. Extended Bayesian information criteria for model selection with large model spaces. Biometrika, 95(3), pp.759-771.

Chen, J. and Chen, Z., 2012. Extended BIC for small-n-large-P sparse GLM. Statistica Sinica, pp.555-574.

Schwarz, G., 1978. Estimating the dimension of a model. The annals of statistics, 6(2), pp.461-464.
}
\seealso{
\code{\link{Rase}}, \code{\link{RaRank}}.
}
