% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/query.R
\docType{methods}
\name{sqlite-query}
\alias{sqlite-query}
\alias{dbSendQuery,SQLiteConnection,character-method}
\alias{dbBind,SQLiteResult-method}
\alias{dbFetch,SQLiteResult-method}
\alias{dbClearResult,SQLiteResult-method}
\title{Execute a SQL statement on a database connection}
\usage{
\S4method{dbSendQuery}{SQLiteConnection,character}(conn, statement,
  params = NULL, ...)

\S4method{dbBind}{SQLiteResult}(res, params, ...)

\S4method{dbFetch}{SQLiteResult}(res, n = -1, ..., row.names = NA)

\S4method{dbClearResult}{SQLiteResult}(res, ...)
}
\arguments{
\item{conn}{an \code{\linkS4class{SQLiteConnection}} object.}

\item{statement}{a character vector of length one specifying the SQL
statement that should be executed.  Only a single SQL statment should be
provided.}

\item{params}{A named list of query parameters to be substituted into
a parameterised query. The elements of the list can be vectors
which all must be of the same length.}

\item{...}{Unused. Needed for compatibility with generic.}

\item{res}{an \code{\linkS4class{SQLiteResult}} object.}

\item{n}{maximum number of records to retrieve per fetch. Use \code{-1} to
retrieve all pending records; \code{0} retrieves only the table definition.}

\item{row.names}{Either \code{TRUE}, \code{FALSE}, \code{NA} or a string.

  If \code{TRUE}, always translate row names to a column called "row_names".
  If \code{FALSE}, never translate row names. If \code{NA}, translate
  rownames only if they're a character vector.

  A string is equivalent to \code{TRUE}, but allows you to override the
  default name.

  For backward compatibility, \code{NULL} is equivalent to \code{FALSE}.}
}
\description{
To retrieve results a chunk at a time, use \code{\link[=dbSendQuery]{dbSendQuery()}},
\code{\link[=dbFetch]{dbFetch()}}, then \code{\link[=dbClearResult]{dbClearResult()}}. Alternatively, if you want all the
results (and they'll fit in memory) use \code{\link[=dbGetQuery]{dbGetQuery()}} which sends,
fetches and clears for you. To run the same prepared query with multiple
inputs, use \code{\link[=dbBind]{dbBind()}}.
For statements that do not return a table,
use \code{\link[=dbSendStatement]{dbSendStatement()}} and \code{\link[=dbExecute]{dbExecute()}} instead of \code{\link[=dbSendQuery]{dbSendQuery()}}
and \code{\link[=dbGetQuery]{dbGetQuery()}}.
See \link{sqlite-meta} for how to extract other metadata from the result set.
}
\examples{
library(DBI)
db <- RSQLite::datasetsDb()

# Run query to get results as dataframe
dbGetQuery(db, "SELECT * FROM USArrests LIMIT 3")

# Send query to pull requests in batches
rs <- dbSendQuery(db, "SELECT * FROM USArrests")
dbFetch(rs, n = 2)
dbFetch(rs, n = 2)
dbHasCompleted(rs)
dbClearResult(rs)

# Parameterised queries are safest when you accept user input
dbGetQuery(db, "SELECT * FROM USArrests WHERE Murder < ?", list(3))

# Or create and then bind
rs <- dbSendQuery(db, "SELECT * FROM USArrests WHERE Murder < ?")
dbBind(rs, list(3))
dbFetch(rs)
dbClearResult(rs)

# Named parameters are a little more convenient
rs <- dbSendQuery(db, "SELECT * FROM USArrests WHERE Murder < :x")
dbBind(rs, list(x = 3))
dbFetch(rs)
dbClearResult(rs)
dbDisconnect(db)

# Passing multiple values is especially useful for statements
con <- dbConnect(RSQLite::SQLite())

dbWriteTable(con, "test", data.frame(a = 1L, b = 2L))
dbReadTable(con, "test")

dbExecute(con, "INSERT INTO test VALUES (:a, :b)",
          params = list(a = 2:4, b = 3:5))
dbReadTable(con, "test")

rs <- dbSendStatement(con, "DELETE FROM test WHERE a = :a AND b = :b")
dbBind(rs, list(a = 3:1, b = 2:4))
dbBind(rs, list(a = 4L, b = 5L))
dbClearResult(rs)
dbReadTable(con, "test")

# Multiple values passed to queries are executed one after another,
# the result appears as one data frame
dbGetQuery(con, "SELECT * FROM TEST WHERE a >= :a", list(a = 0:3))

dbDisconnect(con)

}
\seealso{
The corresponding generic functions
\code{\link[DBI:dbSendQuery]{DBI::dbSendQuery()}}, \code{\link[DBI:dbFetch]{DBI::dbFetch()}}, \code{\link[DBI:dbClearResult]{DBI::dbClearResult()}}, \code{\link[DBI:dbGetQuery]{DBI::dbGetQuery()}},
\code{\link[DBI:dbBind]{DBI::dbBind()}}, \code{\link[DBI:dbSendStatement]{DBI::dbSendStatement()}}, and \code{\link[DBI:dbExecute]{DBI::dbExecute()}}.
}
