% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/classify.r
\name{classify}
\alias{classify}
\title{Classification probabilities for lm.rrpp model fits}
\usage{
classify(fit, test.data = NULL, covariate.values = NULL,
  prior = NULL, CV = TRUE, pc.space = 1, inherent.groups = FALSE)
}
\arguments{
\item{fit}{A linear model fit using \code{\link{lm.rrpp}}.}

\item{test.data}{An optional matrix (or object coercible to a matrix) for classification.  If NULL,
all observed data are used.}

\item{covariate.values}{An optional data frame (with one value per covariate) to control covariate values.
For example, covariate.values = data.frame(x1 = 2, x2 = 4).  This option is used if estimation at a point on a
covariate axis other than the mean value is desired.}

\item{prior}{An option vector with values between 0 and 1 in the order of groups.  If the groups extracted from 
the \code{\link{lm.rrpp}} fit or there order are not clear, it is wise to use the inherent.groups = TRUE option first.
This will display the group information.  If NULL, prior probabilities will be based on group size.  A choice of 
"equal" will force equal prior probabilities.}

\item{CV}{A logical argument to indicate whether a leave-one-out cross-validation (Jackknife) procedure should be
performed.  This is only a logical choice if test data are the same data used for the \code{\link{lm.rrpp}} fit.
In other cases, the argument will be ignored.  The default is CV = TRUE.}

\item{pc.space}{A value between 0 and 1 to indicate the portion of PC space to use for classification.  The default is 
1.0 (the entire PC space).  Note that this is an upper limit, and the PCs included might (and probably do) comprise
less space.}

\item{inherent.groups}{A logical argument in case one wishes to have the inherent groups in the model fit revealed.  If 
TRUE, no other analysis will be done than to reveal the groups.  This argument should always be FALSE to perform 
a classification analysis.}
}
\value{
An object of class \code{classify} is a list containing the following
\item{means}{LS means (PC scores) for inherent groups.}
\item{group}{A factor derived for groups.}
\item{group.n}{Sample size for each group.}
\item{test.data}{PC scores for the test data used.}
\item{Mah.dist.sq}{Generalized Mahalanobis squared distance from test data to group means.}
\item{prior}{Prior probabilities.}
\item{posterior}{Posterior probabilities.}
\item{class}{Classification based on highest posterior probability}
\item{class.matrix}{An indicator matrix for classifications.}
}
\description{
Function calculates posterior probabilities for test data, based on factors in a lm.rrpp fit and 
values for continuous covariates.
}
\details{
This function uses a \code{\link{lm.rrpp}} fit, optional covariate values, optional prior probabilities,
and an optional cross-validation procedure to estimate posterior probabilities.  One can choose the 
principal component subspace in which to do the classification.  For most cases, 100 percent of the PC space is preferred; 
however, when there is a strong signal - groups have different means and observations within groups are close to means - it 
might be desirable to ascertain if the signal remains strong in fewer dimensions.  The pc.space
argument can be toggled, and one might wish to run several analyses with different selections, to get a feel
for how classification improves with increased data dimensionality.

For high dimensional data, training and testing data are first projected onto the principal components of the 
appropriate error covariance matrix from the \code{\link{lm.rrpp}} fit.  (The function will detect whether ordinary 
least squares or generalized least squares was used.)  The model covariance is then re-estimated in the 
defined amount of the PC space.  A ridge regularization step is also employed if it is determined that the covariance 
matrix is ill-conditioned.  In this space, generalized (Mahalanobis) distances are calculated and used to calculate 
Bayesian posterior probabilities, based on a multivariate normal density function.  An option for user-supplied 
prior probabilities is available, or a user can choose between equal or sample-size corrected prior probabilities.

This function should produce similar results as \code{\link[MASS]{lda}} for models that can be used in that 
function, but it will also calculate least squares means if there are covariates in the \code{\link{lm.rrpp}} fit.  
Furthermore, the user can choose specific covariate values for estimation.  For example, one could make classifications
at small organism size or large size (rather than mean size), if size is a covariate.  This is helpful if there are 
factor-covariate interactions to consider, as classification could differ across the span of a covariate.
The function does not find canonical (discriminant) axes, like \code{\link[MASS]{lda}}, however these can be calculated from 
\code{\link{lm.rrpp}} output.  Whereas as \code{\link[MASS]{lda}} only allows a single factor model, \code{classify} can
be used with various model types, and "discriminant" axes would not make as much sense for complex models.  Finally,
 \code{\link[MASS]{lda}} scales data and model design matrices to find (transformed) discriminant functions for within-group
 covariances that are forced to be spherical; i.e., the discriminant functions are normalized.  This is not a step
 taken with \code{classify}, as model design matrices might be more complex than single-factor designs, and obtaining 
 discriminant functions is not the goal.  Nevertheless, posterior probabilities will differ slightly between the two
 functions as a result.

Regardless of variables input, data are projected onto PCs.  The purpose of this function is to predict 
group association, and working in PC space facilitates this objective.

This is a new function and not all limits and scenarios have been tested before its release.  Please report 
any issues or limitations or strange results to the maintainer.  (The function will likely be updated frequently
until all kinks are worked out.)
}
\examples{

# Using the Pupfish data (see lm.rrpp help for more detail)

data(Pupfish)
Pupfish$logSize <- log(Pupfish$CS) 
fit <- lm.rrpp(coords ~ logSize + Sex * Pop, SS.type = "I", 
data = Pupfish, print.progress = FALSE, iter = 0)

classify(fit, inherent.groups = TRUE) # see groups available
class1 <- classify(fit, CV = TRUE)
summary(class1)

# Perfect classification

table(interaction(Pupfish$Sex, Pupfish$Pop), class1$class)

class2 <- classify(fit, CV = TRUE, pc.space = 0.6)
summary(class2)
table(interaction(Pupfish$Sex, Pupfish$Pop), class2$class)

# Allow for heterogenous slopes in the model
fit2 <- lm.rrpp(coords ~ logSize * Sex * Pop, SS.type = "I", 
data = Pupfish, print.progress = FALSE, iter = 0)

# Classification for small-sized fish (previous example for mean-sized fish)

class3 <- classify(fit2, CV = TRUE, 
covariate.values = data.frame(logSize = 4))
summary(class3)

# Classification for large-sized fish 

class4 <- classify(fit2, CV = TRUE, 
covariate.values = data.frame(logSize = 4.5))
summary(class4)
}
\author{
Michael Collyer
}
\keyword{analysis}
