% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/decision1S_boundary.R
\name{decision1S_boundary}
\alias{decision1S_boundary}
\alias{decision1S_boundary.betaMix}
\alias{decision1S_boundary.normMix}
\alias{decision1S_boundary.gammaMix}
\title{Decision Boundary for 1 Sample Designs}
\usage{
decision1S_boundary(prior, n, decision, ...)

\method{decision1S_boundary}{betaMix}(prior, n, decision, ...)

\method{decision1S_boundary}{normMix}(prior, n, decision, sigma, eps = 1e-06, ...)

\method{decision1S_boundary}{gammaMix}(prior, n, decision, eps = 1e-06, ...)
}
\arguments{
\item{prior}{Prior for analysis.}

\item{n}{Sample size for the experiment.}

\item{decision}{One-sample decision function to use; see \code{\link{decision1S}}.}

\item{...}{Optional arguments.}

\item{sigma}{The fixed reference scale. If left unspecified, the
default reference scale of the prior is assumed.}

\item{eps}{Support of random variables are determined as the
interval covering \code{1-eps} probability mass. Defaults to
\eqn{10^{-6}}.}
}
\value{
Returns the critical value \eqn{y_c}.
}
\description{
Calculates the decision boundary for a 1 sample design. This is the
critical value at which the decision function will change from 0
(failure) to 1 (success).
}
\details{
The specification of the 1 sample design (prior, sample
size and decision function, \eqn{D(y)}), uniquely defines the
decision boundary

\deqn{y_c = \max_y\{D(y) = 1\},}{y_c = max_{y}{D(y) = 1},}

which is the maximal value of \eqn{y} whenever the decision \eqn{D(y)}
function changes its value from 1 to 0 for a decision function
with \code{lower.tail=TRUE} (otherwise the definition is \eqn{y_c =
\max_{y}\{D(y) = 0\}}{y_c = max_{y}{D(y) = 0}}). The decision
function may change at most at a single critical value as only
one-sided decision functions are supported. Here,
\eqn{y} is defined for binary and Poisson endpoints as the sufficient
statistic \eqn{y = \sum_{i=1}^{n} y_i} and for the normal
case as the mean \eqn{\bar{y} = 1/n \sum_{i=1}^n y_i}.

The convention for the critical value \eqn{y_c} depends on whether
a left (\code{lower.tail=TRUE}) or right-sided decision function
(\code{lower.tail=FALSE}) is used. For \code{lower.tail=TRUE} the
critical value \eqn{y_c} is the largest value for which the
decision is 1, \eqn{D(y \leq y_c) = 1}, while for
\code{lower.tail=FALSE} then \eqn{D(y > y_c) = 1} holds. This is
aligned with the cumulative density function definition within R
(see for example \code{\link{pbinom}}).
}
\section{Methods (by class)}{
\itemize{
\item \code{betaMix}: Applies for binomial model with a mixture
beta prior. The calculations use exact expressions.

\item \code{normMix}: Applies for the normal model with known
standard deviation \eqn{\sigma} and a normal mixture prior for the
mean. As a consequence from the assumption of a known standard
deviation, the calculation discards sampling uncertainty of the
second moment. The function \code{decision1S_boundary} has an extra
argument \code{eps} (defaults to \eqn{10^{-6}}). The critical value
\eqn{y_c} is searched in the region of probability mass
\code{1-eps} for \eqn{y}.

\item \code{gammaMix}: Applies for the Poisson model with a gamma
mixture prior for the rate parameter.  The function
\code{decision1S_boundary} takes an extra argument \code{eps} (defaults to \eqn{10^{-6}})
which determines the region of probability mass \code{1-eps} where
the boundary is searched for \eqn{y}.
}}

\examples{

# non-inferiority example using normal approximation of log-hazard
# ratio, see ?decision1S for all details
s <- 2
flat_prior <- mixnorm(c(1,0,100), sigma=s)
nL <- 233
theta_ni <- 0.4
theta_a <- 0
alpha <- 0.05
beta  <- 0.2
za <- qnorm(1-alpha)
zb <- qnorm(1-beta)
n1 <- round( (s * (za + zb)/(theta_ni - theta_a))^2 )
theta_c <- theta_ni - za * s / sqrt(n1)

# double criterion design
# statistical significance (like NI design)
dec1 <- decision1S(1-alpha, theta_ni, lower.tail=TRUE)
# require mean to be at least as good as theta_c
dec2 <- decision1S(0.5, theta_c, lower.tail=TRUE)
# combination
decComb <- decision1S(c(1-alpha, 0.5), c(theta_ni, theta_c), lower.tail=TRUE)

# critical value of double criterion design
decision1S_boundary(flat_prior, nL, decComb)

# ... is limited by the statistical significance ...
decision1S_boundary(flat_prior, nL, dec1)

# ... or the indecision point (whatever is smaller)
decision1S_boundary(flat_prior, nL, dec2)

}
\seealso{
Other design1S: 
\code{\link{decision1S}()},
\code{\link{oc1S}()},
\code{\link{pos1S}()}
}
\concept{design1S}
