% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/base-ValueObject.R
\name{ValueObject}
\alias{ValueObject}
\title{Value Object Pattern}
\usage{
ValueObject(given = NA_character_, family = NA_character_)
}
\arguments{
\item{given}{(\code{character}) A character vector with the given name.}

\item{family}{(\code{character}) A character vector with the family name.}
}
\description{
Model a domain concept using natural lingo of the domain
experts, such as "Passenger", "Address", and "Money".
}
\details{
<div class="alert alert-danger">
**Caution:** ValueObject is designed for demonstration purposes. Instead of directly using the design pattern as it appears in the package, you'd have to adjust the source code to the problem you are trying to solve.
</div>
<!-- One line about what the function does -->


A \strong{Value Object} models a domain concept using natural lingo of the domain
experts, such as "Passenger", "Address", and "Money".

Any \strong{Value Object} is created by a function that receives input, applies some
transformations, and outputs the results in some data structure such as a
vector, a list or a data.frame.
\subsection{How It Works}{

In R, a good option for creating a \strong{Value Object} is to follow two
instructions:
\itemize{
\item A \strong{Value Object} is created by a \code{function}, rather than a class method;
and
\item A \strong{Value Object} returns a \code{tibble}, rather than a list or a vector.
}

In essence, a \strong{Value Object} is a data type, like \code{integer}, \code{logical}, \code{Date}
or \code{data.frame} data types to name a few. While the built-in data types in R fit
any application, \strong{Value Objects} are domain specific and as such, they fit
only to a specific application. This is because, \code{integer} is an abstract that
represent whole numbers. This abstract is useful in any application. However, a
\strong{Value Object} represent a high-level abstraction that appears in a particular
domain.

An example of a \strong{Value Object} is the notion of a "Person". Any person in the
world has a name. Needless to say, a person name is spelt by letters, rather
than numbers. A \strong{Value Object} captures these attribute as \code{tibble} columns
and type checks:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{Person <- function(given = NA_character_, family = NA_character_)\{
  stopifnot(is.character(given), is.character(family))
  stopifnot(length(given) == length(family))
  
  return(
    tibble::tibble(given = given, family = family)
    \%>\% tidyr::drop_na(given)
  )
\}
}\if{html}{\out{</div>}}

Instantiating a person \strong{Value Object} is done by calling the \code{Person}
constructor function:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{person <- Person(given = "Bilbo", family = "Baggins")
}\if{html}{\out{</div>}}

Getting to know the advantages of a \strong{Value Object}, we should consider the
typical alternative -- constructing a Person by using the \code{tibble} function
directly:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{person <- tibble::tibble(given = "Bilbo", family = "Baggins")
}\if{html}{\out{</div>}}

Both implementations return objects with identical content and structure, that
is, their column names, column types and cell values are identical. Then, why
would one prefer using a \strong{Value Object} and its constructor over the direct
alternative?

There are four predominant qualities offered by the \strong{Value Object} pattern
which are not offered by the alternative:
\enumerate{
\item Readability. Each \strong{Value Object} captures a concept belonging to the
problem domain. Rather than trying to infer what a \code{tibble} is by looking at
its low-level details, the \strong{Value Object} constructor descries a context
on a high-level.
\item Explicitness. Since the constructor of the \strong{Value Object} is a function,
its expected input arguments and their type can be detailed in a helper
file. Moreover, assigning input arguments with default values of specific
type, such as \code{NA} (logical NA), \code{NA_integer_}, \code{NA_character_}, or
\code{NA_Date} (see \code{lubridate::NA_Date}), expresses clearly the variable types
of the \strong{Value Object}.
\item Coherence. The representation of a \strong{Value Object} is concentrated in one
place -- its constructor. Any change, mainly modifications and extensions,
applied to the constructor promise the change would propagate to all
instances of the Value Objects. That means, no structure discrepancies
between instances that are supposed to represent the same concept.
\item Safety. The constructor may start with \href{https://en.wikipedia.org/wiki/Defensive_programming}{defensive programming} to ensure
the qualities of its input. One important assertion is type checking. Type
checking eliminated the risk of implicit type coercing. Another important
assertion is checking if the lengths of the input arguments meet some
criteria, say all inputs are of the same length, or more restrictively, all
inputs are scalars. Having a set of checks makes the code base more robust.
This is because \strong{Value Objects} are regularly created with the output of
other functions calls, having a set of checks serves as pseudo-tests of
these functions output throughout the code.
}

In addition to these qualities, there are two desirable behaviours which are not
offered by directly calling \code{tibble}:
\enumerate{
\item Null Value Object. Calling the \strong{Value Object} constructor with no input
arguments returns the structure of the \code{tibble} (column names and column
types).
\item Default values for missing input arguments. In this manner, the \strong{Value
Object} has a well-defined behaviour for a person without a family name,
such as Madonna and Bono.
}

In addition to native R data types, a \strong{Value Object} constructor can receive
other \strong{Value Objects} as input arguments. Here are two examples that transmute
Person to other Person-based concepts:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# A Passenger is a Person with a flight booking reference
Passenger <- function(person = Person(), booking_reference = NA_character_)\{
  stopifnot(all(colnames(person) \%in\% colnames(Person())))
  stopifnot(is.character(booking_reference))
  
  return(
    person 
    \%>\% tibble::add_column(booking_reference = booking_reference)
    \%>\% tidyr::drop_na(booking_reference)
  )
\}

person <- Person(given = "Bilbo", family = "Baggins")
passenger <- Passenger(person = person, booking_reference = "B662HR")
print(passenger)
#> # A tibble: 1 × 3
#>   given family  booking_reference
#>   <chr> <chr>   <chr>            
#> 1 Bilbo Baggins B662HR
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# A Diner is a Person that may have dinner reservation
Diner <- function(person = Person(), reservation_time = NA_POSIXct_)\{
  stopifnot(all(colnames(person) \%in\% colnames(Person())))
  stopifnot(is.POSIXct(reservation_time))
  
  return(
    person 
    \%>\% tibble::add_column(reservation_time = reservation_time)
  )
\}

person <- Person(given = "Bilbo", family = "Baggins")
timestamp <- as.POSIXct("2021-01-23 18:00:00 NZDT")
diner <- Diner(person = person, reservation_time = timestamp)
print(diner)
#> # A tibble: 1 × 3
#>   given family  reservation_time   
#>   <chr> <chr>   <dttm>             
#> 1 Bilbo Baggins 2021-01-23 18:00:00
}\if{html}{\out{</div>}}
}

\subsection{When to Use It}{
\itemize{
\item In situations where domain concepts are more important then the database
schema. For example, when you are modelling Passengers, your first instinct
might be to think about the different data sources you'd need for the
analysis. You may envision "FlightDetails" and "CustomerDetails". Next you
will define the relationship between them. Instead, let the domain drive the
design. Create a Passenger \strong{Value Object} with the attributes you must
have, regardless of any particular database schema.
\item In a function that runs within a specific context. Rather than having an
input argument called \code{data} of type \code{data.frame}, use the appropriate
\strong{Value Object} name and pass it its constructor.
}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{Audience <- Person

## Without a Value Object
clean_audience_data <- function(data) 
  dplyr::mutate(.data = data, given = stringr::str_to_title(given))

## With a Value Object
clean_audience_data <- function(attendees = Audience()) 
  dplyr::mutate(.data = attendees, given = stringr::str_to_title(given))
}\if{html}{\out{</div>}}
\itemize{
\item In \href{https://en.wikipedia.org/wiki/Pipeline_(software)}{pipes and filters}
architecture.
}<div class="alert alert-warning">
**Note:** **Value Objects** do not need to have unit-tests. This is because of two reasons: 
 (1) **Value Objects** are often called by other functions that are being tested. That means, **Value Objects** are implicitly tested. 
 (2) **Value Objects** are data types similarly to 'data.frame' or 'list'. As such, they need no testing
</div>

}
}
\examples{
# See more examples at <https://tidylab.github.io/R6P/articles>

# In this example we are appointing elected officials to random ministries, just
# like in real-life.
Person <- ValueObject
Person()

# Create a test for objects of type Person
# * Extract the column names of Person by using its Null Object (returned by Person())
# * Check that the input argument has all the columns that a Person has
is.Person <- function(x) all(colnames(x) \%in\% colnames(Person()))

# A 'Minister' is a 'Person' with a ministry title. We capture that information
# in a new Value Object named 'Minister'.
# The Minister constructor requires two inputs:
# 1. (`Person`) Members of parliament
# 2. (`character`) Ministry titles
Minister <- function(member = Person(), title = NA_character_){
    stopifnot(is.Person(member), is.character(title))
    stopifnot(nrow(member) == length(title) | all(is.na(title)))

    member \%>\% dplyr::mutate(title = title)
}

# Given one or more parliament members
# When appoint_random_ministries is called
# Then the parliament members are appointed to an office.
appoint_random_ministries <- function(member = Person()){
    positions <- c(
        "Arts, Culture and Heritage", "Finance", "Corrections",
        "Racing", "Sport and Recreation", "Housing", "Energy and Resources",
        "Education", "Public Service", "Disability Issues", "Environment",
        "Justice", "Immigration", "Defence", "Internal Affairs", "Transport"
    )

   Minister(member = member, title = sample(positions, size = nrow(member)))
}

# Listing New Zealand elected officials in 2020, we instantiate a Person Object,
# appoint them to random offices and return a Member value object.
set.seed(2020)

parliament_members <- Person(
    given = c("Jacinda", "Grant",     "Kelvin", "Megan", "Chris",   "Carmel"),
    family = c("Ardern", "Robertson", "Davis",  "Woods", "Hipkins", "Sepuloni")
)

parliament_members

appoint_random_ministries(member = parliament_members)
}
\seealso{
Other base design patterns: 
\code{\link{NullObject}()},
\code{\link{Singleton}}
}
\concept{base design patterns}
