\name{PI chart functions: makeChart, findmin, solveChart}

\alias{makeChart}
\alias{findmin}
\alias{solveChart}

\title{Create and solve a prime implicants chart}

\description{
These functions help creating a demo for a prime implicant chart, and also
show how to solve it using a minimum number of prime implicants.  
}

\usage{
makeChart(primes = "", configs = "", snames = "", mv = FALSE, collapse = "*", ...)

findmin(chart, ...)

solveChart(chart, row.dom = FALSE, all.sol = FALSE, depth = NULL, max.comb = 0,
           first.min = FALSE, ...)
}

\arguments{
  \item{primes}{A string containing prime implicants, separated by commas, or a
        matrix of implicants.}                                
  \item{configs}{A string containing  causal configurations, separated by commas,
        or a matrix of causal configurations in the implicants space.}
  \item{snames}{A string containing the sets' names, separated by commas.}
  \item{mv}{Logical, row and column names in multi-value notation.}
  \item{collapse}{Scalar character, how to collapse different parts of the row or
        column names.}
  \item{chart}{An object of class \code{"QCA_pic"} or a logical matrix.}
  \item{row.dom}{Logical, apply row dominance to eliminate redundant prime implicants.}
  \item{all.sol}{Derive all possible solutions, irrespective if the disjunctive number of
        prime implicants is minimal or not.}
  \item{depth}{A maximum number of prime implicants for any disjunctive solution.}
  \item{max.comb}{Numeric, to stop searching for solutions (see Details).}
  \item{first.min}{Logical, to return only the very first minimal solution (see
        Details).}
  \item{...}{Other arguments (mainly for backwards compatibility).}
}

\details{
A PI chart, in this package, is a logical matrix (with \code{TRUE}/\code{FALSE} values),
containing the prime implicants on the rows and the observed positive output configurations
on the columns. Such a chart is produced by \bold{\code{makeChart()}}, and it is useful to
visually determine which prime implicants (if any) are essential.

When \bold{\code{primes}} and \bold{\code{configs}} are character, the individual sets
are identified using the function \bold{\code{\link[admisc]{translate}()}}, using the SOP - Sum Of
Products form, which needs the set names in the absence of any other information. If
products are formed using the standard \bold{\code{*}} operator, specifying the set names
is not mandatory.

When \bold{\code{primes}} and \bold{\code{configs}} are matrices, they have to be specified
at implicants level, where the value \bold{\code{0}} is interpreted as a minimized literal.

The chart is subsequently processed algorithmically by \bold{\code{solveChart()}} to find
the absolute minimal number \code{M} of rows (prime implicants) necessary to cover all columns,
then searches through all possible combinations of \code{M} rows, to find those which actually
cover the columns.

The number of all possible combinations of \code{M} rows increases exponentially with the
number of prime implicants generated by the Quine-McCluskey minimization procedure, and the
solving time quickly grows towards infinity for large PI charts.

To solve the chart in a minimal time, the redundant prime implicants need to first be eliminated.
This is the purpose of the argument \bold{\code{row.dom}}. When activated, it eliminates the
dominated rows (those which cover a smaller number of columns than another, dominant prime
implicant).

The identification of the full model space (including the non-minimal solutions) requires the
entire PI chart and is guaranteed to consume a lot of time (towards infinity for very large PI
charts). This is done by activating the argument \bold{\code{all.sol}}, which automatically
deactivates the argument \bold{\code{row.dom}}.

The argument \bold{\code{depth}} is relevant only when the argument \bold{\code{all.sol}} is
activated, and it is automatically increased if the minimal number of rows \code{M} needed to
cover all columns is larger. By default, it bounds the disjunctive solutions to at most 5 prime
implicants, but this number can be increased to widen the search space, with a cost of
increasing the search time.

The argument \bold{\code{max.comb}} sets a maximum number of combinations to find solutions.
It is counted in (fractions of) billions, defaulted at zero to signal searching to the maximum
possible extent. If too complex, the search is stopped and the algorithm returns all found solutions
up to that point.

For extremly difficult PI charts, the argument \bold{\code{first.min}} controls returning only
one (the very first found) solution.
}

\value{
For \bold{\code{makeChart}}: a logical matrix of class \code{"QCA_pic"}.

For \bold{\code{findmin}}: a numerical scalar.

For \bold{\code{solveChart}}: a matrix containing all possible combinations of PI chart rows
necessary to cover all its columns.
}

\author{
Adrian Dusa
}

\references{
Quine, W.V. (1952) \emph{The Problem of Simplifying Truth Functions},
The American Mathematical Monthly, vol.59, no.8. (Oct., 1952), pp.521-531.

Ragin, Charles C. (1987) \emph{The Comparative Method. Moving beyond qualitative and
quantitative strategies}, Berkeley: University of California Press
}

\examples{
# non-standard products, it needs the set names
chart <- makeChart("a, b, ~c", "abc, a~b~c, a~bc, ~ab~c")

# same with unquoted expressions
chart <- makeChart(c(a, b, ~c), c(abc, a~b~c, a~bc, ~ab~c))

chart
#      abc   a~b~c a~bc  ~ab~c
# a      x     x     x     -  
# b      x     -     -     x  
# ~c     -     x     -     x

findmin(chart)
# 2

solveChart(chart)
# first and second rows (a + b)
# and first and third rows (a + ~c)
# a is an essential prime implicant
#      a + b  a + ~c
#      [,1]   [,2]
# [1,]    1      1
# [2,]    2      3

# using SOP standard product sign
rows <- "EF, ~GH, IJ"
cols <- "~EF*~GH*IJ, EF*GH*~IJ, ~EF*GH*IJ, EF*~GH*~IJ"
chart <- makeChart(rows, cols)
chart
#     ~EF*~GH*IJ EF*GH*~IJ ~EF*GH*IJ EF*~GH*~IJ
# EF      -          x         -         x     
# ~GH     x          -         -         x     
# IJ      x          -         x         -     

solveChart(chart)
# ~GH is redundant
#     EF + IJ
#      [,1]
# [1,]    1
# [2,]    3


# using implicant matrices
primes <- matrix(c(2,2,1,0,2,2,0,2,2,2), nrow = 2)
configs <- matrix(c(2,2,2,1,1,2,2,2,2,1,2,2,2,2,2), nrow = 3)
colnames(primes) <- colnames(configs) <- letters[1:5]

# the prime implicants: a~bce and acde
primes
#      a b c d e
# [1,] 2 1 2 0 2
# [2,] 2 0 2 2 2

# the initial causal combinations: a~bc~de, a~bcde and abcde
configs
#      a b c d e
# [1,] 2 1 2 1 2
# [2,] 2 1 2 2 2
# [3,] 2 2 2 2 2

chartLC <- makeChart(primes, configs, collapse = "")
chartLC
#         a~bc~de a~bcde   abcde  
# a~bce      x       x       -   
# acde       -       x       x   

}

\keyword{functions}
