% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nicfi_utils.R
\name{nicfi_crop_images}
\alias{nicfi_crop_images}
\title{Crop the downloaded NICFI .tif or .vrt file using 'gdalwarp'}
\usage{
nicfi_crop_images(
  input_pth,
  output_pth,
  bbox_AOI,
  threads = 1,
  of = "GTiff",
  resize_method = "lanczos",
  return_raster = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{input_pth}{a valid path to either a .tif or a .vrt (virtual raster) file that should be cropped based on the bounding box using 'gdalwarp'}

\item{output_pth}{a valid path to the output .tif file. This file path can also point to a .vrt file by setting the 'of' parameter to 'VRT'}

\item{bbox_AOI}{a list of the format "list(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax)" that includes the bounding box 'xmin', 'xmax', 'ymin', 'ymax' coordinate values of the Area of Interest (AOI)}

\item{threads}{an integer. In case that this parameter is greater than 1 then multiple threads will be utilized in the 'gdalwarp' function}

\item{of}{a character string specifying the format of the saved file. The default is GeoTIFF (GTiff). For more information see the 'gdalwarp' function of the 'gdalUtils' package}

\item{resize_method}{a character string specifying the resize method. Can be one of 'near', 'bilinear', 'cubic', 'cubicspline', 'lanczos', 'average', 'mode', 'max', 'min', 'med', 'q1', 'q3'. For more information see the 'r' parameter of  the 'gdalwarp' function of the 'gdalUtils' package}

\item{return_raster}{a boolean. If TRUE then the function will return the cropped image as a RasterBrick too}

\item{verbose}{a boolean. If TRUE then information will be printed out in the console}
}
\value{
either NULL or a RasterBrick (if the parameter 'return_raster' is set to TRUE)
}
\description{
Crop the downloaded NICFI .tif or .vrt file using 'gdalwarp'
}
\examples{

\dontrun{

require(PlanetNICFI)

#....................................
# first extract the available Mosaics
#....................................

api_key = 'use_your_planet_nicfi_API_key'

mosaic_files = nicfi_mosaics(planet_api_key = api_key,
                             type = 'monthly',
                             crs_bbox = 4326,
                             URL = 'https://api.planet.com/basemaps/v1/mosaics',
                             verbose = TRUE)

#....................................
# keep the mosaic of 'September 2020'
#....................................

keep_idx = 1
mosaic_ID = mosaic_files$dtbl_mosaic$id[keep_idx]


#................................................................
# then extract the available Quad files for the Mosaic for an AOI
#................................................................

wkt_file = system.file('data_files/Sugar_Cane_Bolivia.wkt', package = "PlanetNICFI")
WKT = readLines(wkt_file, warn = FALSE)

quad_files = nicfi_quads_bbox(planet_api_key = api_key,
                              mosaic_id = mosaic_ID,
                              bbox_AOI = NULL,
                              wkt_AOI = WKT,
                              page_size = 10,
                              crs_bbox = 4326,
                              verbose = TRUE)

#..................................
# formated aria2c download weblinks
#..................................

web_links_aria2c = aria2c_download_paths(mosaic_output = mosaic_files,
                                         mosaic_id = mosaic_ID,
                                         quads_output = quad_files,
                                         img_type = 'tif')

#.........................................................
# download the .tif files that intersect with the bbox AOI
#.........................................................

temp_dir_out = tempdir()

all_threads = parallel::detectCores()
set_threads = length(web_links_aria2c) / 2
num_threads = ifelse(set_threads < all_threads, set_threads, all_threads)
aria_args = '--allow-overwrite --file-allocation=none --retry-wait=5 --max-tries=0'

res_downl = aria2c_bulk_donwload(vector_or_file_path = web_links_aria2c,
                                 default_directory = temp_dir_out,
                                 user = NULL,
                                 password = NULL,
                                 threads = num_threads,
                                 verbose = TRUE,
                                 secondary_args_aria = aria_args)

#........................................
# create a Virtual Raster (VRT) file from
# the downloaded .tif files
#........................................

VRT_out = file.path(temp_dir_out, glue::glue("{mosaic_ID}.vrt"))

res_vrt = create_VRT_from_dir(dir_tifs = temp_dir_out,
                              output_path_VRT = VRT_out,
                              file_extension = '.tif',
                              verbose = TRUE)

#....................................................
# Adjust the Coordinate Reference System of the
# bounding box from 4326 to the one of the .tif files
#....................................................

wkt_sf = sf::st_as_sfc(WKT, crs = 4326)
proj_info = proj_info_extract(path_to_raster = VRT_out)

wkt_transf = sf::st_transform(wkt_sf, crs = proj_info)
bbx_transf = sf::st_bbox(wkt_transf)


#....................................................
# crop the output .vrt file based on the bounding box
#....................................................

pth_crop_out = file.path(temp_dir_out, glue::glue("{mosaic_ID}_CROPPED.tif"))

bbx_crop = list(xmin = as.numeric(bbx_transf['xmin']),
                xmax = as.numeric(bbx_transf['xmax']),
                ymin = as.numeric(bbx_transf['ymin']),
                ymax = as.numeric(bbx_transf['ymax']))

warp_obj = nicfi_crop_images(input_pth = VRT_out,
                             output_pth = pth_crop_out,
                             bbox_AOI = bbx_crop,
                             threads = num_threads,
                             of = 'GTiff',
                             resize_method = 'lanczos',
                             verbose = TRUE)
}
}
