

#' @title Get Test Data
#'
#' @description Some data to test the functionality of the package
#'
#' @param lab A numeric vector indicating from which laboratory the data should be taken.
#'
#' @return If a lab is not NULL, a data.frame with three columns ('X', 'S', 'N') is returned. If lab is NULL, these three columns are supplemented by a fourth column indicating the laboratory.
#'
#' Data grohmann2015collaborative was generated by Grohmann et al. and has been used as exemplary data by Uhlig et al. to assess performance of their statistical approach to validate PCR results. 
#' Data sas.logistic was taken from the part of the SAS manual dealing with logistic regression (\url{https://support.sas.com/documentation/onlinedoc/stat/ex_code/132/logiex14.html}).
#'
#' @references
#' Grohmann et al. Accred Qual Assur (2015) 20: 85. https://doi.org/10.1007/s00769-015-1108-5
#' Uhlig et al. Accred Qual Assur (2015) 20: 75. https://doi.org/10.1007/s00769-015-1112-9
#'
#' @name testdata
#' @aliases grohmann2015collaborative sas.logistic
#'
#' @examples
#' x.all <- grohmann2015collaborative()
#' x.5 <- grohmann2015collaborative(5)
#' sas <- sas.logistic()
NULL

#' @rdname testdata
#' @export
grohmann2015collaborative <- function(lab=NULL){
    N <- 6
    X <- c(0.1, 1, 2, 5, 10, 20)
    LAB <- rbind(
        c(0, 3, 5, 5, 6, 6),
        c(0, 4, 6, 6, 6, 6),
        c(1, 0, 5, 6, 6, 6),
        c(0, 4, 3, 6, 6, 6),
        c(0, 0, 5, 6, 6, 6),
        c(0, 4, 6, 6, 6, 6),
        c(0, 5, 6, 6, 6, 6),
        c(0, 5, 6, 6, 6, 6),
        c(0, 6, 4, 6, 6, 6),
        c(0, 2, 5, 6, 6, 6),
        c(0, 1, 6, 6, 6, 6),
        c(0, 4, 6, 6, 6, 6),
        c(0, 4, 4, 6, 6, 6),
        c(0, 3, 3, 5, 6, 6),
        c(0, 6, 5, 6, 6, 6),
        c(0, 2, 6, 6, 6, 6),
        c(1, 4, 6, 5, 6, 6)
    )
    out <- NULL
    if(is.null(lab) || length(lab)>1){
        RNG <- 1:nrow(LAB)
        if( length(lab)>1 ){ RNG <- intersect(RNG, lab) }
        for(i in RNG){
            out <- rbind(out, cbind( i, X, LAB[i,], N ))
        }
        colnames(out) <- c("lab", "X", "S", "N")
    }else{
        out <- cbind( X, LAB[lab,], N )
        colnames(out) <- c("X", "S", "N")
    }

return(as.data.frame(out))
}

#' @rdname testdata
#' @export
sas.logistic <- function(){
  x <- rbind(
    c(1.5, 123, 8),
    c(4.0, 132, 6),
    c(7.5, 182, 18),
    c(12.5, 140, 14),
    c(17.5, 138, 20),
    c(25.0, 161, 39),
    c(35.0, 133, 19),
    c(47.0, 92,25),
    c(60.0, 74,44)
  )
  colnames(x) <- c("X", "N", "S")

return(as.data.frame(x[, c("X", "S", "N")]))
}
