% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bwsManyOneTest.R
\name{bwsManyOneTest}
\alias{bwsManyOneTest}
\alias{bwsManyOneTest.default}
\alias{bwsManyOneTest.formula}
\alias{bwsManyOneTest}
\title{BWS Many-To-One Comparison Test}
\usage{
bwsManyOneTest(x, ...)

\method{bwsManyOneTest}{default}(x, g, alternative = c("two.sided", "greater",
  "less"), method = c("BWS", "Murakami", "Neuhauser"),
  p.adjust.method = p.adjust.methods, ...)

\method{bwsManyOneTest}{formula}(formula, data, subset, na.action,
  alternative = c("two.sided", "greater", "less"), method = c("BWS",
  "Murakami", "Neuhauser"), p.adjust.method = p.adjust.methods, ...)
}
\arguments{
\item{x}{a numeric vector of data values, or a list of numeric data
vectors.}

\item{\dots}{further arguments to be passed to or from methods.}

\item{g}{a vector or factor object giving the group for the
corresponding elements of \code{"x"}.
Ignored with a warning if \code{"x"} is a list.}

\item{alternative}{the alternative hypothesis. Defaults to \code{two.sided}.}

\item{method}{a character string specifying the test statistic to use. Defaults to \code{BWS}.}

\item{p.adjust.method}{method for adjusting p values (see \code{\link{p.adjust}}).}

\item{formula}{a formula of the form \code{response ~ group} where
\code{response} gives the data values and \code{group} a vector or
factor of the corresponding groups.}

\item{data}{an optional matrix or data frame (or similar: see
\code{\link{model.frame}}) containing the variables in the
formula \code{formula}.  By default the variables are taken from
\code{environment(formula)}.}

\item{subset}{an optional vector specifying a 
subset of observations to be used.}

\item{na.action}{a function which indicates what should happen when
the data contain \code{NA}s.  Defaults to \code{getOption("na.action")}.}
}
\value{
A list with class \code{"PMCMR"} containing the following components:
\describe{
 \item{method}{a character string indicating what type of test was performed.}
 \item{data.name}{a character string giving the name(s) of the data.}
 \item{statistic}{lower-triangle matrix of the estimated
quantiles of the pairwise test statistics.}
 \item{p.value}{lower-triangle matrix of the p-values for the pairwise tests.}
 \item{alternative}{a character string describing the alternative hypothesis.}
\item{p.adjust.method}{a character string describing the method for p-value
adjustment.}
\item{model}{a data frame of the input data.}
\item{dist}{a string that denotes the test distribution.}
}
}
\description{
Performs Baumgartner-Weiß-Schindler many-to-one comparison test.
}
\details{
For many-to-one comparisons (pairwise comparisons with one control)
in an one-factorial layout with non-normally distributed
residuals Baumgartner-Weiß-Schindler's non-parametric test can be performed.
Let there be \eqn{k} groups including the control,
then the number of treatment levels is \eqn{m = k - 1}.
Then \eqn{m} pairwise comparisons can be performed between
the \eqn{i}-th treatment level and the control.
H\eqn{_i: F_0 = F_i} is tested in the two-tailed case against
A\eqn{_i: F_0 \ne F_i, ~~ (1 \le i \le m)}.

This function is a wrapper function that sequentially
calls \code{\link[BWStest]{bws_stat}} and \code{\link[BWStest]{bws_cdf}}
for each pair. For the default test method (\code{"BWS"}) the original
Baumgartner-Weiß-Schindler test statistic B and its corresponding Pr(>|B|)
is calculated. For \code{method == "BWS"} only a two-sided test is possible.

For \code{method == "Murakami"} the modified BWS statistic
denoted B* and its corresponding Pr(>|B*|) is computed by sequentially calling
\code{\link[BWStest]{murakami_stat}} and \code{\link[BWStest]{murakami_cdf}}.
For \code{method == "Murakami"} only a two-sided test is possible.

If \code{alternative == "greater"} then the alternative, if one
population is stochastically larger than the other is tested:
H\eqn{_i: F_0 = F_i} against A\eqn{_i: F_0 \ge F_i, ~~ (1 \le i \le m)}.
The modified test-statistic B* according to Neuhäuser (2001) and its
corresponding Pr(>B*) or Pr(<B*) is computed by sequentally calling
\code{\{link[BWStest]{murakami_stat}} and \code{\{link[BWStest]{murakami_cdf}}
with \code{flavor = 2}.

The p-values can be adjusted to account for Type I error
inflation using any method as implemented in \code{\link{p.adjust}}.
}
\examples{
out <- bwsManyOneTest(weight ~ group, PlantGrowth, p.adjust="holm")
summary(out)

## A two-sample test
set.seed(1245)
x <- c(rnorm(20), rnorm(20,0.3))
g <- gl(2, 20)
summary(bwsManyOneTest(x ~ g, alternative = "less", p.adjust="none"))
summary(bwsManyOneTest(x ~ g, alternative = "greater", p.adjust="none"))

\dontrun{
## Check with the implementation in package BWStest
BWStest::bws_test(x=x[g==1], y=x[g==2], alternative = "less")
BWStest::bws_test(x=x[g==1], y=x[g==2], alternative = "greater")
}
}
\references{
Baumgartner, W., Weiss, P., Schindler, H. (1998), A nonparametric test for the
general two-sample problem, \emph{Biometrics} 54, 1129--1135.

Murakami, H. (2006) K-sample rank test based on modified Baumgartner statistic and its power
comparison, \emph{J. Jpn. Comp. Statist.} 19, 1--13.

Neuhäuser, M. (2001) One-side two-sample and trend tests based on a modified
Baumgartner-Weiss-Schindler statistic.
\emph{Journal of Nonparametric Statistics}, 13, 729--739.
}
\seealso{
\code{\link[BWStest]{murakami_stat}}, \code{\link[BWStest]{murakami_cdf}},
 \code{\link[BWStest]{bws_stat}}, \code{\link[BWStest]{bws_cdf}}.
}
\concept{
ManyOneComparison
}
\keyword{htest}
\keyword{nonparametric}
