% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_factorizations.R
\name{factorizations}
\alias{factorizations}
\alias{fact_pin_eho}
\alias{fact_pin_lk}
\alias{fact_pin_e}
\alias{fact_mpin}
\alias{fact_adjpin}
\title{Factorizations of the different PIN likelihood functions}
\usage{
fact_pin_eho(data, parameters = NULL)

fact_pin_lk(data, parameters = NULL)

fact_pin_e(data, parameters = NULL)

fact_mpin(data, parameters = NULL)

fact_adjpin(data, parameters = NULL)
}
\arguments{
\item{data}{A dataframe with 2 variables: the first
corresponds to buyer-initiated trades (buys), and the second corresponds
to seller-initiated trades (sells).}

\item{parameters}{In the case of the \code{PIN} likelihood
factorization, it is an ordered numeric vector (\eqn{\alpha}, \eqn{\delta},
\eqn{\mu}, \eb, \es). In the case of the \code{MPIN} likelihood factorization,
it is an ordered numeric vector (\strong{\eqn{\alpha}}, \strong{\eqn{\delta}},
\strong{\eqn{\mu}}, \eb, \es), where \strong{\eqn{\alpha}}, \strong{\eqn{\delta}},
and \strong{\eqn{\mu}} are numeric vectors of size \code{J}, where \code{J} is the
number of information layers in the data.
In the case of the \code{AdjPIN} likelihood factorization, it is an ordered
numeric vector (\eqn{\alpha}, \eqn{\delta}, \eqn{\theta}, \eqn{\theta'},
\eb, \es, \mub, \mus, \Db, \Ds). The default value is \code{NULL}.}
}
\value{
If the argument \code{parameters} is omitted, returns a function
object that can be used with the optimization functions \code{optim()},
and \code{neldermead()}.

If the argument \code{parameters} is provided, returns a numeric value of the
log-likelihood function evaluated at the dataset \code{data} and the
parameters \code{parameters}, where \code{parameters} is a numeric vector
following this order (\eqn{\alpha}, \eqn{\delta}, \eqn{\mu}, \eb, \es)
for the factorizations of the \code{PIN} likelihood function, (\strong{\eqn{\alpha}},
\strong{\eqn{\delta}}, \strong{\eqn{\mu}}, \eb, \es) for the factorization of the
\code{MPIN} likelihood function, and (\eqn{\alpha}, \eqn{\delta}, \eqn{\theta},
\eqn{\theta'}, \eb, \es ,\mub, \mus, \Db, \Ds) for the factorization of
the \code{AdjPIN} likelihood function.
}
\description{
The \code{PIN} likelihood function is derived from the original \code{PIN} model as
developed by \insertCite{Easley1992;textual}{PINstimation} and
\insertCite{Easley1996;textual}{PINstimation}. The maximization of the
likelihood function as is leads to computational problems, in particular,
to floating point errors. To remedy to this issue, several
log-transformations or factorizations of the different \code{PIN} likelihood
functions have been suggested.
The main factorizations in the literature are:

\itemize{
\item \code{fact_pin_eho()}: factorization of
\insertCite{Easley2010;textual}{PINstimation}
\item \code{fact_pin_lk()}: factorization of
\insertCite{WilliamLin2011;textual}{PINstimation}
\item \code{fact_pin_e()}: factorization of
\insertCite{Ersan2016;textual}{PINstimation}
}

The factorization of the likelihood function of the multilayer \code{PIN} model,
as developed in \insertCite{Ersan2016;textual}{PINstimation}.
\itemize{
\item \code{fact_mpin()}: factorization of
\insertCite{Ersan2016;textual}{PINstimation}
}

The factorization of the likelihood function of the adjusted \code{PIN} model
\insertCite{Duarte09}{PINstimation}, is derived, and presented in
\insertCite{Ersan2022b;textual}{PINstimation}.
\itemize{
\item \code{fact_adjpin()}: factorization in
\insertCite{Ersan2022b;textual}{PINstimation}
}
}
\details{
The argument 'data' should be a numeric dataframe, and contain
at least two variables. Only the first two variables will be considered:
The first variable is assumed to correspond to the total number of
buyer-initiated trades, while the second variable is assumed to
correspond to the total number of seller-initiated trades. Each row or
observation correspond to a trading day. \code{NA} values will be ignored.

Our tests, in line with \insertCite{WilliamLin2011;textual}{PINstimation},
and \insertCite{ErsanAlici2016;textual}{PINstimation}, demonstrate very
similar results for \code{fact_pin_lk()}, and \code{fact_pin_e()}, both
having substantially better estimates than \code{fact_pin_eho()}.
}
\examples{
# There is a preloaded quarterly dataset called 'dailytrades' with 60
# observations. Each observation corresponds to a day and contains the
# total number of buyer-initiated trades ('B') and seller-initiated
# trades ('S') on that day. To know more, type ?dailytrades

xdata <- dailytrades

# ------------------------------------------------------------------------ #
# Using fact_pin_eho(), fact_pin_lk(), fact_pin_e() to find the likelihood #
# value as factorized by Easley(2010), Lin & Ke (2011), and Ersan(2016).   #
# ------------------------------------------------------------------------ #

# Choose a given parameter set to evaluate the likelihood function at a
# givenpoint  = (alpha, delta, mu, eps.b, eps.s)

givenpoint <- c(0.4, 0.1, 800, 300, 200)

# Use the ouput of fact_pin_e() with the optimization function optim() to
# find optimal estimates of the PIN model.

model <- suppressWarnings(optim(givenpoint, fact_pin_e(xdata)))

# Collect the model estimates from the variable model and display them.

varnames <- c("alpha", "delta", "mu", "eps.b", "eps.s")
estimates <- setNames(model$par, varnames)
show(estimates)

# Find the value of the log-likelihood function at givenpoint

lklValue <- fact_pin_lk(xdata, givenpoint)

show(lklValue)

# ------------------------------------------------------------------------ #
# Using fact_mpin() to find the value of the MPIN likelihood function as   #
# factorized by Ersan (2016).                                              #
# ------------------------------------------------------------------------ #

# Choose a given parameter set to evaluate the likelihood function at a
# givenpoint  = (alpha(), delta(), mu(), eps.b, eps.s) where alpha(), delta()
# and mu() are vectors of size 2.

givenpoint <- c(0.4, 0.5, 0.1, 0.6, 600, 1000, 300, 200)

# Use the output of fact_mpin() with the optimization function optim() to
# find optimal estimates of the PIN model.

model <- suppressWarnings(optim(givenpoint, fact_mpin(xdata)))

# Collect the model estimates from the variable model and display them.

varnames <- c(paste("alpha", 1:2, sep = ""), paste("delta", 1:2, sep = ""),
              paste("mu", 1:2, sep = ""), "eb", "es")
estimates <- setNames(model$par, varnames)
show(estimates)

# Find the value of the MPIN likelihood function at givenpoint

lklValue <- fact_mpin(xdata, givenpoint)

show(lklValue)

# ------------------------------------------------------------------------ #
# Using fact_adjpin() to find the value of the DY likelihood function as   #
# factorized by Ersan and Ghachem (2022b).                                 #
# ------------------------------------------------------------------------ #

# Choose a given parameter set to evaluate the likelihood function
# at a the initial parameter set givenpoint = (alpha, delta,
# theta, theta',eps.b, eps.s, muB, muS, db, ds)

givenpoint <- c(0.4, 0.1, 0.3, 0.7, 500, 600, 800, 1000, 300, 200)

# Use the output of fact_adjpin() with the optimization function
# neldermead() to find optimal estimates of the AdjPIN model.

low <- c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0)
up <- c(1, 1, 1, 1, Inf, Inf, Inf, Inf, Inf, Inf)
model <- nloptr::neldermead(
givenpoint, fact_adjpin(xdata), lower = low, upper = up)

# Collect the model estimates from the variable model and display them.

varnames <- c("alpha", "delta", "theta", "thetap", "eps.b", "eps.s",
              "muB", "muS", "db", "ds")
estimates <- setNames(model$par, varnames)
show(estimates)

# Find the value of the log-likelihood function at givenpoint

adjlklValue <- fact_adjpin(xdata, givenpoint)
show(adjlklValue)
}
\references{
\insertAllCited
}
