# Invisible Auxiliaries ---------------------------------------------------
# 01. graph_transform
#   - support for 'igraph' and 'network' + checking
#   - stop if there is any NA or Inf values conditional on flag
# 02. list_transform
#   - transform the arbitrary data list using 'graph_transform'
#   - check argument : square matrix
# 03. laplacian_unnormalized
# 04. laplacian_normalized
# 05. laplacian_signless




# 01. graph_transform -----------------------------------------------------
# sparse Matrix arguments are checked via inherits(obj, "Matrix")
#' @keywords internal
#' @noRd
graph_transform <- function(obj,NIs="allowed"){
  # 1. package:: igraph
  if (class(obj)=="igraph"){
    output = as_adjacency_matrix(obj)
  # 2. package:: network
  } else if (class(obj)=="network"){
    output = Matrix(as.matrix.network(obj, matrix.type = "adjacency"), sparse=TRUE)
  # 3. simple matrix
  } else {
    output = Matrix(obj, sparse=TRUE)
  }
  if (NIs!="allowed"){
    if ((any(is.na(output)))||(any(is.infinite(output)))){
      stop("* NetworkDistance : inputs of NA, Inf, or -Inf are not allowed.")
    }
  }
  return(output)
}


# 02. list_transform ------------------------------------------------------
#' @keywords internal
#' @noRd
list_transform <- function(A, NIflag="allowed"){
  # 1. size checker
  n = nrow(A[[1]])
  if (ncol(A[[1]])!=n){
    stop("* NetworkDistance : an input list should contain all square matrices.")
  }
  # 2. transform
  listA = list()
  for (i in 1:length(A)){
    tgt = graph_transform(A[[i]], NIs=NIflag)
    if ((nrow(tgt)!=n)||(ncol(tgt)!=n)){
      stop(paste("* NetworkDistance : ",i,"-st/rd/th matrix in the list has non-matching size.",sep=""))
    }
    listA[[i]] = tgt
  }
  return(listA)
}



# 03. laplacian_unnormalized ----------------------------------------------
#' @keywords internal
#' @noRd
laplacian_unnormalized <- function(matA){
  matD = diag(rowSums(matA))-matA
  return(matD)
}
# 04. laplacian_normalized ------------------------------------------------
#' @keywords internal
#' @noRd
laplacian_normalized <- function(matA){
  dd = colSums(matA)
  Dinv2 = diag(1/sqrt(dd))
  Dinv2[which(is.infinite(Dinv2))]=0

  D     = diag(dd)
  output = Dinv2%*%(D-matA)%*%Dinv2
  return(output)
}
# 05. laplacian_signless --------------------------------------------------
#' @keywords internal
#' @noRd
laplacian_signless <- function(matA){
  matD = diag(rowSums(matA))+matA
  return(matD)
}

