runMarkovChainRegenS <-
function(genNextState, genNextStateArgList=NULL, 
                       initialPoint, 
                       regenProbs, regenProbsArgList=NULL,
                       transDens, transDensArgList=NULL,
                       smallMeasure, smallMeasureArgList=NULL,
                       smallFn, smallFnArgList=NULL, numericParams,
                       nsim, nTours) {
  # Check that exactly one of nsim or nTours is supplied
  nsimMissing <- missing(nsim)
  nToursMissing <- missing(nTours)
  if(nsimMissing & nToursMissing)
    stop("Either nsim or nTours has to be supplied.\n") else
  if(!nsimMissing & !nToursMissing)
    stop("Either nsim or nTours has to be supplied, but not both.\n")

  # Check that either regenProbs, or all of smallFn, smallMeasure and 
  # transDens are supplied.
  regenProbsMissing <- missing(regenProbs)
  if(regenProbsMissing){
    transDensMissing <- missing(transDens)
    smallFnMissing <- missing(smallFn)
    smallMeasureMissing <- missing(smallMeasure)
    if(transDensMissing || smallFnMissing || smallMeasureMissing)
      stop("Either regenProbs(), or ALL of transDens(), smallFn() and 
            smallMeasure() have to be supplied.\n")
  }

  # check that the number of numeric parameters is known
  if(missing(numericParams))
    numericParams <- 1:length(initialPoint)

  # Check the supplied functions and arguments.
# CHECKS ON genNextState AND ITS ARGUMENTS. ######################
  # check that the genNextState argument is a function.
  if(class(genNextState)!="function")
    stop("genNextState() has to be of 'function' class.\n")

  # check that genNextStateArgList is not missing and is a list.
  if(!is.null(genNextStateArgList)){
    if(class(genNextStateArgList)!="list")
      stop("genNextStateArgList has to be of 'list' class.\n")

    # After this, we should only have to call genNextState(x=...)
    formals(genNextState) <- c(genNextStateArgList, x=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if one-step generation function works .. ")
  firstPointGenerated <- genNextState(x=initialPoint)
  if(sum(is.na(firstPointGenerated)) > 0)
    stop("NA's generated in genNextState().\n")

  # check length of output argument
  if(length(firstPointGenerated)!=length(initialPoint))
    stop("Input/output states for genNextState() do not match in length.\n")
# END OF CHECKS ON genNextState AND ITS ARGUMENTS. ###############
  cat("genNextState() OK.\n")

  if(regenProbsMissing) {
# CHECKS ON transDens AND ITS ARGUMENTS. ######################
  # check that the genNextState argument is a function.
  if(class(transDens)!="function")
    stop("transDens() has to be of 'function' class.\n")

  # check that transDensArgList is not missing and is a list.
  if(!is.null(transDensArgList)){
    if(class(transDensArgList)!="list")
      stop("transDensArgList has to be of 'list' class.\n")

    # After this, we should only have to call genNextState(x=...)
    formals(transDens) <- c(transDensArgList, x=1, y=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if transition density function works .. ")
  pxy <- transDens(x=initialPoint, y=firstPointGenerated)
  if(is.na(pxy))
    stop("NA's generated by transition density function.\n")

  # check length of output argument
  if(length(pxy)!=1)
    stop("Transition density function should return a real scalar.\n")
# END OF CHECKS ON transDens AND ITS ARGUMENTS. ###############
  cat("transDens() OK.\n")

# CHECKS ON smallMeasure AND ITS ARGUMENTS. ######################
  # check that the smallMeasure argument is a function.
  if(class(smallMeasure)!="function")
    stop("smallMeasure() has to be of 'function' class.\n")

  # check that smallMeasureArgList is not missing and is a list.
  if(!is.null(smallMeasureArgList)){
    if(class(smallMeasureArgList)!="list")
      stop("smallMeasureArgList has to be of 'list' class.\n")

    # After this, we should only have to call genNextState(x=...)
    formals(smallMeasure) <- c(smallMeasureArgList, y=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if small measure function works .. ")
  nuY <- smallMeasure(y=firstPointGenerated)
  if(is.na(nuY))
    stop("NA's generated by small measure function.\n")

  # check length of output argument
  if(length(nuY)!=1)
    stop("Small measure function should return a real scalar.\n")
# END OF CHECKS ON smallMeasure AND ITS ARGUMENTS. ###############
  cat("smallMeasure() OK.\n")

# CHECKS ON smallFn AND ITS ARGUMENTS. ######################
  # check that the smallFn argument is a function.
  if(class(smallFn)!="function")
    stop("smallFn() has to be of 'function' class.\n")

  # check that smallFnArgList is not missing and is a list.
  if(!is.null(smallFnArgList)){
    if(class(smallFnArgList)!="list")
      stop("smallFnArgList has to be of 'list' class.\n")

    # After this, we should only have to call genNextState(x=...)
    formals(smallFn) <- c(smallFnArgList, x=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if small function works .. ")
  sX <- smallFn(x=firstPointGenerated)
  if(is.na(sX))
    cat("NA's generated by small function.\n")

  # check length of output argument
  if(length(sX)!=1)
    stop("Small function should return a real scalar.\n")
# END OF CHECKS ON smallMeasure AND ITS ARGUMENTS. ###############
  cat("smallFn() OK.\n")
  }
  else {
# CHECKS ON regenProbs AND ITS ARGUMENTS. ######################
  # check that the regenProbs argument is a function.
  if(class(regenProbs)!="function")
    stop("regenProbs() has to be of 'function' class.\n")

  # When regenProbsArgList is not NULL, check that it is a list.
  if(!is.null(regenProbsArgList)){
    if(class(regenProbsArgList)!="list")
      stop("regenProbsArgList has to be of 'list' class.\n")

    # After this, we should only have to call genNextState(x=...)
    formals(regenProbs) <- c(regenProbsArgList, x=1, y=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if regenProbs function works .. ")
  firstRegenProb <- regenProbs(x=initialPoint, y=firstPointGenerated)
  if(is.na(firstRegenProb))
    cat("NA's generated by regenProbs function with x=", initialPoint, 
        "and y=", firstPointGenerated, ".\n", sep=" ")

  # check length of output argument
  if(length(firstRegenProb)!=1)
    stop("regenProbs function should return a real scalar.\n")
# END OF CHECKS ON regenProbs AND ITS ARGUMENTS. ###############
  cat("regenProbs() OK.\n")
  }
  cat("All checks done.\n\n")

############# genOneTour function definition ##################
genOneTour <- function(startPt, numericParams) {
  steps <- 1
  Si <- startPt
  complete <- 0

   repeat{
      potentialNewY <- genNextState(x=startPt)
      if(regenProbsMissing) {
        regenProb <- smallFn(x=startPt) *
                     smallMeasure(y=potentialNewY) /
                     transDens(x=startPt, y=potentialNewY)
      }
      else {
        regenProb <- regenProbs(x=startPt, y=potentialNewY)
      }
      complete <- ifelse(runif(1) < regenProb, 1, 0)
      if (complete==1)
        break
      else {
        Si <- Si + potentialNewY
        startPt <- potentialNewY
        steps <- steps + 1
      }
    }

  return(list(out=c(Si[numericParams], steps), newStart=potentialNewY))
}
###############################################################
# if nTours is missing, run the chain for nsim iterations and split that up.
  if(nToursMissing) {
    # Run the Markov chain for nsim steps
    cat("Running Markov chain for", nsim, "iterations .. ", sep=" ")
    outMatrix <- matrix(0, nrow=nsim, ncol=length(initialPoint))
    regenProbsVec <- rep(0, times=nsim)
    bellVars <- rep(0, times=nsim)

    outMatrix[1,] <- firstPointGenerated
    for(i in 2:nsim) {
      outMatrix[i,] <- genNextState(x=outMatrix[i-1,])
      if(regenProbsMissing) {
        regenProbsVec[i-1] <- smallFn(x=outMatrix[i-1,]) *
                              smallMeasure(y=outMatrix[i,]) /
                              transDens(x=outMatrix[i-1,], y=outMatrix[i,])
      }
      else {
        regenProbsVec[i-1] <- regenProbs(x=outMatrix[i-1,], y=outMatrix[i,])
      }
    }

    # generate regeneration points
    bellVars[-nsim] <- ifelse(runif(nsim-1) < regenProbsVec[-nsim], 1, 0)
    cat("Done.\n")

    if(sum(bellVars) >= 2) {
      tourMatrix <- splitIntoTours(outMatrix, bellVars, numericParams)
      return(tourMatrix)
    }
    else {
      cat("No complete tours found. Returning everything.\n")
      return(list(outMatrix, regenProbsVec))
    }
  }
  else { # Run for a specified number of tours
    # Look for first regeneration point
    cat("Looking for first regeneration point .. ")
    findFirstPt <- 0
    while(findFirstPt == 0) {
      potentialNewY <- genNextState(x=initialPoint)
      if(regenProbsMissing) {
        regenProb <- smallFn(x=initialPoint) *
                     smallMeasure(y=potentialNewY) /
                     transDens(x=initialPoint, y=potentialNewY)
      }
      else {
        regenProb <- regenProbs(x=initialPoint, y=potentialNewY)
      }
      complete <- ifelse(runif(1) < regenProb, 1, 0)
      if(complete == 1) {
        findFirstPt <- 1
        newStart <- potentialNewY
      }
      else {
        initialPoint <- potentialNewY
      }
    }
    cat("Done.\n")

    # Run the Markov chain for nTours tours
    cat("Generating", nTours, "tours .. ", sep=" ")
    tourMatrix <- matrix(0, nrow=nTours, ncol=length(numericParams)+1)
    for(i in 1:nTours) {
      tmp <- genOneTour(newStart, numericParams)
      tourMatrix[i,] <- tmp$out
      newStart <- tmp$newStart
    }
    cat("Done.\n")
    colnames(tourMatrix) <- c(paste("S", numericParams, sep=""), "N")

    return(tourMatrix)

  }

}

