#' Calculate standardized proximal treatment effect across time (continuous outcomes)
#'
#' Estimates the time-varying standardized proximal causal excursion effect for
#' **continuous** proximal outcomes in a micro-randomized trial. The estimator
#' uses inverse-probability weighting and can adjust for baseline and
#' time-varying covariates to improve efficiency. Optionally, the effect and
#' scale estimates are smoothed over decision points using LOESS, and
#' participant-level bootstrap confidence intervals can be computed.
#'
#' @param data A data.frame of MRT data (see `data_example_for_standardized_effect`)
#' @param id Column name for participant id
#' @param outcome Column name for the continuous proximal outcome
#' @param treatment Column name for treatment indicator
#' @param time Column name for time / decision point
#' @param rand_prob Column name for randomization probability
#' @param availability Column name for availability indicator
#' @param covariates Optional character vector of covariate column names
#' @param smooth Logical; apply LOESS smoothing across time
#' @param loess_span Numeric; smoother span
#' @param loess_degree Numeric; polynomial degree in LOESS
#' @param do_bootstrap Logical; whether to perform bootstrap over participants
#' @param boot_replications Integer; number of bootstrap replications
#' @param confidence_alpha Numeric; two-sided alpha level for CIs
#'
#' @return A data.frame of class \code{"mrt_effect_size"} containing the
#'   standardized effect for a continuous proximal outcome with columns:
#'   \describe{
#'     \item{time}{Decision point index.}
#'     \item{beta_hat}{Raw (unsmoothed) estimated excursion effect at each time.}
#'     \item{s_hat}{Raw (unsmoothed) estimated outcome scale at each time.}
#'     \item{beta_sm}{Smoothed excursion effect across time (equals \code{beta_hat}
#'       if \code{smooth = FALSE}).}
#'     \item{s_sm}{Smoothed outcome scale across time (equals \code{s_hat}
#'       if \code{smooth = FALSE}).}
#'     \item{estimate}{Standardized effect \code{beta_sm / s_sm}.}
#'     \item{lower}{Lower confidence bound for \code{estimate} (NA if
#'       \code{do_bootstrap = FALSE}).}
#'     \item{upper}{Upper confidence bound for \code{estimate} (NA if
#'       \code{do_bootstrap = FALSE}).}
#'   }
#'
#' @references
#' Luers, B., Klasnja, P., and Murphy, S. (2019).
#' Standardized effect sizes for preventive mobile health interventions in
#' micro-randomized trials. *Prevention Science*, 20(1), 100–109.
#'
#' @export
#' @importFrom stats loess loess.control predict quantile rnorm rbinom plogis
#' @importFrom boot boot
#' @examples
#' data("data_example_for_standardized_effect")
#'
#' ans_ci <- calculate_mrt_effect_size(
#'   data         = data_example_for_standardized_effect,
#'   id           = "id",
#'   outcome      = "outcome",
#'   treatment    = "treatment",
#'   time         = "decision_point",
#'   rand_prob    = "prob_treatment",
#'   availability = "availability",
#'   covariates   = c("covariate1", "covariate2"),
#'   do_bootstrap = TRUE,
#'   boot_replications = 100
#' )
#'
#' # Note: use at least 1000 bootstrap replications for stable CIs.
#'
#' summary(ans_ci)
#' plot(ans_ci)
calculate_mrt_effect_size <- function(
    data, id, outcome, treatment, time,
    rand_prob, availability,
    covariates = NULL,
    smooth = TRUE,
    loess_span = 0.25,
    loess_degree = 1,
    do_bootstrap = TRUE,
    boot_replications = 1000,
    confidence_alpha = 0.05
) {
  call <- match.call()
  ans <- estimate_mrt_std_effect(
    data = data, id = id, outcome = outcome, treatment = treatment, time = time,
    rand_prob = rand_prob, availability = availability,
    covariates = covariates,
    smooth = smooth, loess_span = loess_span, loess_degree = loess_degree,
    do_bootstrap = do_bootstrap, boot_replications = boot_replications,
    confidence_alpha = confidence_alpha
  )

  out <- ans
  names(out)[names(out) == "std_estimate"] <- "estimate"
  names(out)[names(out) == "std_lower"]    <- "lower"
  names(out)[names(out) == "std_upper"]    <- "upper"
  attr(out, "call") <- call
  attr(out, "n_id") <- length(unique(data[[id]]))
  attr(out, "smooth") <- smooth
  attr(out, "loess_span") <- loess_span
  attr(out, "loess_degree") <- loess_degree
  attr(out, "do_bootstrap") <- do_bootstrap
  attr(out, "boot_replications") <- boot_replications
  attr(out, "confidence_alpha") <- confidence_alpha
  class(out) <- c("mrt_effect_size", class(out))
  out
}
