#' Lioness Algorithm for order-of-addition design
#'
#' \code{LA_OofA} returns a \code{n} by \code{k} order-of-addition design matrix generated by lioness algorithm (LA)
#'
#' @param n A positive integer, which stands for the number of rows (or run size). The default setting of \code{n} is \code{k} factorial, which yields a full order-of-addition design matrix. Note that the maximum of \code{n} cannot be greater than \code{k} factorial.
#' @param k A positive integer, which stands for the number of columns (or factor size).
#' @param m A positive integer, which stands for the number of starting lionesses agents. The default is set to be 10, and it is recommended to be no greater than 100.
#' @param N A positive integer, which stands for the number of iterations. The default is set to be 10. A large value of \code{N} will result a high CPU time, and it is recommended to be no greater than 500.
#' @param prun A probability, which stands for the probability of "prey runs away", and it is denoted as \code{p*} in the original paper. The default is set to be 1/(\code{n} - 1).
#' @param OC An optimality criterion. The default setting is "D", which stands for the D-optimality criterion, and in the future we would add other criteria such as A-optimality criterion and E-optimality criterion.
#' @param maxtime A positive number, which indicates the expected maximum CPU time given by user, and it is measured by minutes. For example, maxtime=3.5 indicates the CPU time will be no greater than three and half minutes. The default is set to be 5.
#'
#' @return If all inputs are logical, then the output will be a \code{n} by \code{k} order-of-addition design.
#'
#' @examples
#' #generate a D-optimal full OofA with 4 factors.
#' try=LA_OofA(k=4)
#' try
#' det(MOM(try))  #the determinant of the moment matrix of try
#'
#' #Another example
#' #generate a D-optimal 11-run OofA with 4 factors.
#' try2=LA_OofA(n=11,k=4)
#' try2
#' det(MOM(try2)) #the determinant of the moment matrix of try2
#' @export

LA_OofA=function(n=factorial(k),k,m=10,N=10,prun=1/(n-1),OC="D",maxtime=5){

  maxtime=maxtime*60  #convert minutes to seconds
  timeALL=NULL        #record all cpu time

  C=1  #Initialize counter index

  #step 1 starts
  X=rep(0,n*k*m)    #solution matrix

  dim(X)=c(n,k,m)

  for (i in 1:m) {
    X[,,i]=rOofA(n=n,k=k)
  }
  #step 1 ends

  if(OC=="D"){

    #step 2 starts
    result=rep(0,m)

    for (i in 1:m) {
      result[i]=det(MOM(X[,,i]))
    }

    #step 2 ends

    progressbar = utils::txtProgressBar(min = 0, max = N, style = 3)

    while (C<=N) {  #step 3

      time0=Sys.time()

      temp=cbind(result,1:m)

      temp=temp[order(temp[,1],decreasing=TRUE),]

      #step 4: determine the top 3 agents
      centre=X[,,temp[1,2]]

      LW=X[,,temp[2,2]]

      RW=X[,,temp[3,2]]

      #step 4 ends

      m=6*n+3

      Xnew=rep(0,n*k*m)       #new position matrix

      dim(Xnew)=c(n,k,m)

      Xnew[,,1]=centre

      Xnew[,,2]=LW

      Xnew[,,3]=RW

      #step 5 starts
      #centre troop
      index=4

      for (i in 1:n) {

        Xnew[,,index]=centre
        Xnew[i,,index]=LW[i,]
        index=index+1

        Xnew[,,index]=centre
        Xnew[i,,index]=RW[i,]
        index=index+1

      }

      #LW troop

      for (i in 1:n) {

        Xnew[,,index]=LW
        Xnew[i,,index]=centre[i,]
        index=index+1

        Xnew[,,index]=LW
        Xnew[i,,index]=RW[i,]
        index=index+1

      }

      #RW troop

      for (i in 1:n) {

        Xnew[,,index]=RW
        Xnew[i,,index]=centre[i,]
        index=index+1

        Xnew[,,index]=RW
        Xnew[i,,index]=LW[i,]
        index=index+1

      }

      #step 5 ends
      #Until here, Xnew has been fully updated

      X=Xnew

      for (i in 2:m) {      #step 6
        for (j in 1:n) {    #step 7

          z=stats::runif(1,0,1)             #step 8

          if (z<=prun){

            X[,,i]=LHD::exchange(X=X[,,i],j=j,type="row")   #step 9


          } #step 10: end if

        }   #step 11: end for
      }     #step 12: end for

      #step 13: update det(M) for all
      result=rep(0,m)

      for (i in 1:m) {
        result[i]=det(MOM(X[,,i]))
      }

      time1=Sys.time()
      timediff=time1-time0
      timeALL=c(timeALL,timediff)

      ##########progress bar codes
      utils::setTxtProgressBar(progressbar, C)
      ##########

      if(as.numeric(sum(timeALL)+timediff)<=maxtime){C=C+1}
      if(as.numeric(sum(timeALL)+timediff)>maxtime){C=N+1}
    }


    temp=cbind(result,1:m)

    temp=temp[order(temp[,1],decreasing=TRUE),]

    centre=X[,,temp[1,2]]

  }

  avgtime=round(mean(timeALL),2)
  iterations=length(timeALL)

  close(progressbar)
  print(paste0("average CPU time per iteration is: ", avgtime, " seconds"))
  print(paste0("the number of iterations completed is: ", iterations))


  #step 15: return centre
  centre
}

