% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jmc.R
\name{jmc}
\alias{jmc}
\title{An integrated function for reconstructing data and do the joint modelling}
\usage{
jmc(
  long_data,
  surv_data,
  out,
  FE,
  RE,
  ID,
  cate = NULL,
  intcpt = 1,
  quad.points = 10,
  max.iter = 10000,
  quiet = TRUE,
  do.trace = FALSE
)
}
\arguments{
\item{long_data}{Data matrix for longitudinal in long form. The time variable should be labeled 'time'.}

\item{surv_data}{Data matrix for competing risks data. Each subject has one row of observation (as opposed to the long_data).
First and second column should be the observed event time and censoring indicator, respectively.
The coding for the censoring indicator is as follows: 0 - censored events, 1 - risk 1, 2 - risk 2. Two competing risks are assumed.}

\item{out}{Column name for outcome variable in long_data.}

\item{FE}{Vector of column names that correspond to the fixed effects in long_data. If missing, then all columns except for the outcome and ID columns will be considered.}

\item{RE}{Types/Vector of random effects in long_data. The available type are "intercept", "linear", "quadratic" (time-related random effect specification) or other covariates in the input dataset. If specify other covariates, then they to be numerical vectors.}

\item{ID}{Column name for subject ID number in long_data.}

\item{cate}{Vector of categorical variables in long_data. Default is NULL.}

\item{intcpt}{Specify either 0 or 1. Default is set as 1. 0 means no intercept in random effect.}

\item{quad.points}{Number of quadrature points used in the EM procedure. Default is 20. Must be an even number. Larger values means higher accuracy but more time-consuming.}

\item{max.iter}{Max iterations. Default is 10000.}

\item{quiet}{Logical. Print progress of function. Default is TRUE.}

\item{do.trace}{Logical. Print the parameter estimates during the iterations. Default is FALSE.}
}
\value{
Object of class \code{JMcmprsk} with elements
  \tabular{ll}{
      \code{vcmatrix}    \tab  The variance-covariance matrix for all the parameters. The parameters are in the order: \eqn{\beta}, \eqn{\sigma^2}, \eqn{\gamma}, \eqn{\nu}, and \eqn{\Sigma}. The elements in \eqn{\Sigma} are output in the order along the main diagonal line, then the second main diagonal line, and so on. \cr
      \code{betas} \tab The point  estimates of \eqn{\beta}. \cr
      \code{se_betas} \tab The standard error estimate of \eqn{\beta}. \cr
      \code{gamma_matrix} \tab  The point  estimate of \eqn{\gamma}. \cr
      \code{se_gamma_matrix}   \tab  The standard error estimate of \eqn{\gamma}. \cr
      \code{v_estimate} \tab The point  estimate of \eqn{\nu}. \cr
      \code{se_v_estimate}    \tab The standard error estimate of \eqn{\nu}. \cr
      \code{sigma2_val}     \tab  The point estimate of \eqn{\sigma^2}.\cr
      \code{se_sigma2_val}     \tab  The standard error estimate of \eqn{\sigma^2}.\cr
      \code{sigma_matrix}     \tab The point estimate of \eqn{\Sigma} (only the upper triangle portion of the matrix is output).\cr
      \code{se_sigma}     \tab The standard error estimate of \eqn{\Sigma}.The standard errors are given in this order: main diagonal, the second main diagonal, and so on. \cr
      \code{loglike}     \tab Log Likelihood.\cr
  }
}
\description{
Reconstruct data into a regular longitudinal format as a refined dataset and do joint modelling for this refined data with continuous outcome.
}
\examples{

require(JMcmprsk)
set.seed(123)
data(lung)
yread <- lung[, c(1,2:11)]
cread <- unique(lung[, c(1, 12, 13, 6:10)])
\donttest{
#Please note only those variables that will appear in the model can be included
res <- jmc(long_data = yread, surv_data = cread, out = "FVC",
           FE = c("time", "FVC0", "FIB0", "CYC", "FVC0.CYC",
                  "FIB0.CYC", "time.CYC"),
           RE = "linear", ID = "ID",cate = NULL, intcpt = 0,
           quad.points = 8, quiet = FALSE)
}

#make up two categorical variables and add them into yread
sex <- sample(c("Female", "Male"), nrow(cread), replace = TRUE)
race <- sample(c("White", "Black", "Asian", "Hispanic"),
               nrow(cread), replace = TRUE)
ID <- cread$ID
cate_var <- data.frame(ID, sex, race)
if (require(dplyr)) {
  yread <- dplyr::left_join(yread, cate_var, by = "ID")
}
\donttest{
# run jmc function again for yread file with two added categorical variables
res2 <- jmc(long_data = yread, surv_data = cread,
            out = "FVC", cate = c("sex", "race"),
            FE = c("time", "FVC0", "FIB0", "CYC", "FVC0.CYC",
                "FIB0.CYC", "time.CYC"),
            RE = "time", ID = "ID", intcpt = 0,
            quad.points = 8, quiet = FALSE)
res2
# Extract the parameter estimates of longitudinal sub-model fixed effects
 beta <- coef(res2, coeff = "beta")
 beta
## Linear hypothesis of testing all coefficients of beta's equal 0
 linearTest(res2, coeff="beta")
 ## Linear hypothesis of testing beta1=beta2
 ## create a linear contrast for beta1=beta2 (intercept not included in Lb)
 Lb <- matrix(c(1, -1, 0, 0, 0, 0, 0, 0, 0, 0, 0), ncol = length(beta)-1, nrow = 1)
 linearTest(res2, coeff="beta", Lb = Lb)
 # Extract the parameter estimates of survival sub-model fixed effects
 gamma <- coef(res2, coeff = "gamma")
 gamma
 ## Linear hypothesis of testing all coefficients of gamma's equal 0
 linearTest(res2, coeff="gamma")
 ## Linear hypothesis of testing gamma11=gamma21
 ##  (the coefficients of first covariate from
 ##    both risk functions are equal)
 Lg <- matrix(c(1, 0, 0, 0, 0, -1, 0, 0, 0, 0), ncol = length(gamma), nrow = 1)
 linearTest(res2, coeff="gamma", Lg = Lg)
 ## Extract the standard errors for the longitudinal portion
 summary(res2, coeff = "longitudinal", digits = 4)
 ## Extract the standard errors for the survival portion
 summary(res2, coeff = "survival", digits = 4)
 }
}
\seealso{
\code{\link{jmc_0}}
}
