#' Facet Chart
#'
#' Creates a facet chart, showing the facets of a test.
#'
#' @param data SEM estimates in the appropriate format, given by the input
#'   functions.
#' @param subradius integer; same unit as center distances; radius of the facet
#'   circles; defaults to 0, in which case an appropriate value is estimated.
#' @param file_name character; name of the file to save. Supported formats are:
#'   "pdf" (highest quality and smallest file size), "png", "jpeg"; defaults to "none".
#' @param size integer; changes the size of most chart objects simultaneously.
#' @param font character; text font, use extrafonts to access additional fonts;
#'   defaults to "sans", which is "Helvetica".
#' @param rotate_radians integer; radian angle to rotate the chart
#'   counter-clockwise by; use fractions of pi (e.g. pi/2 = 90 degrees).
#' @param rotate_degrees integer; angle in degrees to rotate the chart
#'   counter-clockwise by.
#' @param file_width integer; file width in inches; defaults to 10.
#' @param file_height integer; file height in inches; defaults to 10.
#' @param dpi integer; resolution in dots per inch for "png" and "jpeg" files;
#'   defaults to 500.
#' @param color accent color; defaults to "black".
#' @param fade integer; brightness of the gray tones between 0 = "black" and 100
#'   = "white" in steps of 1; defaults to 85.
#' @param tick numeric; axis tick position; defaults to 0, in which case an
#'   appropriate value is estimated.
#' @param cor_labels logical; if \code{TRUE}, shows latent correlations between
#'   facets; defaults to \code{TRUE}.
#' @param dist_test_label integer; position of the test label relative to the
#'   surrounding circle; defaults to 2/3, in which case the test label is
#'   displayed 2/3 of the way from the center to the surrounding circle.
#' @param rotate_test_label_radians integer; radian angle to rotate the test
#'   label counter-clockwise by; use fractions of pi (e.g. pi/2 = 90 degrees).
#' @param rotate_test_label_degrees integer; angle in degrees to rotate the
#'   global label counter-clockwise by.
#' @param size_test_label integer; test font size relative to default.
#' @param size_facet_labels integer; facet font size relative to default.
#' @param width_axes integer; radial axis width relative to default.
#' @param width_circles integer; facet circle outline width relative to default.
#' @param width_tick integer; axis tick line width relative to default.
#' @param size_tick_label integer; axis tick font size relative to default.
#' @param size_cor_labels integer; correlation font size relative to default.
#'
#' @details Pdf files will be vector based and can be scaled arbitrarily. For
#'   other formats use \code{file_width}, \code{file_height}, and \code{dpi} to
#'   avoid later rescaling and loss of quality.
#'
#' @return Object of the class "ggplot".
#'
#' @seealso \code{\link{item_chart}} \code{\link{nested_chart}}
#'
#' @examples
#' # as simple as that:
#' facet_chart(SMTQ)
#'
#' @export
facet_chart <- function(
  data,
  subradius = 0,
  file_name = "none",
  size = 1,
  font = "sans",
  rotate_radians = 0,
  rotate_degrees = 0,
  file_width = 10,
  file_height = 10,
  dpi = 500,
  color = "black",
  fade = 85,
  tick = 0,
  cor_labels = TRUE,
  dist_test_label = 2 / 3,
  rotate_test_label_radians = 0,
  rotate_test_label_degrees = 0,
  size_cor_labels = 1,
  size_test_label = 1,
  size_facet_labels = 1,
  width_axes = 1,
  width_circles = 1,
  width_tick = 1,
  size_tick_label = 1){

  coord <- coord_facets(
    data = data,
    subradius = subradius,
    tick = tick,
    rotate_radians = rotate_radians,
    rotate_degrees = rotate_degrees,
    dist_test_label = dist_test_label,
    rotate_test_label_radians =rotate_test_label_radians,
    rotate_test_label_degrees = rotate_test_label_degrees)

  myipv <- plot_facets(
    coord = coord,
    size = size,
    file_name = file_name,
    file_width = file_width,
    file_height = file_height,
    dpi = dpi,
    color = color,
    fade = fade,
    font = font,
    cor_labels = cor_labels,
    size_cor_labels = size_cor_labels,
    size_test_label = size_test_label,
    size_facet_labels = size_facet_labels,
    width_axes = width_axes,
    width_circles = width_circles,
    width_tick = width_tick,
    size_tick_label = size_tick_label)

  return(myipv)
}



#' Plot Facets
#'
#' Generates a facet chart from coordinates.
#'
#' @param coord list generated by \code{\link{coord_facets}} or
#'   \code{\link{coord_nested}}.
#' @param size integer; changes the size of most chart objects simultaneously.
#' @param file_name character; name of the file to save. Supported formats are:
#'   "pdf" (highest quality and smallest file size), "png", "jpeg"; defaults to "none".
#' @param file_width integer; file width in inches; defaults to 10.
#' @param file_height integer; file height in inches; defaults to 10.
#' @param dpi integer; resolution in dots per inch for "png" and "jpeg" files;
#'   defaults to 500.
#' @param color accent color; defaults to "black".
#' @param fade integer; brightness of the gray tones between 0 = "black" and 100
#'   = "white" in steps of 1; defaults to 85.
#' @param cor_labels logical; if \code{TRUE}, shows latent correlations between
#'   facets; defaults to \code{TRUE}.
#' @param font character; text font, use extrafonts to access additional fonts;
#'   defaults to "sans", which is "Helvetica".
#' @param size_test_label integer; test font size relative to default.
#' @param size_facet_labels integer; facet font size relative to default.
#' @param width_axes integer; radial axis width relative to default.
#' @param width_circles integer; facet circle outline width relative to default.
#' @param width_tick integer; axis tick line width relative to default.
#' @param size_tick_label integer; axis tick font size relative to default.
#' @param size_cor_labels integer; correlation font size relative to default.
#'
#' @details Use \code{\link{facet_chart}} to create facet charts.
#'
#' @return Object of the class "ggplot".
#'
#' @seealso \code{\link{coord_facets}} \code{\link{facet_chart}}
plot_facets <- function(
  coord,
  size = 1,
  file_name = "none",
  file_width = 10,
  file_height = 10,
  dpi = 500,
  color = "black",
  fade = 85,
  font = "sans",
  cor_labels = TRUE,
  size_cor_labels = 1,
  size_test_label = 1,
  size_facet_labels = 1,
  width_axes = 1,
  width_circles = 1,
  width_tick = 1,
  size_tick_label = 1){


  # preparation ----------------------------------------------------------------

  if (cor_labels == TRUE) {
    cors <- coord$cors
  } else cors <- NULL

  # some calculations are not possible within aes_string(), so aesthetics are
  # prepared here
  facet_labels <- row.names(coord$c_circs)

  tick <- signif(sqrt((coord$axis_tick$x ^ 2) + (coord$axis_tick$y ^ 2)), 1)
  message(paste("Axis tick set to ", tick," based on the data.", sep = ""))
  tick_label_label <- as.character(formatC(tick, format = "fg"))
  tick_label_x <- 1.3 * coord$axis_tick$x
  tick_label_y <- 1.3 * coord$axis_tick$y


  # chart ----------------------------------------------------------------------

  myipv <- ggplot2::ggplot(coord$c_circs) +


    ## initializing -----------------

  ggplot2::coord_fixed() +
    ggplot2::theme(
      axis.line        = ggplot2::element_blank(),
      axis.text.x      = ggplot2::element_blank(),
      axis.text.y      = ggplot2::element_blank(),
      axis.ticks       = ggplot2::element_blank(),
      axis.title.x     = ggplot2::element_blank(),
      axis.title.y     = ggplot2::element_blank(),
      legend.position  = "none",
      panel.background = ggplot2::element_blank(),
      panel.border     = ggplot2::element_blank(),
      panel.grid.major = ggplot2::element_blank(),
      panel.grid.minor = ggplot2::element_blank(),
      plot.background  = ggplot2::element_blank(),
      text             = ggplot2::element_text(size = 16, family = font),
      plot.margin      = ggplot2::margin(0, 0, 0, 0, "in")) +
    ggplot2::aes() +


    ## layers -----------------------

  # ordered from bottom to top, for correct overlap

  # center dot
  ggplot2::geom_point(
    ggplot2::aes(x = 0, y = 0),
    size = 3 * size * width_axes) +

    # axis tick
    ggforce::geom_circle(
      ggplot2::aes(x0 = 0, y0 = 0, r = tick),
      linetype = "dotted",
      size = .5 * min(size, 1) * width_tick) +

    # outer axis segments
    ggplot2::geom_segment(
      data = coord$c_axes,
      ggplot2::aes_string(x = "x2", y = "y2", xend = "x3", yend = "y3"),
      size = .5 * size * width_axes,
      color = paste("gray", fade, sep = "")) +

    # tick label
    ggplot2::geom_text(
      data = coord$axis_tick,
      ggplot2::aes(x = tick_label_x,
                   y = tick_label_y,
                   label = tick_label_label),
      angle = (coord$axis_tick$phi - pi / 48 - pi / 2) * 180 / pi,
      family = font,
      size = 4 * size * size_tick_label) +

    # test circle
    ggforce::geom_circle(
      data = coord$c_circs[1, ],
      ggplot2::aes_string(x0 = "x", y0 = "y", r = "radius"),
      size = .5 * size * width_axes,
      color = paste("gray", fade, sep = "")) +

    # facet circles
    ggforce::geom_circle(
      data = coord$c_circs[-1, ],
      ggplot2::aes_string(x0 = "x", y0 = "y", r = "radius"),
      size = .5 * size * width_circles,
      color = color) +

    # facet labels
    ggplot2::geom_text(
      ggplot2::aes_string(x = "x", y = "y", label = "facet_labels"),
      family = font,
      size = 6 * size * size_facet_labels) +

    # inner axis segments
    ggplot2::geom_segment(
      data = coord$c_axes,
      ggplot2::aes_string(x = "x0", y = "y0", xend = "x1", yend = "y1"),
      size = 2 * size * width_axes,
      color = color) +

    # test label
    ggplot2::geom_text(
      data = coord$test_label,
      ggplot2::aes_string(x = "x", y = "y", label = "label"),
      family = font,
      size = 8 * size * size_test_label,
      fontface = "bold",
      color = color)


  ## optional layers ----------------

  # correlations
  if (!is.null(cors)) {
    myipv <- myipv +
      ggplot2::geom_text(
        data = cors,
        ggplot2::aes_string(x = "x", y = "y", label = "label"),
        family = font,
        size = 4 * size * size_cor_labels)
  }


  # optional file save ---------------------------------------------------------

  ## .pdf ---------------------------

  if (substring(file_name, nchar(file_name)-3+1) == "pdf") {
    ggplot2::ggsave(file_name,
                    myipv,
                    width = file_width,
                    height = file_height,
                    units = "in",
                    dpi = dpi)
  }


  ## .png ---------------------------

  if (substring(file_name, nchar(file_name)-3+1) == "png") {
    ggplot2::ggsave(file_name,
                    myipv,
                    width = file_width,
                    height = file_height,
                    units = "in",
                    dpi = dpi)
  }


  ## .jpeg --------------------------

  if (substring(file_name, nchar(file_name)-3+1) == "peg") {
    ggplot2::ggsave(file_name,
                    myipv,
                    width = file_width,
                    height = file_height,
                    units = "in",
                    dpi = dpi)
  }


  # return ---------------------------------------------------------------------

  return(myipv)
}
