% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MINDEX_SP.R
\name{MINDEX_SP}
\alias{MINDEX_SP}
\title{The Multipoint Morisita Index for Spatial Patterns}
\usage{
MINDEX_SP(X, scaleQ=1:5, mMin=2, mMax=5, Wlim_x=NULL, Wlim_y=NULL)
}
\arguments{
\item{X}{A \eqn{N \times 2}{N x 2} \code{matrix}, \code{data.frame} or \code{data.table} containing the \eqn{X} and \eqn{Y}
coordinates of \eqn{N} data points. The \eqn{X} coordinates must be given in the first column
and the \eqn{Y} coordinates in the second column.}

\item{scaleQ}{Either a single value or a vector. It contains the value(s) of \eqn{Q^{(1/2)}}{Q^(1/2)}
chosen by the user where \eqn{Q} is the number of cells (or quadrats) of
the \eqn{2D} grid (by default: \code{scaleQ = 1:5}).}

\item{mMin}{The minimum value of \eqn{m} (by default: \code{mMin = 2}).}

\item{mMax}{The maximum value of \eqn{m} (by default: \code{mMax = 5}).}

\item{Wlim_x}{A vector controlling the spatial extent of the \eqn{2D} gird along the
\eqn{X} axis. It consists of two real values, i.e. \code{Wlim_x <- c(a,b)} where
\eqn{b > a} (by default: \code{Wlim_x <- c(min(X[,1]),max(X[,1]))}).}

\item{Wlim_y}{A vector controlling the spatial extent of the \eqn{2D} gird along the
\eqn{Y} axis. It consists of two real values, i.e. \code{Wlim_y <- c(a,b)} where
\eqn{b > a} (by default: \code{Wlim_y <- c(min(X[,2]),max(X[,2]))}).}
}
\value{
A \code{data.frame} containing the value of the m-Morisita index for each value of
\eqn{\delta}{delta} and \eqn{m}.
}
\description{
Computes the multipoint Morisita index for spatial patterns (i.e. 2-dimensional patterns).
}
\details{
\enumerate{
\item \eqn{Q^{(1/2)}}{Q^(1/2)} is the number of grid cells (or quadrats) along each of the two axes.
\item \eqn{Q^{(1/2)}}{Q^(1/2)} is directly related to \eqn{\delta}{delta} (see References).
\item \eqn{\delta}{delta} is the diagonal length of the grid cells.
}
}
\examples{
sim_dat <- SwissRoll(1000)

m <- 2
scaleQ <- 1:15 # It starts with a grid of 1^2 cell (or quadrat).
               # It ends with a grid of 15^2 cells (or quadrats).
mMI <- MINDEX_SP(sim_dat[,c(1,2)], scaleQ, m, 5)

plot(mMI[,1],mMI[,2],pch=19,col="black",xlab="",ylab="")
title(xlab=expression(delta),cex.lab=1.5,line=2.5)
title(ylab=expression(I['2,'*delta]),cex.lab=1.5,line=2.5)

\dontrun{
require(colorRamps)
colfunc <- colorRampPalette(c("blue","red"))
color <- colfunc(4)
dev.new(width=5,height=4)
plot(mMI[5:15,1],mMI[5:15,2],pch=19,col=color[1],xlab="",ylab="",
     ylim=c(1,max(mMI[,5])))
title(xlab=expression(delta),cex.lab=1.5,line=2.5)
title(ylab=expression(I['2,'*delta]),cex.lab=1.5,line=2.5)
for(i in 3:5){
  points(mMI[5:15,1],mMI[5:15,i],pch=19,col=color[i-1])
}
legend.text<-c("m=2","m=3","m=4","m=5")
legend.pch=c(19,19,19,19)
legend.lwd=c(NA,NA,NA,NA)
legend.col=c(color[1],color[2],color[3],color[4])
legend("topright",legend=legend.text,pch=legend.pch,lwd=legend.lwd,
       col=legend.col,ncol=1,text.col="black",cex=0.9,box.lwd=1,bg="white")

xlim_l <- c(-5,5)     # By default, the spatial extent of the grid is set so
ylim_l <- c(-6,6)     # that it is the same as the spatial extent of the data.
xlim_s <- c(-0.6,0.2) # But it can be modified to cover either a larger (l)
ylim_s <- c(-1,0.5)   # or a smaller (s) study area (or validity domain).

mMI_l <- MINDEX_SP(sim_dat[,c(1,2)], scaleQ, m, 5, xlim_l, ylim_l)
mMI_s <- MINDEX_SP(sim_dat[,c(1,2)], scaleQ, m, 5, xlim_s, ylim_s)
}
}
\references{
J. Golay, M. Kanevski, C. D. Vega Orozco and M. Leuenberger (2014).
The multipoint Morisita index for the analysis of spatial patterns,
\href{http://www.sciencedirect.com/science/article/pii/S0378437114002659}{Physica A 406:191–202}.

L. Telesca, J. Golay and M. Kanevski (2015). Morisita-based space-clustering analysis of Swiss seismicity,
\href{http://www.sciencedirect.com/science/article/pii/S0378437114008425}{Physica A 419:40–47}.

L. Telesca, M. Lovallo, J. Golay and M. Kanevski (2016). Comparing seismicity declustering techniques
by means of the joint use of Allan Factor and Morisita index,
\href{http://link.springer.com/article/10.1007/s00477-015-1030-8}{Stochastic Environmental Research and Risk Assessment 30(1):77-90}.
}
\author{
Jean Golay \email{Jean.Golay@unil.ch}
}
