% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transdistfuncs.r
\name{est.transdist.temporal}
\alias{est.transdist.temporal}
\title{Change in mean transmission distance over time}
\usage{
est.transdist.temporal(
  epi.data,
  gen.t.mean,
  gen.t.sd,
  t1,
  max.sep,
  max.dist,
  n.transtree.reps = 10,
  mean.equals.sd = FALSE,
  theta.weights = NULL,
  parallel = FALSE,
  n.cores = NULL
)
}
\arguments{
\item{epi.data}{a three-column matrix giving the coordinates (\code{x} and \code{y}) and time of infection (\code{t} for all cases in an epidemic (columns must be in \code{x}, \code{y}, \code{t} order)}

\item{gen.t.mean}{mean generation time of the infecting pathogen}

\item{gen.t.sd}{standard deviation of generation time of the infecting pathogen}

\item{t1}{time step to begin estimation of transmission distance}

\item{max.sep}{maximum number of time steps allowed between two cases (passed to the \code{get.transdist.theta} function)}

\item{max.dist}{maximum spatial distance between two cases considered in calculation}

\item{n.transtree.reps}{number of time to simulate transmission trees when estimating the weights of theta (passed to the \code{est.transdist.theta.weights} function, default = 10). Higher values of this parameter cause significant increases in computation time.}

\item{mean.equals.sd}{logical term indicating if the mean and standard deviation of the transmission kernel are expected to be equal (default = FALSE)}

\item{theta.weights}{use external matrix of theta weights. If NULL (default) the matrix of theta weights is automatically estimated by calling the \code{est.transdist.theta.weights} function}

\item{parallel}{run time steps in parallel (default = FALSE)}

\item{n.cores}{number of cores to use when \code{parallel} = TRUE (default = NULL, which uses half the available cores)}
}
\value{
a numeric matrix containing the point estimate for mean transmission distance for each unique time step of the epidemic and the sample size $n$ used to make the estimate 
NAs are returned for time steps which contain fewer than three cases
}
\description{
Estimates the change in mean transmission distance over the duration of the epidemic by running \code{est.trandsdist} on all cases 
occuring up to each time point.
}
\examples{
\donttest{
     
set.seed(123)

# Exponentially distributed transmission kernel with mean and standard deviation = 100
dist.func <- alist(n=1, a=1/100, rexp(n, a)) 

# Simulate epidemic
a <- sim.epidemic(R=2,
                  gen.t.mean=7,
                  gen.t.sd=2,
                  tot.generations=7,
                  min.cases=30,
                  trans.kern.func=dist.func)

a <- a[sample(1:nrow(a), 50),] # subsample a to 50 observations

# Estimate mean transmission kernel over time
b <- est.transdist.temporal(epi.data=a,
                            gen.t.mean=7,
                            gen.t.sd=2,
                            t1=0,
                            max.sep=1e10,
                            max.dist=1e10,
                            n.transtree.reps=5,
                            mean.equals.sd=TRUE,
                            n.cores=2)
b

plot(b[,2], pch=19, col='grey', ylim=c(min(b[,2], na.rm=TRUE), max(b[,2], na.rm=TRUE)), 
     xlab='Time step', ylab='Estimated mean of transmission kernel')
abline(h=100, col='red', lty=2)
axis(3, b[,2])

low <- loess(b[,2] ~ as.vector(1:length(b[,2])))
low <- predict(low, newdata=data.frame(as.vector(1:length(b[,2]))))
lines(low, lwd=3, col='blue')

}
}
\references{
Salje H, Cummings DAT and Lessler J (2016). “Estimating infectious disease transmission distances using the overall distribution of cases.” Epidemics, 17, pp. 10–18. ISSN 1755-4365, doi: \href{https://www.sciencedirect.com/science/article/pii/S1755436516300317}{10.1016/j.epidem.2016.10.001}.
}
\seealso{
Other transdist: 
\code{\link{est.transdist.bootstrap.ci}()},
\code{\link{est.transdist.temporal.bootstrap.ci}()},
\code{\link{est.transdist.theta.weights}()},
\code{\link{est.transdist}()},
\code{\link{get.transdist.theta}()}
}
\author{
John Giles, Justin Lessler, and Henrik Salje
}
\concept{transdist}
