% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/10-UserEfficiency.R
\name{leff}
\alias{leff}
\title{Calculates Relative Efficiency for Locally Optimal Designs}
\usage{
leff(formula, predvars, parvars, family = gaussian(), inipars,
  type = c("D", "PA"), fimfunc = NULL, xopt, wopt, x, w,
  npar = length(inipars), prob = NULL)
}
\arguments{
\item{formula}{A nonlinear model \code{\link[stats]{formula}}.
A symbolic description of the model consists of predictors and the unknown model parameters.
Will be coerced to a \code{\link[stats]{formula}} if necessary.}

\item{predvars}{A vector of characters. Denotes the predictors in the \code{\link[stats]{formula}}.}

\item{parvars}{A vector of characters. Denotes the unknown parameters in the \code{\link[stats]{formula}}.}

\item{family}{A description of the response distribution and link function to be used in the model.
This can be a family function, a call to a family function or a character string naming the family.
 Every family function has a link argument allowing to specify the link function to be applied on the response variable.
  If not specified, default links are used. For details see \code{\link[stats]{family}}.
   By default, a linear gaussian model \code{gaussian()} is applied.}

\item{inipars}{Vector. Initial values for the unknown parameters. It will be passed to the information matrix and also probability function.}

\item{type}{A character. \code{"D"} denotes D-efficiency and \code{"PA"} denotes average P-efficiency.}

\item{fimfunc}{A function. Returns the FIM as a \code{matrix}. Required when \code{formula} is missing. See 'Details' of \code{\link{minimax}}.}

\item{xopt}{Vector of design (support) points of optimal design (\eqn{\xi_2}). Similar to \code{x}.}

\item{wopt}{Vector of corresponding design weights for \code{xopt}.}

\item{x}{Vector of design (support) points of \eqn{\xi_1}. See 'Details' of \code{\link{leff}}.}

\item{w}{Vector of corresponding design weights for \code{x}.}

\item{npar}{Number of model parameters.  Used when \code{fimfunc} is given instead of \code{formula} to specify the number of model parameters.
If not given, the sensitivity plot may be shifted below the y-axis. When \code{NULL}, it will be set here to \code{length(inipars)}.}

\item{prob}{Either \code{formula} or a \code{function}. when function, its argument is \code{x} and \code{param} same as the arguments in \code{fimfunc}.
\code{prob} as a function takes the design points and vector of parameters and returns the probability of success at each design points.
See 'Examples'.}
}
\value{
A value between 0 and 1.
}
\description{
Given a vector of initial estimates for the parameters, this function calculates the D-and PA- efficiency of a design \eqn{\xi_1} with respect to a design \eqn{\xi_2}.
Usually, \eqn{\xi_2} is an  optimal design.
}
\details{
For a known \eqn{\theta_0}, relative D-efficiency is
\deqn{exp(\frac{log|M(\xi, \theta_0)| - log|M(\xi^*, \theta_0)|}{npar})}{
exp\{(log|M(\xi, \theta_0)| - log|M(\xi*, \theta_0)|)/npar\}.}
The relative P-efficiency is
\deqn{\exp(\log(\sum_{i=1}^k w_ip(x_i, \theta_0) - \log(\sum_{i=1}^k w^*_ip(x^*_i, \theta_0))}{
exp(log (\sum w_i p(x_i, \theta_0) - log(\sum w*_i p(x*_i, \theta_0)),
}
where \eqn{x^*}{x*} and \eqn{w^*}{w*} are the support points and the corresponding weights of the optimal design, respectively.

 The argument  \code{x} is the vector of design points.
 For design points with more than one dimension,
   it is a concatenation of the design points, but \strong{dimension-wise}.
   For example, let the model has three predictors   \eqn{(I, S, Z)}.
    Then,  (three-dimensional) design points of a two-point optimal design are
   \eqn{\{\mbox{point1} = (I_1, S_1, Z_1), \mbox{point2} = (I_2, S_2, Z_2)\}}{{point1 = (I1, S1, Z1), point2 = (I2, S2, Z2)}}.
    Then, the argument \code{x} is equivalent to
    \code{x = c(I1, I2, S1, S2, Z1, Z2)}.
}
\examples{
p <- c(1, -2, 1, -1)
prior4.4 <- uniform(p -1.5, p + 1.5)
formula4.4 <- ~exp(b0+b1*x1+b2*x2+b3*x1*x2)/(1+exp(b0+b1*x1+b2*x2+b3*x1*x2))
prob4.4 <- ~1-1/(1+exp(b0 + b1 * x1 + b2 * x2 + b3 * x1 * x2))
predvars4.4 <-  c("x1", "x2")
parvars4.4 <- c("b0", "b1", "b2", "b3")


# Locally D-optimal design is as follows:
## weight and point of D-optimal design
# Point1     Point2     Point3     Point4
# /1.00000 \\ /-1.00000\\ /0.06801 \\ /1.00000 \\
# \\-1.00000/ \\-1.00000/ \\1.00000 / \\1.00000 /
#   Weight1    Weight2    Weight3    Weight4
# 0.250      0.250      0.250      0.250

xopt_D <- c(1, -1, .0680, 1, -1, -1, 1, 1)
wopt_D <- rep(.25, 4)

# Let see if we use only three of the design points, what is the relative efficiency.
leff(formula = formula4.4, predvars = predvars4.4, parvars = parvars4.4, family = binomial(),
     x = c(1, -1, .0680,  -1, -1, 1), w = c(.33, .33, .33),
     inipars = p,
     xopt = xopt_D, wopt = wopt_D)
# Wow, it heavily drops!


# Locally P-optimal design has only one support point and is -1 and 1
xopt_P <- c(-1, 1)
wopt_P <- 1

# What is the relative P-efficiency of the D-optimal design with respect to P-optimal design?
leff(formula = formula4.4, predvars = predvars4.4, parvars = parvars4.4, family = binomial(),
     x = xopt_D, w = wopt_D,
     inipars = p,
     type = "PA",
     prob = prob4.4,
     xopt = xopt_P, wopt = wopt_P)
# .535


}
\references{
McGree, J. M., Eccleston, J. A., and Duffull, S. B. (2008). Compound optimal design criteria for nonlinear models. Journal of Biopharmaceutical Statistics, 18(4), 646-661.
}
