% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/UnivariateFitter.R
\name{UnivariateFitters}
\alias{UnivariateFitters}
\alias{UnivariateFitter}
\alias{Fitters}
\alias{GenUnivariateFitter}
\title{Functions used to fit GeDS objects w/univariate spline regression
component}
\usage{
UnivariateFitter(
  X,
  Y,
  Z = NULL,
  offset = rep(0, NROW(Y)),
  weights = rep(1, length(X)),
  beta = 0.5,
  phi = 0.5,
  min.intknots = 0,
  max.intknots = 300,
  q = 2,
  extr = range(X),
  show.iters = FALSE,
  tol = as.double(1e-12),
  stoptype = c("SR", "RD", "LR"),
  higher_order = TRUE,
  intknots = NULL,
  only_predictions = FALSE
)

GenUnivariateFitter(
  X,
  Y,
  Z = NULL,
  offset = rep(0, NROW(Y)),
  weights = rep(1, length(X)),
  family = gaussian(),
  beta = 0.5,
  phi = 0.5,
  min.intknots = 0,
  max.intknots = 300,
  q = 2,
  extr = range(X),
  show.iters = F,
  tol = as.double(1e-12),
  stoptype = c("SR", "RD", "LR"),
  higher_order = TRUE
)
}
\arguments{
\item{X}{a numeric vector containing \eqn{N} sample values of the covariate
chosen to enter the spline regression component of the predictor model.}

\item{Y}{a vector of size \eqn{N} containing the observed values of the
response variable \eqn{y}.}

\item{Z}{a design matrix with \eqn{N} rows containing other covariates
selected to enter the parametric component of the predictor model (see
\code{\link[=formula.GeDS]{formula}}). If no such covariates are selected, it
is set to \code{NULL} by default.}

\item{offset}{a vector of size \eqn{N} that can be used to specify a fixed
covariate to be included in the predictor model avoiding the estimation of
its corresponding regression coefficient. In case more than one covariate is
fixed, the user should sum the corresponding coordinates of the fixed
covariates to produce one common \eqn{N}-vector of coordinates. The
\code{offset} argument is particularly useful when using 
\code{GenUnivariateFitter} if the link function used is not the identity.}

\item{weights}{an optional vector of size \eqn{N} of `prior weights' to be
put on the observations in the fitting process in case the user requires
weighted GeDS fitting. It is \code{NULL} by default.}

\item{beta}{numeric parameter in the interval \eqn{[0,1]} tuning the knot
placement in stage A of GeDS. See the description of \code{\link{NGeDS}} or
\code{\link{GGeDS}}.}

\item{phi}{numeric parameter in the interval \eqn{[0,1]} specifying the
threshold for the stopping rule  (model selector) in stage A of GeDS. See
also \code{stoptype} and details in the description of \code{\link{NGeDS}} or
\code{\link{GGeDS}}.}

\item{min.intknots}{optional parameter allowing the user to set a minimum
number of internal knots required. By default equal to zero.}

\item{max.intknots}{optional parameter allowing the user to set a maximum
number of internal knots to be added by the GeDS estimation algorithm. By
default equal to the number of internal knots \eqn{\kappa} for the saturated
GeDS model (i.e. \eqn{\kappa=N-2}).}

\item{q}{numeric parameter which allows to fine-tune the stopping rule of
stage A of GeDS, by default equal to 2. See details in the description of
\code{\link{NGeDS}} or \code{\link{GGeDS}}.}

\item{extr}{numeric vector of 2 elements representing the left-most and
right-most limits of the interval embedding the sample values of \code{X}. By
default equal correspondingly to the smallest and largest values of \code{X}.}

\item{show.iters}{logical variable indicating whether or not to print 
information at each step. By default equal to \code{FALSE}.}

\item{tol}{numeric value indicating the tolerance to be used in the knot
placement steps in stage A. By default equal to 1E-12. See details below.}

\item{stoptype}{a character string indicating the type of GeDS stopping rule
to be used. It should be either \code{"SR"}, \code{"RD"} or \code{"LR"},
partial match allowed. See details of \code{\link{NGeDS}} or
\code{\link{GGeDS}}.}

\item{higher_order}{a logical that defines whether to compute the higher
order fits (quadratic and cubic) after stage A is run. Default is
\code{TRUE}.}

\item{intknots}{vector of initial internal knots from which to start the GeDS
Stage A iterations. See Section 3 of Kaishev et al. (2016). Default is \code{NULL}.}

\item{only_predictions}{logical, if \code{TRUE} only predictions are computed.}

\item{family}{a description of the error distribution and link function to be
used in the model. This can be a character string naming a family function
(e.g. \code{"gaussian"}), the family function itself (e.g.
\code{\link[stats]{gaussian}}) or the result of a call to a family function
(e.g. \code{gaussian()}). See \link[stats]{family} for details on family
functions.}
}
\value{
A \code{\link{GeDS-Class}} object, but without the \code{Formula},
\code{extcall}, \code{terms} and \code{znames} slots.
}
\description{
These are computing engines called by \code{\link{NGeDS}} and
\code{\link{GGeDS}}, needed for the underlying fitting procedures.
}
\details{
The functions \code{UnivariateFitter} and \code{GenUnivariateFitter} are in
general not intended to be used directly, they should be called through
\code{\link{NGeDS}} and \code{\link{GGeDS}}. However, in case there is a need
for multiple GeDS fitting (as may be the case e.g. in Monte Carlo simulations)
it may be efficient to use the fitters outside the main functions.

The argument \code{tol} is used in the knot placement procedure of stage A of
the GeDS algorithm in order to check whether the current knot \eqn{\delta^*} 
is set at an acceptable location or not. If there exists a knot \eqn{\delta_i}
such that \eqn{|\delta^* - \delta_i| < }\code{tol}, \eqn{\delta^*}, then the
new knot is considered to be coalescent with an existing one, it is discarded
and the algorithm seeks alternative knot locations. By default it is equal to
1e-12.

See \code{\link{NGeDS}} and \code{\link{GGeDS}}, Kaishev et al. (2016) and
Dimitrova et al. (2023) for further details.
}
\examples{
# Examples similar to the ones
# presented in NGeDS and in GGeDS

# Generate a data sample for the response variable
# Y and the covariate X
set.seed(123)
N <- 500
f_1 <- function(x) (10*x/(1+100*x^2))*4+4
X <- sort(runif(N ,min = -2, max = 2))
# Specify a model for the mean of Y to include only
# a component non-linear in X, defined by the function f_1
means <- f_1(X)
# Add (Normal) noise to the mean of Y
Y <- rnorm(N, means, sd = 0.1)

# Fit a Normal GeDS regression model using the fitter function
(Gmod <- UnivariateFitter(X, Y, beta = 0.6, phi = 0.995,
           extr = c(-2,2)))

##############################################################
# second: very similar example, but based on Poisson data
set.seed(123)
X <- sort(runif(N , min = -2, max = 2))
means <- exp(f_1(X))
Y <- rpois(N,means)
(Gmod2 <- GenUnivariateFitter(X, Y, beta = 0.2,
            phi = 0.995, family = poisson(), extr = c(-2,2)))

# a plot showing quadratic and cubic fits,
# in the predictor scale
plot(X,log(Y), xlab = "x", ylab = expression(f[1](x)))
lines(Gmod2, n = 3, col = "red")
lines(Gmod2, n = 4, col = "blue", lty = 2)
legend("topleft", c("Quadratic","Cubic"),
     col = c("red","blue"), lty = c(1,2))

}
\references{
Kaishev, V.K., Dimitrova, D.S., Haberman, S., & Verrall, R.J. (2016).
Geometrically designed, variable knot regression splines.
\emph{Computational Statistics}, \strong{31}, 1079--1105. \cr
DOI: \doi{10.1007/s00180-015-0621-7}

Dimitrova, D. S., Kaishev, V. K., Lattuada, A. and Verrall, R. J.  (2023).
Geometrically designed variable knot splines in generalized (non-)linear
models.
\emph{Applied Mathematics and Computation}, \strong{436}. \cr
DOI: \doi{10.1016/j.amc.2022.127493}
}
\seealso{
\code{\link{NGeDS}} and \code{\link{GGeDS}}.
}
