% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CPF.R
\name{CPF}
\alias{CPF}
\title{Conditional Pareto Front simulations}
\usage{
CPF(fun1sims, fun2sims, response, paretoFront = NULL, f1lim = NULL,
  f2lim = NULL, refPoint = NULL, n.grid = 100, compute.VorobExp = TRUE,
  compute.VorobDev = TRUE)
}
\arguments{
\item{fun1sims}{numeric matrix containing the conditional simulations of the first output (one sample in each row),}

\item{fun2sims}{numeric matrix containing the conditional simulations of the second output (one sample in each row),}

\item{response}{a matrix containing the value of the two objective functions, one
output per row,}

\item{paretoFront}{optional matrix corresponding to the Pareto front of the observations. It is 
estimated from \code{response} if not provided,}

\item{f1lim}{optional vector (see details),}

\item{f2lim}{optional vector (see details),}

\item{refPoint}{optional vector (see details),}

\item{n.grid}{integer determining the grid resolution,}

\item{compute.VorobExp}{optional boolean indicating whether the Vorob'ev Expectation
should be computed. Default is \code{TRUE},}

\item{compute.VorobDev}{optional boolean indicating whether the Vorob'ev deviation
should be computed. Default is \code{TRUE}.}
}
\value{
A list which is given the S3 class "\code{CPF}".
\itemize{
 \item{\code{x, y}}{: locations of grid lines at which the values of the attainment
  are computed,}
 \item{\code{values}}{: numeric matrix containing the values of the attainment on the grid,}
 \item{\code{PF}}{: matrix corresponding to the Pareto front of the observations,}
 \item{\code{responses}}{: matrix containing the value of the two objective functions, one
objective per column,}
\item{\code{fun1sims, fun2sims}}{: conditional simulations of the first/second output,}
\item{\code{VE}}{: Vorob'ev expectation, computed if \code{compute.VorobExp = TRUE} (default),}
\item{\code{beta_star}}{: Vorob'ev threshold, computed if \code{compute.VorobExp = TRUE} (default),}
\item{\code{VD}}{: Vorov'ev deviation, computed if \code{compute.VorobDev = TRUE} (default),}
}
}
\description{
Compute (on a regular grid) the empirical attainment function from conditional simulations 
of Gaussian processes corresponding to two objectives. This is used to estimate the Vorob'ev
expectation of the attained set and the Vorob'ev deviation.
}
\details{
Works with two objectives. The user can provide locations of grid lines for
 computation of the attainement function with vectors \code{f1lim} and \code{f2lim}, in the form of regularly spaced points. 
 It is possible to provide only \code{refPoint} as a reference for hypervolume computations.
 When missing, values are determined from the axis-wise extrema of the simulations.
}
\examples{
library(DiceDesign)
set.seed(42)

nvar <- 2

fname <- "P1" # Test function

# Initial design
nappr <- 10
design.grid <- maximinESE_LHS(lhsDesign(nappr, nvar, seed = 42)$design)$design
response.grid <- t(apply(design.grid, 1, fname))

# kriging models: matern5_2 covariance structure, linear trend, no nugget effect
mf1 <- km(~., design = design.grid, response = response.grid[,1])
mf2 <- km(~., design = design.grid, response = response.grid[,2])

# Conditional simulations generation with random sampling points 
nsim <- 40
npointssim <- 150 # increase for better results
Simu_f1 <- matrix(0, nrow = nsim, ncol = npointssim)
Simu_f2 <- matrix(0, nrow = nsim, ncol = npointssim)
design.sim <- array(0, dim = c(npointssim, nvar, nsim))

for(i in 1:nsim){
  design.sim[,,i] <- matrix(runif(nvar*npointssim), nrow = npointssim, ncol = nvar)
  Simu_f1[i,] <- simulate(mf1, nsim = 1, newdata = design.sim[,,i], cond = TRUE,
                          checkNames = FALSE, nugget.sim = 10^-8)
  Simu_f2[i,] <- simulate(mf2, nsim = 1, newdata = design.sim[,,i], cond = TRUE, 
                          checkNames = FALSE, nugget.sim = 10^-8)
}

# Attainment and Voreb'ev expectation + deviation estimation
CPF1 <- CPF(Simu_f1, Simu_f2, response.grid)

# Details about the Vorob'ev threshold  and Vorob'ev deviation
summary(CPF1)

# Graphics
plot(CPF1)

}
\references{
M. Binois, D. Ginsbourger and O. Roustant (2015), Quantifying Uncertainty on Pareto Fronts with Gaussian process conditional simulations, 
\emph{European Journal of Operational Research}, 243(2), 386-394. \cr \cr
C. Chevalier (2013), \emph{Fast uncertainty reduction strategies relying on Gaussian process models}, University of Bern, PhD thesis. \cr \cr
I. Molchanov (2005), \emph{Theory of random sets}, Springer.
}
\seealso{
Methods \code{coef}, \code{summary} and \code{plot} can be used to get the coefficients from a \code{CPF} object, 
to obtain a summary or to display the attainment function (with the Vorob'ev expectation if \code{compute.VorobExp} is \code{TRUE}).
}

