#' @export
#' @title Some Examples of Security Pricing
#' @aliases gemSecurityPricingExample
#' @description These examples illustrate how to find the equilibrium of a security market by the function sdm2 and by computing marginal utility of securities (see Sharpe, 2008).
#' @param ... arguments to be passed to the function sdm2.
#' @return  A general equilibrium.
#' @references Danthine, J. P., Donaldson, J. (2005, ISBN: 9780123693808) Intermediate Financial Theory. Elsevier Academic Press.
#' @references Sharpe, William F. (2008, ISBN: 9780691138503) Investors and Markets: Portfolio Choices, Asset Prices, and Investment Advice. Princeton University Press.
#' @references Xu Gao (2018, ISBN: 9787300258232) Twenty-five Lectures on Financial Economics. Beijing: China Renmin University Press. (In Chinese)
#' @references https://web.stanford.edu/~wfsharpe/apsim/index.html
#' @seealso \code{\link{gemSecurityPricing}}.
#' @examples
#' \donttest{
#' #### an example of Danthine and Donaldson (2005, section 8.3).
#' uf <- function(x) 0.5 * x[1] + 0.9 * (1 / 3 * log(x[2]) + 2 / 3 * log(x[3]))
#'
#' ge <- sdm2(
#'   A = function(state) {
#'     VMU <- marginal_utility(state$last.A %*% dg(state$last.z), diag(3), uf, state$p)
#'     Ratio <- sweep(VMU, 2, colMeans(VMU), "/")
#'
#'     A <- state$last.A * Ratio
#'     prop.table(A, 2)
#'   },
#'   B = matrix(0, 3, 2),
#'   S0Exg = matrix(c(
#'     10, 5,
#'     1, 4,
#'     2, 6
#'   ), 3, 2, TRUE),
#'   names.commodity = c("secy1", "secy2", "secy3"),
#'   names.agent = c("agt1", "agt2"),
#'   numeraire = "secy1",
#'   ts = TRUE
#' )
#'
#' ge$p
#'
#' #### an example of Sharpe (2008, chapter 2)
#' secy1 <- c(1, 0, 0, 0, 0)
#' secy2 <- c(0, 1, 1, 1, 1)
#' secy3 <- c(0, 5, 3, 8, 4) - 3 * secy2
#' secy4 <- c(0, 3, 5, 4, 8) - 3 * secy2
#' # unit security payoff matrix
#' USP <- cbind(secy1, secy2, secy3, secy4)
#'
#' prob <- c(0.15, 0.25, 0.25, 0.35)
#' wt <- prop.table(c(1, 0.96 * prob)) # weights
#'
#' gamma.agt1 <- 1.5
#' gamma.agt2 <- 2.5
#'
#' ge <- sdm2(
#'   A = function(state) {
#'     Payoff <- USP %*% (state$last.A %*% dg(state$last.z))
#'
#'     VMU <- marginal_utility(Payoff, USP, list(
#'       function(x) CES(alpha = 1, beta = wt, x = x, es = 1 / gamma.agt1),
#'       function(x) CES(alpha = 1, beta = wt, x = x, es = 1 / gamma.agt2)
#'     ), price = state$p)
#'     Ratio <- sweep(VMU, 2, colMeans(VMU), "/")
#'
#'     A <- state$last.A * ratio_adjust(Ratio, coef = 0.05, method = "linear")
#'
#'     A <- prop.table(A, 2)
#'   },
#'   B = matrix(0, 4, 2),
#'   S0Exg = matrix(c(
#'     49, 49,
#'     30, 30,
#'     10, 0,
#'     0, 10
#'   ), 4, 2, TRUE),
#'   names.commodity = c("secy1", "secy2", "secy3", "secy4"),
#'   names.agent = c("agt1", "agt2"),
#'   numeraire = "secy1"
#' )
#'
#' ge$p
#' ge$p[3:4] + 3 * ge$p[2]
#'
#' #### an example of Xu (2018, section 10.4, P151)
#' secy1 <- c(1, 0, 0)
#' secy2 <- c(0, 1, 0)
#' secy3 <- c(0, 0, 1)
#' prob <- c(0.5, 0.5)
#' wt <- c(1, prob)
#' USP <- cbind(secy1, secy2, secy3)
#'
#' gamma.agt1 <- 1
#' gamma.agt2 <- 0.5
#'
#' ge <- sdm2(
#'   A = function(state) {
#'     Payoff <- USP %*% (state$last.A %*% dg(state$last.z))
#'
#'     VMU <- marginal_utility(Payoff, USP, list(
#'       # Here CRRA(...)$u, CRRA(...)$CE and CES functions are interexchangeable.
#'       function(x) CRRA(x, gamma = gamma.agt1, p = wt)$u,
#'       function(x) CES(alpha = 1, beta = wt, x = x, es = 1 / gamma.agt2)
#'     ), state$p)
#'     Ratio <- sweep(VMU, 2, colMeans(VMU), "/")
#'
#'     A <- state$last.A * Ratio
#'     prop.table(A, 2)
#'   },
#'   B = matrix(0, 3, 2),
#'   S0Exg = matrix(c(
#'     1, 0,
#'     0, 0.5,
#'     0, 2
#'   ), 3, 2, TRUE),
#'   names.commodity = c("secy1", "secy2", "secy3"),
#'   names.agent = c("agt1", "agt2"),
#'   numeraire = "secy1",
#'   maxIteration = 1,
#'   ts = TRUE
#' )
#'
#' ge$p #c(1, (1 + sqrt(5)) / 4, (1 + sqrt(17)) / 16)
#'
#' ## the same as above.
#' dst.agt1 <- node_new("util",
#'   type = "CD", alpha = 1, beta = c(0.5, 0.25, 0.25),
#'   "secy1", "secy2", "secy3"
#' )
#'
#' dst.agt2 <- node_new("util",
#'   type = "CES", alpha = 1, beta = c(2, 1, 1), sigma = 0.5,
#'   "secy1", "secy2", "secy3"
#' )
#'
#' ge <- sdm2(
#'   A = list(dst.agt1, dst.agt2),
#'   B = matrix(0, 3, 2),
#'   S0Exg = matrix(c(
#'     1, 0,
#'     0, 0.5,
#'     0, 2
#'   ), 3, 2, TRUE),
#'   names.commodity = c("secy1", "secy2", "secy3"),
#'   names.agent = c("agt1", "agt2"),
#'   numeraire = "secy1",
#'   maxIteration = 1,
#'   ts = TRUE
#' )
#'
#' ge$p
#' }

gemSecurityPricingExample <- function(...) sdm2(...)
