% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_spike_paths.R
\name{estimate_spike_paths}
\alias{estimate_spike_paths}
\title{Estimate spike train, underlying calcium concentration, and changepoints
based on a fluorescence trace. Automatic tuning parameter selection within
a range of values [lambda_min, lambda_max]}
\usage{
estimate_spike_paths(dat, gam, lambda_min = 0.01, lambda_max = 10,
  constraint = FALSE, EPS = 1e-04, max_iters = 10)
}
\arguments{
\item{dat}{fluorescence data}

\item{gam}{a scalar value for the AR(1) decay parameter}

\item{lambda_min}{minimum lamba value}

\item{lambda_max}{maximum lamba value}

\item{constraint}{boolean specifying constrained or unconstrained
optimization problem (see below)}

\item{EPS}{double specifying the minimum calcium value}

\item{max_iters}{maximum number of tuning parameters attempted}
}
\value{
Returns a list with elements:

\code{path_stats} a dataframe with summary statistics (number of spikes, tuning parameters, cost)

\code{path_fits} a list with estimated_spikes object for each tuning parameter

\code{approximate_path} a boolean indicating whether an early stopping criterion condition occurred
}
\description{
Estimate spike train, underlying calcium concentration, and changepoints
based on a fluorescence trace. Automatic tuning parameter selection within
a range of values [lambda_min, lambda_max]
}
\details{
This algorithm solves the optimization problems

\strong{AR(1) model:}

minimize_{c1,...,cT} 0.5 sum_{t=1}^T ( y_t - c_t )^2 + lambda sum_{t=2}^T
1_[c_t != max(gam c_{t-1}, EPS)]

for the global optimum, where y_t is the observed fluorescence at the tth
timestep.

\strong{Constrained AR(1) model:}

minimize_{c1,...,cT} 0.5 sum_{t=1}^T ( y_t - c_t )^2 + lambda sum_{t=2}^T
1_[c_t != max(gam c_{t-1}, EPS)]

subject to c_t >= max(gam c_{t-1}, EPS), t = 2, ..., T

We introduce the constant EPS > 0, to avoid
arbitrarily small calcium concentrations that would result in numerical
instabilities. In practice, this means that the estimated calcium
concentration decays according to the AR(1) model for values greater than EPS and
is equal to EPS thereafter.

When estimating the spikes, it is not necessary to explicitly compute the
calcium concentration. Therefore, if only the spike times are required, the
user can avoid this computation cost by setting the estimate_calcium
boolean to false. Because estimating the calcium requires additional computation time,
we suggest estimating the calcium only if it is needed.

Given the set of estimated spikes produced from the estimate_spike, the
calcium concentration can be estimated with the estimate_calcium function
(see examples below).

For additional information see:

1. Jewell, Hocking, Fearnhead, and Witten (2018) <arXiv:1802.07380> and

2. Jewell, Sean; Witten, Daniela. Exact spike train inference via l0 optimization. 
Ann. Appl. Stat. 12 (2018), no. 4, 2457--2482. doi:10.1214/18-AOAS1162. 
https://projecteuclid.org/euclid.aoas/1542078052
}
\examples{

sim <- simulate_ar1(n = 500, gam = 0.95, poisMean = 0.009, sd = 0.05, seed = 1)
plot(sim)

## Fits for tuning parameters between [0.1, 10]
fits <- estimate_spike_paths(dat = sim$fl, gam = 0.95, lambda_min = 0.1, lambda_max = 10)
print(fits)
plot(fits)
print(fits$path_fits[[1]])
plot(fits$path_fits[[1]])

## Fits for a single tuning parameter

# AR(1) model
fit <- estimate_spikes(dat = sim$fl, gam = 0.95, lambda = 1)
print(fit)

# compute fitted values from prev. fit
fit <- estimate_calcium(fit)
plot(fit)

# or
fit <- estimate_spikes(dat = sim$fl, gam = 0.95, lambda = 1, estimate_calcium = TRUE)
plot(fit)

# Constrained AR(1) model
fit <- estimate_spikes(dat = sim$fl, gam = 0.95, lambda = 1, constraint = TRUE,
                                                    estimate_calcium = TRUE)
print(fit)
plot(fit)

}
\seealso{
\strong{Estimate spikes:}
\code{\link{estimate_spikes}}
\code{\link{estimate_calcium}}

\strong{Simulate:}
\code{\link{simulate_ar1}}
}
