% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geometryfns.R
\name{auto_BAUs}
\alias{auto_BAUs}
\title{Automatic BAU generation}
\usage{
auto_BAUs(
  manifold,
  type = NULL,
  cellsize = NULL,
  isea3h_res = NULL,
  data = NULL,
  nonconvex_hull = TRUE,
  convex = -0.05,
  tunit = NULL,
  xlims = NULL,
  ylims = NULL,
  spatial_BAUs = NULL,
  ...
)
}
\arguments{
\item{manifold}{object of class \code{manifold}}

\item{type}{either ``grid'' or ``hex'', indicating whether gridded or hexagonal BAUs should be used. If \code{type} is unspecified, ``hex'' will be used if we are on the sphere, and ``grid'' will used otherwise}

\item{cellsize}{denotes size of gridcell when \code{type} = ``grid''. Needs to be of length 1 (square-grid case) or a vector of length \code{dimensions(manifold)} (rectangular-grid case)}

\item{isea3h_res}{resolution number of the isea3h DGGRID cells for when type is ``hex'' and manifold is the surface of a \code{sphere}}

\item{data}{object of class \code{SpatialPointsDataFrame}, \code{SpatialPolygonsDataFrame},  \code{STIDF}, or \code{STFDF}. Provision of \code{data} implies that the domain is bounded, and is thus necessary when the manifold is a \code{real_line, plane}, or \code{STplane}, but is not necessary when the manifold is the surface of a \code{sphere}}

\item{nonconvex_hull}{flag indicating whether to use \code{fmesher} to generate a non-convex hull. Otherwise a convex hull is used}

\item{convex}{convex parameter used for smoothing an extended boundary when working on a bounded domain (that is, when the object \code{data} is supplied); see details}

\item{tunit}{temporal unit when requiring space-time BAUs. Can be "secs", "mins", "hours", etc.}

\item{xlims}{limits of the horizontal axis (overrides automatic selection)}

\item{ylims}{limits of the vertical axis (overrides automatic selection)}

\item{spatial_BAUs}{object of class \code{SpatialPolygonsDataFrame} or \code{SpatialPixelsDataFrame} representing the spatial BAUs to be used in a spatio-temporal setting (if left \code{NULL}, the spatial BAUs are constructed automatically using the data)}

\item{...}{currently unused}
}
\description{
This function calls the generic function \code{auto_BAU} (not exported) after a series of checks and is the easiest way to generate a set of Basic Areal Units (BAUs) on the manifold being used; see details.
}
\details{
\code{auto_BAUs} constructs a set of Basic Areal Units (BAUs) used both for data pre-processing and for prediction. As such, the BAUs need to be of sufficienly fine resolution so that inferences are not affected due to binning.

Two types of BAUs are supported by \code{FRK}: ``hex'' (hexagonal) and ``grid'' (rectangular). In order to have a ``grid'' set of BAUs, the user should specify a cellsize of length one, or of length equal to the dimensions of the manifold, that is, of length 1 for \code{real_line} and of length 2 for the surface of a \code{sphere} and \code{plane}. When a ``hex'' set of BAUs is desired, the first element of \code{cellsize} is used to determine the side length by dividing this value by approximately 2. The argument \code{type} is ignored with \code{real_line} and ``hex'' is not available for this manifold.

  If the object \code{data} is provided, then automatic domain selection may be carried out by employing the \code{fmesher} function \code{fm_nonconvex_hull_inla}, which finds a (non-convex) hull surrounding the data points (or centroids of the data polygons). This domain is extended and smoothed using the parameter \code{convex}. The parameter \code{convex} should be negative, and a larger absolute value for \code{convex} results in a larger domain with smoother boundaries.
}
\examples{
## First a 1D example
library(sp)
set.seed(1)
data <- data.frame(x = runif(10)*10, y = 0, z= runif(10)*10)
coordinates(data) <- ~x+y
Grid1D_df <- auto_BAUs(manifold = real_line(),
                       cellsize = 1,
                       data=data)
\dontrun{spplot(Grid1D_df)}

## Now a 2D example
data(meuse)
coordinates(meuse) = ~x+y # change into an sp object

## Grid BAUs
GridPols_df <- auto_BAUs(manifold = plane(),
                         cellsize = 200,
                         type = "grid",
                         data = meuse,
                         nonconvex_hull = 0)
\dontrun{plot(GridPols_df)}

## Hex BAUs
HexPols_df <- auto_BAUs(manifold = plane(),
                        cellsize = 200,
                        type = "hex",
                        data = meuse,
                        nonconvex_hull = 0)
\dontrun{plot(HexPols_df)}
}
\seealso{
\code{\link{auto_basis}} for automatically constructing basis functions.
}
