\name{erpfatest}
\alias{erpfatest}
\title{
Adaptive Factor-Adjustement for multiple testing of ERP data
}
\description{
An adaptive factor-adjusted FDR- and FWER-controlling multiple testing procedures for ERP data.
The procedure is described in detail in Sheu, Perthame, Lee, and Causeur (2016).
}
\usage{
erpfatest(dta, design, design0 = NULL, method = "BH", nbf = NULL, nbfmax = 15, 
   alpha = 0.05, pi0 = 1, wantplot = FALSE, s0 = NULL, min.err = 0.001, 
   maxiter = 5, verbose = FALSE)
}
\arguments{
  \item{dta}{Data frame containing the ERP measurements: each column corresponds to a time frame and each row to a curve.}
  \item{design}{Design matrix of the full model for the relationship between the ERP and the experimental variables.
          Typically the output of the function model.matrix}
  \item{design0}{Design matrix of the null model. Typically a submodel of the full model, obtained by removing columns from design.
          Default is NULL, corresponding to the model with no covariates.}
  \item{method}{FDR- or FWER- controlling multiple testing procedures as available in the function p.adjust. Default is "BH",
          for the Benjamini-Hochberg procedure (see Benjamini and Hochberg, 1995).}
  \item{nbf}{Number of factors in the residual covariance model.
          Default is NULL: the number of factors is determined by minimization of the variance inflation criterion as
          suggested in Friguet et al. (2009).}
  \item{nbfmax}{Only required if nbf=NULL. The largest possible number of factors.}
  \item{alpha}{The FDR or FWER control level. Default is 0.05}
  \item{pi0}{An estimate of the proportion of true null hypotheses, which can be plugged into an FDR controlling multiple
          testing procedure to improve its efficiency.
          Default is 1, corresponding to the classical FDR controlling procedures. If NULL, the proportion is estimated
          using the function pval.estimate.eta0 of package fdrtool, with the method proposed by Storey and Tibshirani (2003).}
  \item{wantplot}{If TRUE, a diagnostic plot is produced to help choosing the number of factors. Only active if nbf=NULL.}
  \item{s0}{Prior knowledge of the time frames for which no signal is expected. For example,
          s0=c(1:50, 226:251) specifies that the first 50 time frames and time frames between 226 and 251 are
          known not to contain ERP signal. s0 can also be specified by giving the lower and upper fraction of the
          entire time interval in which the signal is to be searched for. For example: s0=c(0.2, 0.9) means that
          ERP signals are not expected for
          for the first 20 percent and last 10 percent of the time frames measured. Defaul is NULL and it
          initiates a data-driven determination of s0.}
  \item{min.err}{Control parameter for convergence of the iterative algorithm. Default is 1e-03.}
  \item{maxiter}{Maximum number of iterations in algorithms. Default is 5.}
  \item{verbose}{If TRUE, details are printed along the iterations of the algorithm. Default is FALSE.}
}
\details{The method is described in Sheu et al. (2016). It combines a decorrelation step based on a regression factor model as
in Leek and Storey (2008), Friguet et al. (2009) or Sun et al. (2012) with an adaptive estimation of the ERP signal.
The multiple testing corrections of the p-values are described in the help documentation of the function p.adjust of
package stats.}
\value{
\item{pval}{p-values of the Adaptive Factor-Adjusted tests.}
\item{correctedpval}{Corrected p-values, for the multiplicity of tests. Depends on the multiple testing method (see
 function p.adjust).}
\item{significant}{Indices of the time points for which the test is positive.}
\item{pi0}{Value for pi0: if the input argument pi0 is NULL, the output is the estimated proportion of
               true null hypotheses using the method by Storey and Tibshirani (2003).}
\item{test}{Factor-Adjusted F-statistics.}
\item{df1}{Residual degrees of freedom for the full model.}
\item{df0}{Residual degrees of freedom for the null model.}
\item{nbf}{Number of factors for the residual covariance model.}
\item{signal}{Estimated signal: a pxT matrix, where p is the difference between the numbers of parameters in the full and null models and T the number of frames.}
\item{r2}{R-squared values for each of the T linear models.}
}
\references{

Causeur, D., Chu, M.-C., Hsieh, S., Sheu, C.-F. 2012. A factor-adjusted multiple testing procedure for ERP data analysis. Behavior Research Methods, 44, 635-643.

Friguet, C., Kloareg, M., Causeur, D. 2009. A factor model approach to multiple testing under dependence. Journal of the American Statistical Association, 104, 1406-1415.

Leek, J.T., Storey, J.D. 2008. A general framework for multiple testing dependence. Proceedings of the National Academy of Sciences of the United States of America, 105, 18718-18723.

Sheu, C.-F., Perthame, E., Lee Y.-S. and Causeur, D. 2016. Accounting for time dependence in large-scale multiple testing of event-related potential data. To appear in Annals of Applied Statistics.

Storey, J. D., Tibshirani, R. 2003. Statistical significance for genome-wide experiments. Proceedings of the National Academy of Sciences of the United States of America, 100, 9440-9445.

Sun, Y., Zhang, N.R., Owen, A.B. 2012. Multiple hypothesis testing adjusted for latent variables, with an
 application to the AGEMAP gene expression data. The Annals of Applied Statistics, 6, no. 4, 1664-1688.

}
\author{
David Causeur - david.causeur@agrocampus-ouest.fr
}
\seealso{
\code{\link{erpavetest}}, \code{\link{erptest}}, \code{\link{gbtest}}, \code{\link[stats]{p.adjust}}, \code{\link[fdrtool]{pval.estimate.eta0}}
}
\examples{
require(mnormt)
require(fdrtool)

data(erpcz)
data(simerp)

# Paired t-tests for the comparison of ERP curves between two groups

tests = erpfatest(erpcz[,1:251],design=model.matrix(~Subject+Instruction,data=erpcz),
   design0=model.matrix(~Subject,data=erpcz),nbf=3,s0=c(1:50,226:251),min.err=1e-01)

frames = seq(0,1001,4)
plot(frames,tests$signal,type="l",xlab="Time (ms)",ylab="Difference ERP curves")
points(frames[tests$significant],rep(0,length(tests$significant)),pch=16,col="blue")
title("Paired comparison at electrode CZ")

# Tests for significance of correlations

tests = erpfatest(simerp[,1:251],design=model.matrix(~y,data=simerp),
   nbf=5,min.err=1e-01,maxiter=10)
plot(frames,sign(tests$signal)*sqrt(tests$r2),type="l",
   xlab="Time (ms)",ylab="Correlation",ylim=c(-1,1))
points(frames[tests$significant],rep(-1,length(tests$significant)),
   pch=16,col="blue")
title("Simulation")

}
\keyword{ERP data}
\keyword{FDR}
\keyword{Factor-adjustment}
\keyword{Multiple testing}
