% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DynForest.R
\name{DynForest}
\alias{DynForest}
\title{Random forest with multivariate longitudinal endogenous covariates}
\usage{
DynForest(
  timeData = NULL,
  fixedData = NULL,
  idVar = NULL,
  timeVar = NULL,
  timeVarModel = NULL,
  Y = NULL,
  ntree = 200,
  mtry = NULL,
  nodesize = 1,
  minsplit = 2,
  cause = 1,
  nsplit_option = "quantile",
  ncores = NULL,
  seed = 1234,
  verbose = TRUE
)
}
\arguments{
\item{timeData}{A data.frame containing the id and time measurements variables and the time-dependent predictors.}

\item{fixedData}{A data.frame containing the id variable and the time-fixed predictors. Categorical variables should be characterized as factor.}

\item{idVar}{A character indicating the name of variable to identify the subjects}

\item{timeVar}{A character indicating the name of time variable}

\item{timeVarModel}{A list for each time-dependent predictors containing a list of formula for fixed and random part from the mixed model}

\item{Y}{A list of output which should contain: \code{type} defines the nature of the outcome, can be "\code{surv}", "\code{numeric}" or "\code{factor}"; .}

\item{ntree}{Number of trees to grow. Default value set to 200.}

\item{mtry}{Number of candidate variables randomly drawn at each node of the trees. This parameter should be tuned by minimizing the OOB error. Default is defined as the square root of the number of predictors.}

\item{nodesize}{Minimal number of subjects required in both child nodes to split. Cannot be smaller than 1.}

\item{minsplit}{(Only with survival outcome) Minimal number of events required to split the node. Cannot be smaller than 2.}

\item{cause}{(Only with competing events) Number indicates the event of interest.}

\item{nsplit_option}{A character indicates how the values are chosen to build the two groups for the splitting rule (only for continuous predictors). Values are chosen using deciles (\code{nsplit_option}="quantile") or randomly (\code{nsplit_option}="sample"). Default value is "quantile".}

\item{ncores}{Number of cores used to grow trees in parallel. Default value is the number of cores of the computer-1.}

\item{seed}{Seed to replicate results}

\item{verbose}{A logical controlling the function progress. Default is \code{TRUE}}
}
\value{
DynForest function returns a list with the following elements:\tabular{ll}{
\code{data} \tab A list containing the data used to grow the trees \cr
\tab \cr
\code{rf} \tab A table with each tree in column. Provide multiple characteristics about the tree building \cr
\tab \cr
\code{type} \tab Outcome type \cr
\tab \cr
\code{times} \tab A numeric vector containing the time-to-event for all subjects \cr
\tab \cr
\code{cause} \tab Indicating the cause of interest \cr
\tab \cr
\code{causes} \tab A numeric vector containing the causes indicator \cr
\tab \cr
\code{Inputs} \tab A list of 3 elements: \code{Longitudinal}, \code{Numeric} and \code{Factor}. Each element contains the names of the predictors \cr
\tab \cr
\code{Longitudinal.model} \tab A list of longitudinal markers containing the formula used for modeling in the random forest \cr
\tab \cr
\code{param} \tab A list containing the hyperparameters \cr
\tab \cr
\code{comput.time} \tab Computation time \cr
}
}
\description{
Build a random forest using multivariate longitudinal endogenous covariates
}
\details{
The function currently supports survival (competing or single event), continuous or categorical outcome.

FUTUR IMPLEMENTATIONS:
\itemize{
\item Continuous longitudinal outcome
\item Functional data analysis
}
}
\examples{
\donttest{
data(pbc2)

# Get Gaussian distribution for longitudinal predictors
pbc2$serBilir <- log(pbc2$serBilir)
pbc2$SGOT <- log(pbc2$SGOT)
pbc2$albumin <- log(pbc2$albumin)
pbc2$alkaline <- log(pbc2$alkaline)

# Sample 100 subjects
set.seed(1234)
id <- unique(pbc2$id)
id_sample <- sample(id, 100)
id_row <- which(pbc2$id\%in\%id_sample)

pbc2_train <- pbc2[id_row,]

timeData_train <- pbc2_train[,c("id","time",
                                "serBilir","SGOT",
                                "albumin","alkaline")]

# Create object with longitudinal association for each predictor
timeVarModel <- list(serBilir = list(fixed = serBilir ~ time,
                                     random = ~ time),
                     SGOT = list(fixed = SGOT ~ time + I(time^2),
                                 random = ~ time + I(time^2)),
                     albumin = list(fixed = albumin ~ time,
                                    random = ~ time),
                     alkaline = list(fixed = alkaline ~ time,
                                     random = ~ time))

# Build fixed data
fixedData_train <- unique(pbc2_train[,c("id","age","drug","sex")])

# Build outcome data
Y <- list(type = "surv",
          Y = unique(pbc2_train[,c("id","years","event")]))

# Run DynForest function
res_dyn <- DynForest(timeData = timeData_train, fixedData = fixedData_train,
                     timeVar = "time", idVar = "id",
                     timeVarModel = timeVarModel, Y = Y,
                     ntree = 50, nodesize = 5, minsplit = 5,
                     cause = 2, ncores = 2, seed = 1234)
}
}
\references{
\itemize{
\item Devaux A., Helmer C., Genuer R., Proust-Lima C. (2023). Random survival forests with multivariate longitudinal endogenous covariates. SMMR \url{doi:10.1177/09622802231206477}
\item Devaux A., Proust-Lima C., Genuer R. (2023). Random Forests for time-fixed and time-dependent predictors: The DynForest R package. arXiv \url{doi:10.48550/arXiv.2302.02670}
}
}
\seealso{
\code{\link{summary.DynForest} \link{compute_OOBerror} \link{compute_VIMP} \link{compute_gVIMP} \link{predict.DynForest} \link{plot.DynForest}}
}
\author{
Anthony Devaux (\email{anthony.devauxbarault@gmail.com})
}
