#' Summarize bootstrap abundance uncertainty estimate output
#'
#' A simple function to calculate summaries of bootstrap output generated by \code{\link{bootdht}}.
#'
#' Summaries are only made for numeric outputs. Both median and mean are reported to allow assessment of bias. The coefficient of variation reported (in column \code{cv}) is based on the median calculated from the bootstraps.
#'
#' @param object output from \code{bootdht}
#' @param alpha value to use in confidence interval calculation (to obtain \code{alpha}/2 and 1-\code{alpha}/2 intervals
#' @param ... for S3 compatibility, unused.
#' @return a \code{data.frame} of summary statistics
#' @export
#' @importFrom stats median quantile
summary.dht_bootstrap <- function(object, alpha=0.05, ...){

  x <- list()

  x$nboot <- attr(object, "nboot")
  x$nbootsuccess <- attr(object, "nboot")-attr(object, "nbootfail")
  x$nbootfail <- attr(object, "nbootfail")
  x$alpha <- alpha

  class(object) <- "list"
  object <- as.data.frame(object)

  numcols <- unlist(lapply(object, is.numeric))

  # build a summary object
  sumfun <- function(x){
    if(is.numeric(x)){
      xx <- data.frame(median   = median(x, na.rm=TRUE),
                       mean     = mean(x, na.rm=TRUE),
                       se       = sqrt(var(x, na.rm=TRUE)),
                       lcl      = quantile(x, (alpha/2)),
                       ucl      = quantile(x, 1-(alpha/2)))
      xx$cv <- xx$se/xx$median
    }else{
      xx <- NULL
    }
    return(xx)
  }

  tn <- lapply(object, sumfun)
  x$tab <- do.call(rbind.data.frame, tn)

  class(x) <- "summary.dht_bootstrap"
  return(x)
}
