% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Deriv.R
\name{Deriv}
\alias{Deriv}
\alias{drule}
\title{Symbollic differentiation of an expression or function}
\usage{
Deriv(f, x = if (is.function(f)) NULL else all.vars(if (is.character(f))
  parse(text = f) else f), env = if (is.function(f)) environment(f) else
  parent.frame(), use.D = FALSE, cache.exp = TRUE, nderiv = NULL)
}
\arguments{
\item{f}{An expression or function to be differentiated.
f can be \itemize{
 \item a user defined function: \code{function(x) x**n}
 \item a string: \code{"x**n"}
 \item an expression: \code{expression(x**n)}
 \item a call: \code{call("^", quote(x), quote(n))}
 \item a language: \code{quote(x**n)}
 \item a right hand side of a formula: \code{~ x**n} or \code{y ~ x**n}
}}

\item{x}{An optional character vector with variable name(s) with resptect to which
\code{f} must be differentiated. If not provided (i.e. x=NULL), x is
guessed either from\ code{names(formals(f))} (if \code{f} is a function)
or from all variables in f in other cases.
To differentiate expressions including components of lists or vectors, i.e. by expressions like
\code{p[1]}, \code{theta[["alpha"]]} or \code{theta$beta}, the vector of
variables \code{x}
must be a named vector. For the cited examples, \code{x} must be given
as follows \code{c(p="1", theta="alpha", theta="beta")}. Note the repeated name \code{theta} which must be provided for every component of the list \code{theta} by which a
differerentiation is required.}

\item{env}{An environment where the symbols and functions are searched for.
Defaults to \code{parent.frame()} for \code{f} expression and to
\code{environment(f)} if \code{f} is a function. For primitive function,
it is set by default to .GlobalEnv}

\item{use.D}{An optional logical (default FALSE), indicates if base::D()
must be used for differentiation of basic expressions.}

\item{cache.exp}{An optional logical (default TRUE), indicates if
final expression must be optimized with cached subexpressions.
If enabled, repeated calculations are made only once and their
results stored in cache variables which are then reused.}

\item{nderiv}{An optional integer vector of derivative orders to calculate.
Default NULL value correspond to one differentiation. If length(nderiv)>1,
the resulting expression is a list where each component corresponds to derivative order
given in nderiv. Value 0 corresponds to the original function or expression  non
differentiated. All values must be non negative. If the entries in nderiv
are named, their names are used as names in the returned list. Otherwise
the value of nderiv component is used as a name in the resulting list.}
}
\value{
\itemize{
 \item a function if \code{f} is a function
 \item an expression if \code{f} is an expression
 \item a character string if \code{f} is a character string
 \item a language (usually a so called 'call' but may be also a symbol or just a numeric) for other types of \code{f}
}
}
\description{
Symbollic differentiation of an expression or function
}
\details{
R already contains two differentiation functions: D and deriv. D does
simple univariate differentiation.  "deriv" uses D do to multivariate
differentiation.  The output of "D" is an expression, whereas the output of
"deriv" can be an executable function.

R's existing functions have several limitations.  They can probably be fixed,
but since they are written in C, this would probably require a lot of work.
Limitations include:
\itemize{
 \item The derivatives table can't be modified at runtime, and is only available
in C.
 \item Function cannot substitute function calls.  eg:
f <- function(x, y) x + y; deriv(~f(x, x^2), "x")
}

So, here are the advantages of this implementation:

\itemize{
 \item It is entirely written in R, so would be easier to maintain.
 \item Can do multi-variate differentiation.
 \item Can differentiate function calls:
 \itemize{
   \item if the function is in the derivative table, then the chain rule
is applied.  For example, if you declared that the derivative of
sin is cos, then it would figure out how to call cos correctly.
   \item if the function is not in the derivative table (or it is anonymous),
then the function body is substituted in.
   \item these two methods can be mixed.  An entry in the derivative table
need not be self-contained -- you don't need to provide an infinite
chain of derivatives.
 }
 \item It's easy to add custom entries to the derivatives table, e.g.
  
  \code{drule[["cos"]] <- alist(x=-sin(x))}
  
  The chain rule will be automatically applied if needed.
 \item The output is an executable function, which makes it suitable
     for use in optimization problems.
 \item Compound functions (i.e. piece-wise functions based on if-else operator) can
     be differentiated (cf. examples section).
 \item in case of multiple derivatives (e.g. gradient and hessian calculation),
     caching can make calculation economies for both
}

Two work environments \code{drule} and \code{simplifications} are
exported in the package namescape.
As their names indicate, they contain tables of derivative and
simplification rules.
To see the list of defined rules do \code{ls(drule)}.
To add your own derivative rule for a function called say \code{sinpi(x)} calculating sin(pi*x), do \code{drule[["sinpi"]] <- alist(x=pi*cospi(x))}.
Here, "x" stands for the first and unique argument in \code{sinpi()} definition. For a function that might have more than one argument,
e.g. \code{log(x, base=exp(1))}, the drule entry must be a list with a named rule
per argument. See \code{drule$log} for an example to follow.
After adding \code{sinpi} you can differentiate expressions like \code{Deriv(~ sinpi(x^2), "x")}. The chain rule will automatically apply.

NB. In \code{abs()} and \code{sign()} function, singularity treatment at point 0 is left to user's care.

NB2. In Bessel functions, derivatives are calculated only by the first argument,
     not by the \code{nu} argument which is supposed to be constant.
}
\examples{

\dontrun{f <- function(x) x^2}
\dontrun{Deriv(f)}
# function (x)
# 2 * x

\dontrun{f <- function(x, y) sin(x) * cos(y)}
\dontrun{Deriv(f)}
# function (x, y)
# c(x = cos(x) * cos(y), y = -(sin(x) * sin(y)))

\dontrun{f_ <- Deriv(f)}
\dontrun{f_(3, 4)}
#              x         y
# [1,] 0.6471023 0.1068000

\dontrun{Deriv(~ f(x, y^2), "y")}
# -(2 * (y * sin(x) * sin(y^2)))

\dontrun{Deriv(quote(f(x, y^2)), c("x", "y"), cache.exp=FALSE)}
# c(x = cos(x) * cos(y^2), y = -(2 * (y * sin(x) * sin(y^2))))

\dontrun{Deriv(expression(sin(x^2) * y), "x")}
# expression(2*(x*y*cos(x^2)))

Deriv("sin(x^2) * y", "x") # differentiate only by x
"2 * (x * y * cos(x^2))"

Deriv("sin(x^2) * y", cache.exp=FALSE) # differentiate by all variables (here by x and y)
"c(x = 2 * (x * y * cos(x^2)), y = sin(x^2))"

# Compound function example (here abs(x) smoothed near 0)
fc <- function(x, h=0.1) if (abs(x) < h) 0.5*h*(x/h)**2 else abs(x)-0.5*h
Deriv("fc(x)", "x", cache.exp=FALSE)
"if (abs(x) < h) x/h else sign(x)"

# Example of a first argument that cannot be evaluated in the current environment:
\dontrun{
  suppressWarnings(rm("xx", "yy"))
  Deriv(xx^2+yy^2)
}
# c(xx = 2 * xx, yy = 2 * yy)

# Automatic differentiation (AD), note itermediate variable 'd' assignment
\dontrun{Deriv(~{d <- ((x-m)/s)^2; exp(-0.5*d)}, "x")}
#{
#   d <- ((x - m)/s)^2
#   .d_x <- 2 * ((x - m)/s^2)
#   -(0.5 * (.d_x * exp(-(0.5 * d))))
#}

# Custom derivation rule
\dontrun{
  myfun <- function(x, y=TRUE) NULL # do something usefull
  dmyfun <- function(x, y=TRUE) NULL # myfun derivative by x.
  drule[["myfun"]] <- alist(x=dmyfun(x, y), y=NULL) # y is just a logical
  Deriv(myfun(z^2, FALSE), "z")
  # 2 * (z * dmyfun(z^2, FALSE))
}
# Differentiantion by list components
\dontrun{
  theta <- list(m=0.1, sd=2.)
  x <- names(theta)
  names(x)=rep("theta", length(theta))
  Deriv(~exp(-(x-theta$m)**2/(2*theta$sd)), x, cache.exp=FALSE)
# c(theta_m = exp(-((x - theta$m)^2/(2 * theta$sd))) *
#  (x - theta$m)/theta$sd, theta_sd = 2 * (exp(-((x - theta$m)^2/
#  (2 * theta$sd))) * (x - theta$m)^2/(2 * theta$sd)^2))
}
}
\author{
Andrew Clausen (original version) and Serguei Sokol (maintainer)
}
\concept{
symbollic differentiation
}

