#' Compute and Visualize Code Saturation
#'
#' This function summarizes code counts and their proportional representation
#' across media titles (e.g., interviews, focus groups, or other qualitative data sources).
#' It can optionally produce a formatted table and/or a ggplot visualization
#' showing saturation by code frequency or proportion.
#'
#' @param code_counts A tibble or data frame containing columns:
#'   - `code`: the code label
#'   - `count`: total number of excerpts coded with that code
#'   - `n_media_titles`: number of distinct media titles (e.g., transcripts) in which the code appears.
#'   This object is typically generated by `create_code_summary()`.
#' @param total_media_titles Optional numeric value indicating the total number of media titles.
#'   If `NULL` (default), the function uses the maximum value of `n_media_titles`.
#' @param table_min_count Minimum count threshold for including a code in the output table.
#'   Defaults to 1.
#' @param table_min_prop Minimum proportion threshold (relative to total media titles)
#'   for including a code in the output table. Defaults to `NULL` (no proportion filter).
#' @param output_type Character string indicating the output format for the table:
#'   either `"tibble"` (default) or `"kable"`.
#' @param plot Logical; if `TRUE`, produces a ggplot visualization.
#'   Defaults to `FALSE`.
#' @param plot_min_count Minimum count threshold for codes to include in the plot.
#'   Defaults to `table_min_count` if `NULL`.
#' @param plot_min_prop Minimum proportion threshold for codes to include in the plot.
#'   Defaults to `table_min_prop` if `NULL`.
#' @param plot_metric Character string indicating what to plot:
#'   `"prop"` for proportions, `"count"` for counts, or `"both"` for dual-axis plot.
#'   Defaults to `"prop"`.
#' @param fill_color Character string specifying the fill color for bars in the plot.
#'   Defaults to `"steelblue"`.
#'
#' @return
#' - If `plot = FALSE`: returns a tibble (or kable table) summarizing code frequencies
#'   and proportions.
#' - If `plot = TRUE`: returns a list with two elements:
#'   - `table`: the filtered tibble
#'   - `plot`: a ggplot2 object.
#'
#' @examples
#' # Example dataset
#' code_counts <- tibble::tibble(
#'   code = c("Belonging", "Resilience", "Stress", "Hope"),
#'   count = c(15, 10, 8, 5),
#'   n_media_titles = c(8, 6, 5, 3)
#' )
#'
#' # Basic usage (returns a tibble)
#' set_saturation(code_counts)
#'
#' # Apply count and proportion filters, return a kable table
#' set_saturation(
#'   code_counts,
#'   total_media_titles = 10,
#'   table_min_count = 5,
#'   table_min_prop = 0.3,
#'   output_type = "kable"
#' )
#'
#' # Generate a plot of proportions
#' res <- set_saturation(
#'   code_counts,
#'   total_media_titles = 10,
#'   plot = TRUE,
#'   plot_metric = "prop"
#' )
#' res$plot
#'
#' # Plot both count and proportion using dual y-axes
#' res <- set_saturation(
#'   code_counts,
#'   total_media_titles = 10,
#'   plot = TRUE,
#'   plot_metric = "both",
#'   fill_color = "darkgreen"
#' )
#' res$plot
#'
#' @export
set_saturation <- function(code_counts,
                           total_media_titles = NULL,
                           table_min_count = 1,
                           table_min_prop = NULL,
                           output_type = c("tibble", "kable"),
                           plot = FALSE,
                           plot_min_count = NULL,
                           plot_min_prop = NULL,
                           plot_metric = c("prop", "count", "both"),
                           fill_color = "steelblue") {
  output_type <- match.arg(output_type)
  plot_metric <- match.arg(plot_metric)

  if (is.list(code_counts) && "table" %in% names(code_counts)) {
    code_counts <- code_counts$table
  }

  if (!is.data.frame(code_counts)) {
    stop("`code_counts` must be a tibble or data frame (from create_code_summary()).")
  }
  if (!all(c("code", "count", "n_media_titles") %in% names(code_counts))) {
    stop("`code_counts` must contain columns `code`, `count`, and `n_media_titles`.")
  }

  if (is.null(total_media_titles)) {
    total_media_titles <- max(code_counts$n_media_titles, na.rm = TRUE)
  }

  # --- Compute proportions and filter for table ---
  df <- code_counts %>%
    dplyr::filter(count >= table_min_count) %>%
    dplyr::mutate(
      prop_media_titles = round(n_media_titles / total_media_titles, 2)
    ) %>%
    dplyr::select("code", "count", "prop_media_titles")

  if (!is.null(table_min_prop)) {
    df <- df %>% dplyr::filter(prop_media_titles >= table_min_prop)
  }

  df <- df %>% dplyr::arrange(dplyr::desc(count))

  caption_text <- paste(
    "Code Counts with Transcript Proportions (table_min_count =", table_min_count,
    ", table_min_prop =",
    ifelse(is.null(table_min_prop), "none", table_min_prop),
    ")"
  )

  table_out <- if (output_type == "kable") {
    knitr::kable(df, caption = caption_text, digits = 2)
  } else {
    df
  }

  # --- Plot output ---
  if (plot) {
    if (is.null(plot_min_count)) plot_min_count <- table_min_count
    if (is.null(plot_min_prop)) plot_min_prop <- table_min_prop

    plot_df <- df %>% dplyr::filter(count >= plot_min_count)
    if (!is.null(plot_min_prop)) {
      plot_df <- plot_df %>% dplyr::filter(prop_media_titles >= plot_min_prop)
    }

    plot_df <- plot_df %>% dplyr::arrange(prop_media_titles)

    if (plot_metric == "prop") {
      p <- ggplot2::ggplot(plot_df, ggplot2::aes(
        x = reorder(code, prop_media_titles),
        y = prop_media_titles
      )) +
        ggplot2::geom_col(fill = fill_color) +
        ggplot2::coord_flip() +
        ggplot2::labs(
          x = "Code",
          y = "Proportion of Media Titles",
          title = "Proportion of Media Titles"
        ) +
        ggplot2::theme_minimal()

    } else if (plot_metric == "count") {
      p <- ggplot2::ggplot(plot_df, ggplot2::aes(
        x = reorder(code, count),
        y = count
      )) +
        ggplot2::geom_col(fill = fill_color) +
        ggplot2::coord_flip() +
        ggplot2::labs(
          x = "Code",
          y = "Excerpt Frequency",
          title = "Code Counts"
        ) +
        ggplot2::theme_minimal()

    } else if (plot_metric == "both") {
      scale_factor <- max(plot_df$count, na.rm = TRUE) / max(plot_df$prop_media_titles, na.rm = TRUE)

      p <- ggplot2::ggplot(plot_df, ggplot2::aes(
        x = reorder(code, count),
        y = count
      )) +
        ggplot2::geom_col(fill = fill_color) +
        ggplot2::coord_flip() +
        ggplot2::scale_y_continuous(
          name = "Excerpt Frequency",
          sec.axis = ggplot2::sec_axis(~ . / scale_factor,
                                       name = "Proportion of Media Titles")
        ) +
        ggplot2::labs(
          x = "Code",
          title = "Code Saturation"
        ) +
        ggplot2::theme_minimal()
    }

    return(list(table = df, plot = p))
  }

  return(table_out)
}
