% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DAISIE_sim_cr.R
\name{DAISIE_sim}
\alias{DAISIE_sim}
\alias{DAISIE_sim_cr}
\title{Simulate (non-)oceanic islands with given parameters under
  time-constant rates}
\usage{
DAISIE_sim_cr(
  time,
  M,
  pars,
  replicates,
  divdepmodel = "CS",
  nonoceanic_pars = c(0, 0),
  num_guilds = NULL,
  prop_type2_pool = NA,
  replicates_apply_type2 = TRUE,
  sample_freq = 25,
  plot_sims = TRUE,
  hyper_pars = create_hyper_pars(d = 0, x = 0),
  area_pars = create_area_pars(max_area = 1, current_area = 1, proportional_peak_t = 0,
    total_island_age = 0, sea_level_amplitude = 0, sea_level_frequency = 0,
    island_gradient_angle = 0),
  cond = 0,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{time}{Numeric defining the length of the simulation in time units.
For example, if an island is known to be 4 million years old, setting
time = 4 will simulate the entire life span of the island; setting time = 2
will stop the simulation at the mid-life of the island.}

\item{M}{Numeric defining the size of mainland pool, i.e. the number of
species that can potentially colonize the island.}

\item{pars}{A numeric vector containing the model parameters:
\itemize{
  \item{\code{pars[1]}: lambda^c (cladogenesis rate)}
  \item{\code{pars[2]}: mu (extinction rate)}
  \item{\code{pars[3]}: K (carrying capacity), set K=Inf for diversity
  independence.}
  \item{\code{pars[4]}: gamma (immigration rate)}
  \item{\code{pars[5]}: lambda^a (anagenesis rate)}
  \item{\code{pars[6]}: lambda^c (cladogenesis rate) for either type 2 species
  or rate set 2 in rate shift model}
  \item{\code{pars[7]}: mu (extinction rate) for either type 2 species or rate
  set 2 in rate shift model}
  \item{\code{pars[8]}: K (carrying capacity) for either type 2 species or rate
  set 2 in rate shift model, set K=Inf for diversity independence.}
  \item{\code{pars[9]}: gamma (immigration rate) for either type 2 species
  or rate set 2 in rate shift model}
  \item{\code{pars[10]}: lambda^a (anagenesis rate) for either type 2
  species or rate set 2 in rate shift model}
}
Elements 6:10 are required only when type 2 species are included
or in the rate shift model. For \code{\link{DAISIE_sim_relaxed_rate}()}
\code{pars[6]} is the standard deviation of the gamma distribution for the
relaxed parameter and the parameter chosen by the \code{relaxed_par}
argument is the mean of the gamma distribution for the relaxed parameter.}

\item{replicates}{Integer specifying number of island replicates to be
simulated.}

\item{divdepmodel}{Option divdepmodel = 'CS' runs a model with clade-specific
carrying capacity, where diversity-dependence operates only within single
clades, i.e. only among species originating from the same mainland
colonist. Option divdepmodel = 'IW' runs a model with island-wide
carrying capacity, where diversity-dependence operates within and among
clades. Option divdepmodel = 'GW' runs a model with diversity-dependence
operates within a guild.}

\item{nonoceanic_pars}{A vector of length two with:
\itemize{
  \item{[1]: the probability of sampling a species from the mainland}
  \item{[2]: the probability of the species sampled from the mainland
  being nonendemic}
}}

\item{num_guilds}{The number of guilds on the mainland. The number of
mainland species is divided by the number of guilds when
\code{divdepmodel = "GW"}}

\item{prop_type2_pool}{Fraction of mainland species that belongs to the
second subset of species (type 2). Applies only when two types of species
are simulated (length(pars) = 10). For \code{\link{DAISIE_dataprep}()}
applies only if number_clade_types = 2.  In \code{\link{DAISIE_dataprep}()}
the default \code{"proportional"} sets the fraction to be proportional to
the number of clades of distinct macroevolutionary process that have
colonised the island.}

\item{replicates_apply_type2}{Applies only when two types of species are
being simulated. Default replicates_apply_type2 = TRUE runs simulations
until the number of islands where a type 2 species has colonised is equal
to the specified number of replicates. This is recommended if
\code{prop_type2_pool} is small or if the rate of immigration of type two
species (\code{pars[9]}) is low, meaning that more replicates are needed to
achieved an adequate sample size of islands with type 2 species. Setting
\code{replicates_apply_type2 = FALSE} simulates islands up to the
specified number of replicates regardless of whether type 2 species have
colonised or not.}

\item{sample_freq}{Numeric specifing the number of units times should be
divided by for plotting purposes. Larger values will lead to plots with
higher resolution, but will also run slower.}

\item{plot_sims}{\code{Default = TRUE} plots species-through-time (STT)
plots. It detects how many types of species are present. If only one type
of species is present, STT is plotted for all species. If two types are
present, three plots are produced: STT for all, STT for type 1 and STT for
type 2.}

\item{hyper_pars}{A named list of numeric hyperparameters for the rate
calculations as returned by \code{\link{create_hyper_pars}()}:
\itemize{
  \item{[1]: is d the scaling parameter for exponent for calculating
  cladogenesis rate}
  \item{[2]: is x the exponent for calculating extinction rate}
}}

\item{area_pars}{A named list containing area and sea level parameters as
created by \code{\link{create_area_pars}()}:
\itemize{
  \item{[1]: maximum area}
  \item{[2]: current area}
  \item{[3]: value from 0 to 1 indicating where in the island's history the
  peak area is achieved}
  \item{[4]: total island age}
  \item{[5]: amplitude of area fluctuation from sea level}
  \item{[6]: frequency of sine wave of area change from sea level}
  \item{[7]: angle of the slope of the island}
}}

\item{cond}{cond = 0 : conditioning on island age \cr cond = 1 :
conditioning on island age and non-extinction of the island biota \cr.
cond > 1 : conditioning on island age and having at least cond colonizations
on the island. This last option is not yet available for the IW model \cr}

\item{verbose}{In simulation and dataprep functions a logical,
\code{Default = TRUE} gives intermediate output should be printed.
For ML functions a numeric determining if intermediate output should be
printed, \code{Default = 0} does not print, \code{verbose = 1} prints
intermediate output of the parameters and loglikelihood, \code{verbose = 2}
means also intermediate progress during loglikelihood computation is shown.}

\item{...}{Any arguments to pass on to plotting functions.}
}
\value{
A list. The highest level of the least corresponds to each individual
replciate. The first element of each replicate is composed of island
information containing:
\itemize{
  \item{\code{$island_age}: A numeric with the island age.}
  \item{\code{$not_present}: the number of mainland lineages that are not
    present on the island. It is only present if only 1 type of species is
    simulated. Becomes \code{$not_present_type1}: the number of mainland
    lineages of type 1 that are not present on the island and
    \code{$not_present_type2}: the number of mainland lineages of type 2
    that are not present on the island, if two types are simulated.}
  \item{\code{$stt_all}: STT table for all species on the island
    (nI - number of non-endemic species; nA - number of anagenetic species,
    nC - number of cladogenetic species, present - number of independent
    colonisations present)}
  \item{\code{$stt_stt_type1}: STT table for type 1 species on the island -
    only if 2 types of species were simulated (nI - number of non-endemic
    species; nA - number of anagenetic species, nC - number of cladogenetic
    species, present - number of independent colonisations present).}
  \item{\code{$stt_stt_type2}: STT table for type 2 species on the island
     - only if 2 types of species were simulated (nI - number of non-endemic
     species; nA - number of anagenetic species, nC - number of cladogenetic
     species, present - number of independent colonisations present ).}
  \item{\code{$brts_table}: Only for simulations under \code{"IW"}. Table
containing information on order of events in the data, for use in maximum
likelihood optimization.).}
}
The subsequent elements of the list pertaining to each replcate contain
information on a single colonist lineage on the island and have 4 components:
\itemize{
  \item{\code{$branching_times}: island age and stem age of the
    population/species in the case of Non-endemic, Non-endemic_MaxAge and
    Endemic anagenetic species.

    For cladogenetic species these should
    be island age and branching times of the radiation including the
    stem age of the radiation.}
  \item{\code{$stac}: An integer ranging from 1 to 4
  indicating the status of the colonist:}
  \enumerate{
    \item Non_endemic_MaxAge
    \item Endemic
    \item Endemic&Non_Endemic
    \item Non_endemic_MaxAge
}
\item{\code{$missing_species}: number of island species that were
not sampled for particular clade (only applicable for endemic clades)}
\item{\code{$type_1or2}: whether the colonist belongs to type 1 or type 2}
}
}
\description{
This function simulates islands with given cladogenesis,
extinction, Kprime, immigration and anagenesis parameters, all of which
modelled as time-constant parameters. If a single
parameter set is provided (5 parameters) it simulates islands where all
species have the same macro-evolutionary process. If two paramater sets
(10 parameters) are provided, it simulates islands where two different
macro-evolutionary processes operate, one applying to type 1 species and
other to type 2 species. Further, it allows for the simulation of
non-oceanic islands, generating islands for which the starting condition
includes potential endemic and non-endemic species.
}
\examples{
## Simulate 2 islands for 1 million years, where all species have equal
## rates. Pool size 100.

clado_rate <- 0.5
ext_rate <- 0.2
carr_cap <- Inf
immig_rate <- 0.05
ana_rate <- 1
sim_pars <- c(clado_rate, ext_rate, carr_cap, immig_rate, ana_rate)
set.seed(1)
island_replicates <- DAISIE_sim_cr(
  time = 1,
  M = 100,
  pars = sim_pars,
  replicates = 2,
  plot_sims = FALSE,
  verbose = FALSE
)

## Simulate 2 islands for 1 million years with two types of species (type1
## and type 2). Pool size 100
## Fraction of type 2 species in source pool is 0.15. Function will
## simulate until number of islands where type 2 species has colonised is
## equal to number specified in replicates.

clado_rate <- 0.5
ext_rate <- 0.2
carr_cap <- Inf
immig_rate <- 0.005
ana_rate <- 1
sim_pars_type1 <- c(clado_rate, ext_rate, carr_cap, immig_rate, ana_rate)
sim_pars_type2 <- sim_pars_type1 * 2
set.seed(1)
island_replicates_2types <- DAISIE_sim_cr(
  time = 1,
  M = 100,
  pars = c(sim_pars_type1, sim_pars_type2),
  replicates = 2,
  prop_type2_pool = 0.15,
  plot_sims = FALSE,
  verbose = FALSE
)
## Simulate two non-oceanic island for 1 million years.
## Pool size 500. Island area as a proportion
## of mainland is 0.1, proportion of native species is 0.9.
clado_rate <- 0.5
ext_rate <- 0.2
carr_cap <- Inf
immig_rate <- 0.005
ana_rate <- 1
sim_pars <- c(clado_rate, ext_rate, carr_cap, immig_rate, ana_rate)
set.seed(1)
island_replicates <- DAISIE_sim_cr(
  time = 1,
  M = 500,
  pars = sim_pars,
  replicates = 2,
  nonoceanic_pars = c(0.1, 0.9),
  plot_sims = FALSE,
  verbose = FALSE
)

## Simulate 2 islands for 1 million years with a shift in immigration rate
## at 0.195 Ma, and plot the species-through-time plot. Pool size 296.

pars_before_shift <- c(0.079, 0.973, Inf, 0.136, 0.413)
pars_after_shift <- c(0.079, 0.973, Inf, 0.652, 0.413)
tshift <- 0.195
set.seed(1)
island_shift_replicates <- DAISIE_sim_cr_shift(
  time = 1,
  M = 296,
  pars = c(pars_before_shift, pars_after_shift),
  replicates = 2,
  shift_times = tshift,
  plot_sims = FALSE,
  verbose = FALSE
)
}
\references{
Valente, L.M., A.B. Phillimore and R.S. Etienne (2015).
Equilibrium and non-equilibrium dynamics simultaneously operate in the
Galapagos islands. Ecology Letters 18: 844-852.
}
\seealso{
\code{\link{DAISIE_plot_sims}()} for plotting STT of simulation
outputs.

Other simulation models: 
\code{\link{DAISIE_sim_cr_shift}()},
\code{\link{DAISIE_sim_relaxed_rate}()},
\code{\link{DAISIE_sim_time_dep}()}
}
\author{
Luis Valente, Albert Phillimore, Joshua Lambert, Shu Xie, Pedro
Neves, Richèl J. C. Bilderbeek, Rampal Etienne
}
\concept{simulation models}
\keyword{models}
