% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/map_stochastic_changes.R
\name{map_stochastic_changes}
\alias{map_stochastic_changes}
\title{Finds all state changes on a tree using stochastic character mapping}
\usage{
map_stochastic_changes(
  cladistic_matrix,
  time_tree,
  time_bins,
  n_simulations = 10,
  polymorphism_behaviour = "equalp",
  uncertainty_behaviour = "equalp",
  inapplicable_behaviour = "missing"
)
}
\arguments{
\item{cladistic_matrix}{A character-taxon matrix in the format imported by \link{read_nexus_matrix}.}

\item{time_tree}{A time-scaled tree (phylo object) that represents the relationships of the taxa in \code{cladistic_matrix}.}

\item{time_bins}{A vector of ages representing the boundaries of a series of time bins.}

\item{n_simulations}{The number of simulations to perform (passed to \link{make.simmap}.}

\item{polymorphism_behaviour}{What to do with polymorphic (&) characters. One of "equalp", "missing", or "random". See details.}

\item{uncertainty_behaviour}{What to do with uncertain (/) characters. One of "equalp", "missing", or "random". See details.}

\item{inapplicable_behaviour}{What to do with inapplicable characters. Only one option currently ("missing"). See details.}
}
\value{
\item{all_state_changes}{A matrix of rows for each change with columns corresponding to the character, the simulation number, the edge number, the time the change occurred, and the start and end states.}
\item{character_times}{A vector of the sampled tree-length (in Ma) for each character.}
\item{binned_edge_lengths}{A matrix of time bins (columns) and characters (rows) indicating the sampled tree-length (in Ma).}
\item{binned_terminal_edge_lengths}{As above, but for terminal edges only.}
\item{binned_internal_edge_lengths}{As above, but for internal edges only.}
}
\description{
Takes a cladistic matrix and time-scaled tree and makes point estimates for every character change using stochastic character mapping.
}
\details{
Important: this function is not yet complete and should not be used.

A wrapper function for \link{make.simmap} in the \link{phytools} package.

This function is intended to enumerate all possible changes on a tree (including to and from missing or inapplicable states) under the assumptions of stochastic character mapping as an alternative means of establishing branch-lengths (for rate analyses) or recording the state occupied at a particular point in time for disparity analyses.
}
\examples{

# Set random seed:
set.seed(2)

# Use Day 2016 as source matrix:
cladistic_matrix <- day_2016

# Prune out continuous characters:
cladistic_matrix <- prune_cladistic_matrix(
  cladistic_matrix =
    cladistic_matrix, blocks2prune = 1
)

# Prune out majority of characters so
# example runs quickly:
cladistic_matrix <- prune_cladistic_matrix(
  cladistic_matrix =
    cladistic_matrix, characters2prune = 1:32
)

# Generete random tree for matrix taxa:
time_tree <- ape::rtree(n = nrow(day_2016$matrix_1$matrix))

# Add taxon names to tree:
time_tree$tip.label <- rownames(x = day_2016$matrix_1$matrix)

# Add root age to tree:
time_tree$root.time <- max(diag(x = ape::vcv(phy = time_tree)))

# Get all state changes for two simulations:
state_changes <-
  map_stochastic_changes(
    cladistic_matrix = cladistic_matrix,
    time_tree = time_tree, time_bins = seq(time_tree$root.time, 0,
      length.out = 3
    ), n_simulations = 2
  )

# View matrix of all stochstic character changes:
state_changes$all_state_changes

# View vector of sampled time for each character:
state_changes$character_times

# View matrix of edge lengths in each time bin:
state_changes$binned_edge_lengths

# View matrix of terminal edge lengths in each time bin:
state_changes$binned_terminal_edge_lengths

# View matrix of internal edge lengths in each time bin:
state_changes$binned_internal_edge_lengths
}
\author{
Graeme T. Lloyd \email{graemetlloyd@gmail.com}
}
