% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CST_Analogs.R
\name{CST_Analogs}
\alias{CST_Analogs}
\title{Downscaling using Analogs based on large scale fields.}
\usage{
CST_Analogs(
  expL,
  obsL,
  expVar = NULL,
  obsVar = NULL,
  region = NULL,
  criteria = "Large_dist",
  excludeTime = NULL,
  time_expL = NULL,
  time_obsL = NULL,
  nAnalogs = NULL,
  AnalogsInfo = FALSE,
  ncores = NULL
)
}
\arguments{
\item{expL}{An 's2dv_cube' object containing the experimental field on the 
large scale for which the analog is aimed. This field is used to in all the
criterias. If parameter 'expVar' is not provided, the function will return 
the expL analog. The element 'data' in the 's2dv_cube' object must have, at
least, latitudinal and longitudinal dimensions. The object is expect to be 
already subset for the desired large scale region. Latitudinal dimension 
accepted names: 'lat', 'lats', 'latitude', 'y', 'j', 'nav_lat'. Longitudinal 
dimension accepted names: 'lon', 'lons','longitude', 'x', 'i', 'nav_lon'.}

\item{obsL}{An 's2dv_cube' object containing the observational field on the
large scale. The element 'data' in the 's2dv_cube' object must have the same 
latitudinal and longitudinal dimensions as parameter 'expL' and a temporal 
dimension with the maximum number of available observations.}

\item{expVar}{An 's2dv_cube' object containing the experimental field on the
local scale, usually a different variable to the parameter 'expL'. If it is 
not NULL (by default, NULL), the returned field by this function will be the 
analog of parameter 'expVar'.}

\item{obsVar}{An 's2dv_cube' containing the field of the same variable as the 
passed in parameter 'expVar' for the same region.}

\item{region}{A vector of length four indicating the minimum longitude, 
the maximum longitude, the minimum latitude and the maximum latitude.}

\item{criteria}{A character string indicating the criteria to be used for the
selection of analogs:
\itemize{
\item{Large_dist} minimum Euclidean distance in the large scale pattern; 
\item{Local_dist} minimum Euclidean distance in the large scale pattern 
and minimum Euclidean distance in the local scale pattern; and
\item{Local_cor} minimum Euclidean distance in the large scale pattern, 
minimum Euclidean distance in the local scale pattern and highest 
correlation in the local variable to downscale.}
Criteria 'Large_dist' is recommended for CST_Analogs, for an advanced use of 
the criterias 'Local_dist' and 'Local_cor' use 'Analogs' function.}

\item{excludeTime}{An array of N named dimensions (coinciding with time 
dimensions in expL)of character string(s) indicating the date(s) of the 
observations in the format "dd/mm/yyyy" to be excluded during the search of 
analogs. It can be NULL but if expL is not a forecast (time_expL contained in
time_obsL), by default time_expL will be removed during the search of analogs.}

\item{time_expL}{A character string indicating the date of the experiment 
in the same format than time_obsL (i.e. "yyyy-mm-dd"). By default it is NULL 
and dates are taken from element \code{$attrs$Dates} from expL.}

\item{time_obsL}{A character string indicating the date of the observations 
in the date format (i.e. "yyyy-mm-dd"). By default it is NULL and dates are 
taken from element \code{$attrs$Dates} from obsL.}

\item{nAnalogs}{Number of Analogs to be selected to apply the criterias 
'Local_dist' or 'Local_cor'. This is not the necessary the number of analogs 
that the user can get, but the number of events with minimum distance in 
which perform the search of the best Analog. The default value for the 
'Large_dist' criteria is 1, for 'Local_dist' and 'Local_cor' criterias must
be greater than 1 in order to match with the first criteria, if nAnalogs is
NULL for 'Local_dist' and 'Local_cor' the default value will be set at the 
length of 'time_obsL'. If AnalogsInfo is FALSE the function returns just 
the best analog.}

\item{AnalogsInfo}{A logical value. TRUE to get a list with two elements: 
1) the downscaled field and 2) the AnalogsInfo which contains: 
a) the number of the best analogs, b) the corresponding value of the metric 
used in the selected criteria (distance values for Large_dist and Local_dist, 
correlation values for Local_cor), c)dates of the analogs). The analogs are 
listed in decreasing order, the first one is the best analog (i.e if the 
selected criteria is Local_cor the best analog will be the one with highest 
correlation, while for Large_dist criteria the best analog will be the day 
with minimum Euclidean distance). Set to FALSE to get a single analog, the 
best analog, for instance for downscaling.}

\item{ncores}{The number of cores to use in parallel computation}
}
\value{
An 's2dv_cube' object containing an array with the dowscaled values of 
the best analogs in element 'data'. If 'AnalogsInfo' is TRUE, 'data' is a list 
with an array of the downscaled fields and the analogs information in 
elements 'analogs', 'metric' and 'dates'.
}
\description{
This function perform a downscaling using Analogs. To compute 
the analogs, the function search for days with similar large scale conditions
to downscaled fields to a local scale. The large scale and the local scale 
regions are defined by the user. The large scale is usually given by 
atmospheric circulation as sea level pressure or geopotential height 
(Yiou et al, 2013) but the function gives the possibility to use another 
field. The local scale will be usually given by precipitation or temperature 
fields, but might be another variable.The analogs function will find the best
analogs based in Minimum Euclidean distance in the large scale pattern
(i.e.SLP).

The search of analogs must be done in the longest dataset posible. This is 
important since it is necessary to have a good representation of the 
possible states of the field in the past, and therefore, to get better 
analogs. 
This function has not constrains of specific regions, variables to downscale,
or data to be used (seasonal forecast data, climate projections data, 
reanalyses data). The regrid into a finner scale is done interpolating with 
CST_Load. Then, this interpolation is corrected selecting the analogs in the 
large and local scale in based of the observations. The function is an 
adapted version of the method of Yiou et al 2013. For an advanced search of 
Analogs (multiple Analogs, different criterias, further information from the
metrics and date of the selected Analogs) use the'Analog'
function within 'CSTools' package.
}
\examples{
expL <- rnorm(1:200)
dim(expL) <- c(member = 10, lat = 4, lon = 5)
obsL <- c(rnorm(1:180), expL[1, , ]*1.2)
dim(obsL) <- c(time = 10, lat = 4, lon = 5)
time_obsL <- as.POSIXct(paste(rep("01", 10), rep("01", 10), 1994:2003, sep = "-"), 
                       format = "\%d-\%m-\%y")
dim(time_obsL) <- c(time = 10)
time_expL <- time_obsL[1]
lon <-  seq(-1, 5, 1.5)
lat <- seq(30, 35, 1.5)
coords <- list(lon = seq(-1, 5, 1.5), lat = seq(30, 35, 1.5))
attrs_expL <- list(Dates = time_expL)
attrs_obsL <- list(Dates = time_obsL)
expL <- list(data = expL, coords = coords, attrs = attrs_expL)
obsL <- list(data = obsL, coords = coords, attrs = attrs_obsL)
class(expL) <- 's2dv_cube'
class(obsL) <- 's2dv_cube'
region <- c(min(lon), max(lon), min(lat), max(lat))
downscaled_field <- CST_Analogs(expL = expL, obsL = obsL, region = region)

}
\references{
Yiou, P., T. Salameh, P. Drobinski, L. Menut, R. Vautard,
and M. Vrac, 2013 : Ensemble reconstruction of the atmospheric column 
from surface pressure using analogues.  Clim. Dyn., 41, 1419-1437. 
\email{pascal.yiou@lsce.ipsl.fr}
}
\seealso{
\code{\link{CST_Load}}, \code{\link[s2dv]{Load}} and 
\code{\link[s2dv]{CDORemap}}
}
\author{
M. Carmen Alvarez-Castro, \email{carmen.alvarez-castro@cmcc.it}

Maria M. Chaves-Montero, \email{mariadm.chaves@cmcc.it}

Veronica Torralba, \email{veronica.torralba@cmcc.it}

Nuria Perez-Zanon \email{nuria.perez@bsc.es}
}
