% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/modelKriging.R
\name{modelKriging}
\alias{modelKriging}
\title{Kriging Model}
\usage{
modelKriging(x, y, distanceFunction, control = list())
}
\arguments{
\item{x}{list of samples in input space}

\item{y}{column vector of observations for each sample}

\item{distanceFunction}{a suitable distance function of type f(x1,x2), returning a scalar distance value, preferably between 0 and 1.
Maximum distances larger 1 are no problem, but may yield scaling bias when different measures are compared.
   Should be non-negative and symmetric.  It can also be a list of several distance functions. In this case, Maximum Likelihood Estimation (MLE) is used
   to determine the most suited distance measure.}

\item{control}{(list), with the options for the model building procedure:\cr
\code{lower} lower boundary for theta, default is \code{1e-6}\cr
\code{upper} upper boundary for theta, default is \code{100}\cr
\code{corr} function to be used for correlation modelling, default is \code{fcorrGauss}\cr
\code{algTheta}  algorithm used to find theta (as well as p and lambda), default is \code{\link{optimInterface}}.\cr
\code{algThetaControl}  list of controls passed to \code{algTheta}.\cr
\code{optimizeP} boolean that specifies whether the exponents (\code{p}) should be optimized. Else they will be set to two. \cr
\code{useLambda} whether or not to use the regularization constant lambda (nugget effect). Default is \code{FALSE}.\cr
\code{lambdaLower} lower boundary for lambda, default is \code{-6}\cr
\code{lambdaUpper} upper boundary for lambda, default is \code{0}\cr
\code{distances} a distance matrix. If available, this matrix is used for model building, instead of calculating the distance matrix using the parameters \code{distanceFunction}. Default is \code{NULL}.}
}
\value{
an object of class \code{modelKriging} containing the options and determined parameters for the model:\cr
\code{x} (see parameters)\cr
\code{y} (see parameters)\cr
\code{lower} (see parameters)\cr
\code{upper} (see parameters)\cr
\code{algTheta} (see parameters)\cr
\code{algThetaControl} (see parameters)\cr
\code{optimizeP} (see parameters)\cr
\code{theta} activity or width parameter theta, a parameter of the correlation function determined with MLE\cr
\code{log10Theta} log10 \code{theta} (i.e. \code{log10(theta)})\cr
\code{lambda} regularization constant (nugget) lambda \cr
\code{log10Lambda} log10 of regularization constant (nugget) lambda (i.e. \code{log10(lambda)})\cr
\code{p} exponent p, parameter of the correlation function determined with MLE (if \code{optimizeP} is \code{TRUE})\cr
\code{yMu} vector of observations y, minus MLE of mu\cr
\code{SSQ} Maximum Likelihood Estimate (MLE) of model parameter sigma^2\cr
\code{mu} MLE of model parameter mu\cr
\code{Psi} correlation matrix Psi\cr
\code{Psinv} inverse of Psi\cr
\code{nevals} number of Likelihood evaluations during MLE of theta/lambda/p\cr
\code{scaling} Default is FALSE. If TRUE: Distances values are divided by maximum distance to independent of the scale of the distance function.\cr
\code{distanceFunctionIndexMLE} If a list of several distance measures (\code{distanceFunction}) was given, this parameter contains the index value of the measure chosen with MLE.
}
\description{
Implementation of a distance-based Kriging model, e.g., for mixed or combinatorial input spaces.
It is based on employing suitable distance measures for the samples in input space.
}
\details{
The basic Kriging implementation is based on the work of Forrester et al. (2008).
For adaptation of Kriging to mixed or combinatorial spaces, as well as
choosing distance measures with Maximum Likelihood Estimation, see the other two references (Zaefferer et al., 2014).
}
\examples{
#Set random number generator seed
set.seed(1)
#Simple test landscape
fn <- landscapeGeneratorUNI(1:5,distancePermutationHamming)
#Generate data for training and test
x <- unique(replicate(40,sample(5),FALSE))
xtest <- x[-(1:15)]
x <- x[1:15]
#Determin true objective function values
y <- fn(x)
ytest <- fn(xtest)
#Build model
fit <- modelKriging(x,y,distancePermutationHamming,
    control=list(algThetaControl=list(method="L-BFGS-B"),useLambda=FALSE))
#Predicted obj. function values
ypred <- predict(fit,xtest)$y
#Uncertainty estimate
fit$predAll <- TRUE
spred <- predict(fit,xtest)$s
#Plot
plot(ytest,ypred,xlab="true value",ylab="predicted value",
    pch=20,xlim=c(0.3,1),ylim=c(min(ypred)-0.1,max(ypred)+0.1))
segments(ytest, ypred-spred,ytest, ypred+spred)
epsilon = 0.02
segments(ytest-epsilon,ypred-spred,ytest+epsilon,ypred-spred)
segments(ytest-epsilon,ypred+spred,ytest+epsilon,ypred+spred)
abline(0,1,lty=2)
#Use a different/custom optimizer (here: SANN) for maximum likelihood estimation:
#(Note: Bound constraints are recommended, to avoid Inf values.
#This is really just a demonstration. SANN does not respect bound constraints.)
optimizer1 <- function(x,fun,lower=NULL,upper=NULL,control=NULL,...){
  res <- optim(x,fun,method="SANN",control=list(maxit=100),...)
  list(xbest=res$par,ybest=res$value,count=res$counts)
}
fit <- modelKriging(x,y,distancePermutationHamming,
                   control=list(algTheta=optimizer1,useLambda=FALSE))
#One-dimensional optimizer (Brent). Note, that Brent will not work when
#several parameters have to be set, e.g., when using nugget effect (lambda).
#However, Brent may be quite efficient otherwise.
optimizer2 <- function(x,fun,lower,upper,control=NULL,...){
 res <- optim(x,fun,method="Brent",lower=lower,upper=upper,...)
 list(xbest=res$par,ybest=res$value,count=res$counts)
}
fit <- modelKriging(x,y,distancePermutationHamming,
                    control=list(algTheta=optimizer2,useLambda=FALSE))
}
\references{
Forrester, Alexander I.J.; Sobester, Andras; Keane, Andy J. (2008). Engineering Design via Surrogate Modelling - A Practical Guide. John Wiley & Sons.

Zaefferer, Martin; Stork, Joerg; Friese, Martina; Fischbach, Andreas; Naujoks, Boris; Bartz-Beielstein, Thomas. (2014). Efficient global optimization for combinatorial problems. In Proceedings of the 2014 conference on Genetic and evolutionary computation (GECCO '14). ACM, New York, NY, USA, 871-878. DOI=10.1145/2576768.2598282 http://doi.acm.org/10.1145/2576768.2598282

Zaefferer, Martin; Stork, Joerg; Bartz-Beielstein, Thomas. (2014). Distance Measures for Permutations in Combinatorial Efficient Global Optimization. In Parallel Problem Solving from Nature - PPSN XIII (p. 373-383). Springer International Publishing.
}
\seealso{
\code{\link{predict.modelKriging}}
}

