% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/liftone_constrained_GLM.R
\name{liftone_constrained_GLM}
\alias{liftone_constrained_GLM}
\title{Find constrained D-optimal approximate design for generalized linear models (GLM)}
\usage{
liftone_constrained_GLM(
  X,
  W,
  g.con,
  g.dir,
  g.rhs,
  lower.bound,
  upper.bound,
  reltol = 1e-05,
  maxit = 500,
  random = TRUE,
  nram = 3,
  w00 = NULL,
  epsilon = 1e-12
)
}
\arguments{
\item{X}{Model matrix, with nrow = num of design points and ncol = num of parameters}

\item{W}{Diagonal of W matrix in Fisher information matrix, can be calculated from W_func_GLM in package}

\item{g.con}{A matrix of numeric constraint coefficients, one row per constraint, on column per variable (to be used in as const.mat lp() and mat in Rglpk_solve_LP())}

\item{g.dir}{Vector of character strings giving the direction of the constraint: each value should be one of "<," "<=," "=," "==," ">," or ">=". (In each pair the two values are identical.) to be used as const.dir in lp() and dir in Rglpk_solve_LP()}

\item{g.rhs}{Vector of numeric values for the right-hand sides of the constraints. to be used as const.rhs in lp() and rhs in Rglpk_solve_LP().}

\item{lower.bound}{A function to determine lower bound r_i1 in Step 3 of Constrained lift-one algorithm from Yifei, H., Liping, T., Yang, J. (2023) Constrained D-optimal design for paid research study}

\item{upper.bound}{A function to determine upper bound r_i2 in Step 3 of Constrained lift-one algorithm from Yifei, H., Liping, T., Yang, J. (2023) Constrained D-optimal design for paid research study}

\item{reltol}{The relative convergence tolerance, default value 1e-5}

\item{maxit}{The maximum number of iterations, default value 500}

\item{random}{TRUE or FALSE, if TRUE then the function will run with additional "nram" number of random initial points, default to be TRUE}

\item{nram}{When random == TRUE, the function will generate nram number of initial points, default is 3}

\item{w00}{Specified initial design proportion; default to be NULL, this will generate a random initial design}

\item{epsilon}{A very small number, for comparison of >0, <0, ==0, to reduce errors, default 1e-12}
}
\value{
w is the approximate D-optimal design

w0 is the initial design used to get optimal design w

maximum is the maximized |F| value

itmax is the number of iterations

convergence is TRUE or FALSE, if TRUE means the reported design is converged

deriv.ans is the derivative from step 6 of constrained lift-one algorithm

gmax is the maximum g function in step 8 of constrained lift-one algorithm

reason is the lift-one loops break reason, either "all derivatives <=0" or "gmax <=0"
}
\description{
Find constrained D-optimal approximate design for generalized linear models (GLM)
}
\examples{
#Example 6 in Section 3.4 of Yifei, H., Liping, T., Yang, J. (2025)
#Constrained D-optimal design for paid research study

#main effect model beta_0, beta_1, beta_21, beta_22
beta = c(0, -0.1, -0.5, -2)

#gives the 6 categories (0,0,0), (0,1,0),(0,0,1),(1,0,0),(1,1,0),(1,0,1)
X.liftone=matrix(data=c(1,0,0,0,1,0,1,0,1,0,0,1,1,1,0,0,1,1,1,0,1,1,0,1),
ncol=4, byrow=TRUE)

#calculate W matrix based on beta's under logit link
W_matrix=W_func_GLM(X= X.liftone, b=beta)

m=6 #number of categories
nsample = 200
rc = c(50, 40, 10, 200, 150, 50)/nsample
g.con = matrix(0,nrow=(2*m+1), ncol=m)
g.con[1,] = rep(1, m)
g.con[2:(m+1),] = diag(m)
g.con[(m+2):(2*m+1), ] = diag(m)
g.dir = c("==", rep("<=", m), rep(">=", m))
g.rhs = c(1, rc, rep(0, m))

lower.bound=function(i, w){
  nsample = 200
  rc = c(50, 40, 10, 200, 150, 50)/nsample
  m=length(w) #num of categories
  temp = rep(0,m)
  temp[w>0]=1-pmin(1,rc[w>0])*(1-w[i])/w[w>0];
  temp[i]=0;
  max(0,temp);
}
upper.bound=function(i, w){
  nsample = 200
  rc = c(50, 40, 10, 200, 150, 50)/nsample
  m=length(w) #num of categories
  rc[i];
  min(1,rc[i]);
}

approximate_design = liftone_constrained_GLM(X=X.liftone, W=W_matrix,
g.con=g.con, g.dir=g.dir, g.rhs=g.rhs, lower.bound=lower.bound,
upper.bound=upper.bound, reltol=1e-10, maxit=100, random=TRUE, nram=4,
w00=NULL, epsilon = 1e-8)



}
