% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BuyseTest.R
\name{BuyseTest}
\alias{BuyseTest}
\title{Generalized Pairwise Comparisons (GPC)}
\usage{
BuyseTest(formula, data, method.tte = NULL, correction.uninf = NULL,
  model.tte = NULL, method.inference = NULL, n.resampling = NULL,
  neutral.as.uninf = NULL, keep.pairScore = NULL, treatment = NULL,
  endpoint = NULL, type = NULL, threshold = NULL, censoring = NULL,
  operator = NULL, strata = NULL, alternative = NULL, seed = 10,
  cpus = NULL, trace = NULL, keep.comparison)
}
\arguments{
\item{formula}{[formula] a symbolic description of the model to be fitted. The response variable should be a binary variable defining the treatment arms. 
The rest of the formula should indicate the strata variables (if any) and the endpoints by order of priority.}

\item{data}{[data.frame] dataset.}

\item{method.tte}{[character] defines the method used to compare the observations of a pair in presence of right censoring (i.e. \code{"timeToEvent"} endpoints).
Can be \code{"Gehan"} or \code{"Peron"}. \code{"Gehan"} only scores pairs that can be decidedly classified as favorable, unfavorable, or neutral.
\code{"Peron"} uses the empirical survival curves of each group to also score the pairs that cannot be decidedly classified (see Peron et al. for more details).
Default value read from \code{BuyseTest.options()}.}

\item{correction.uninf}{[integer] should a correction be applied to remove the bias due to the presence of uninformative pairs?
0 indicates no correction, 1 impute the average score of the informative pair, and 2 performs inverse probability of censoring weights.
Default value read from \code{BuyseTest.options()}.}

\item{model.tte}{[list] optional survival models relative to each time to each time to event endpoint.
Models must \code{prodlim} objects and stratified on the treatment and strata variable.}

\item{method.inference}{[character] should the asymptotic theory (\code{"asymptotic"}),
or a permutation test (\code{"permutation"} or \code{"stratified permutation"}),
or bootstrap resampling (\code{"bootstrap"} or \code{"stratified boostrap"})
be used to compute p-values and confidence intervals.}

\item{n.resampling}{[integer] the number of simulations used for computing the confidence interval and the p.values. See details.
Default value read from \code{BuyseTest.options()}.}

\item{neutral.as.uninf}{[logical] should paired classified as neutral be re-analyzed using endpoints of lower priority.
Default value read from \code{BuyseTest.options()}.}

\item{keep.pairScore}{[logical] should the result of each pairwise comparison be kept?}

\item{treatment}{[character] the name of the treatment variable identifying the control and the experimental group.
Disregarded if the argument \code{formula} is defined.}

\item{endpoint}{[character vector] the name of the endpoint variable(s).
Disregarded if the argument \code{formula} is defined.}

\item{type}{[character vector] the type of each endpoint: \code{"binary"}, \code{"continuous"} or \code{"timeToEvent"}.}

\item{threshold}{[numeric vector] critical values used to compare the pairs (threshold of minimal important difference).
There must be one threshold for each endpoint variable.
Disregarded if the argument \code{formula} is defined.}

\item{censoring}{[character vector] the name of the binary variable(s) indicating whether the endpoint was observed or censored.
There must be one threshold for each endpoint variable.
Must value \code{NA} when the endpoint is not a time to event.
Disregarded if the argument \code{formula} is defined.}

\item{operator}{[character vector] the sign defining a favorable endpoint:
">0" indicates that higher values are favorable while "<0" indicates the opposite.
Disregarded if the argument \code{formula} is defined.}

\item{strata}{[numeric vector] if not \code{NULL}, the GPC will be applied within each group of patient defined by the strata variable(s).
Disregarded if the argument \code{formula} is defined.}

\item{alternative}{[character] the alternative hypothesis.
Must be one of \code{"two.sided"}, \code{"greater"} or \code{"less"}.
Default value read from \code{BuyseTest.options()}.}

\item{seed}{[integer, >0] the seed to consider for the permutation test.
If \code{NULL} no seed is set.}

\item{cpus}{[integer, >0] the number of CPU to use.
Only the permutation test can use parallel computation.
Default value read from \code{BuyseTest.options()}.}

\item{trace}{[integer] should the execution of the function be traced ? \code{0} remains silent
and \code{1}-\code{3} correspond to a more and more verbose output in the console.
Default value read from \code{BuyseTest.options()}.}

\item{keep.comparison}{Obsolete. Alias for 'keep.pairScore'.}
}
\value{
An \R object of class \code{\linkS4class{BuyseRes}}.
}
\description{
Performs Generalized Pairwise Comparisons for binary, continuous and time-to-event endpoints.
}
\details{
\bold{treatment:} The variable corresponding to \code{treatment} in data must have only two levels (e.g. \code{0} and \code{1}). \cr
\bold{endpoint, threshold, censoring, operator, and type:}  they must have the same length. \cr
\code{threshold} must be \code{NA} for binary endpoints and positive for continuous or time to event endpoints. \cr
\code{censoring} must be \code{NA} for binary or continuous endpoints and indicate a variable in data for time to event endpoints. 
Short forms for endpoint \code{type} are \code{"bin"} (binary endpoint), \code{"cont"} (continuous endpoint), \
code{"TTE"} (time-to-event endpoint). 
\bold{operator:} when the operator is set to \code{"<0"} the corresponding column in the dataset is multiplied by \code{-1}.

\bold{n.resampling:} The number of permutation replications must be specified to enable the computation of the confidence intervals and the p.value. 
A large number of permutations (e.g. \code{n.resampling=10000}) are needed to obtain accurate CI and p.value. See (Buyse et al., 2010) for more details. 

\bold{cpus parallelization:} Argument \code{cpus} can be set to \code{"all"} to use all available cpus.
The detection of the number of cpus relies on the \code{detectCores} function from the \emph{parallel} package .

\bold{Dealing with neutral or uninformative pairs:} Neutral pairs correspond to pairs for which the difference between the endpoint of the control observation and the endpoint of the treatment observation is (in absolute value) below the threshold. When \code{threshold=0}, neutral pairs correspond to pairs with equal endpoint.\cr
Uninformative pairs correspond to pairs for which the censoring prevent from classifying them into favorable, unfavorable or neutral. Neutral or uninformative pairs for an endpoint with priority \code{l} are, when available, analyzed on the endpoint with priority \code{l-1}.

\bold{method.tte:} the \code{method.tte="Peron"} is recommended in presence of right censored observations since it gives a more efficient estimator than \code{method.tte="Gehan"}.

\bold{method.inference:} the \code{method.inference="asymptotic"} estimate the distribution of the net benefit or win ratio statistics
based on the asymptotic theory of the U-statistics (see formula 2.2 in Bebu et al. 2016).
Its current implementation is not valid in small sample or when using \code{method.tte="Peron"}, or  \code{correction.uninf=1}, or \code{correction.uninf=2}.

\bold{correction.uninf:} in presence of uninformative pairs, the proportion of favorable, unfavorable, or neutral pairs is underestimated.
Inverse probability of censoring weights (\code{correction.uninf=2}) is only recommanded when the analysis is stopped after the first endpoint with uninformative pairs.
Imputing the average score of the informative pairs (\code{correction.uninf=1}) gives equivalent results at the first endpoint but better behaves at latter endpoints.
Note that both corrections will convert the whole proportion of uninformative pairs of a given endpoint into favorable, unfavorable, or neutral pairs.
}
\examples{
# reset the default value of the number of permuation sample
BuyseTest.options(method.inference = "none") # no permutation test

#### simulate some data ####
set.seed(10)
df.data <- simBuyseTest(1e2, n.strata = 2)

                                       # display 
if(require(prodlim)){
   resKM_tempo <- prodlim(Hist(eventtime,status)~Treatment, data = df.data)
   plot(resKM_tempo)
}

#### one time to event endpoint ####
BT <- BuyseTest(Treatment ~ TTE(eventtime, censoring = status), data= df.data)

summary(BT) # net benefit
summary(BT, percentage = FALSE)  
summary(BT, statistic = "winRatio") # win Ratio

## bootstrap to compute the CI
\dontrun{
    BT <- BuyseTest(Treatment ~ TTE(eventtime, censoring = status), data=df.data,
                    method.inference = "permutation", n.resampling = 1e3)
}
\dontshow{
    BT <- BuyseTest(Treatment ~ TTE(eventtime, censoring = status), data=df.data,
                    method.inference = "permutation", n.resampling = 1e1, trace = 0)
}
summary(BT, statistic = "netBenefit") ## default
summary(BT, statistic = "winRatio") 

## parallel boostrap
\dontrun{
    BT <- BuyseTest(Treatment ~ TTE(eventtime, censoring = status), data=df.data,
                    method.inference = "permutation", n.resampling = 1e3, cpus = 2)
    summary(BT)
}

## method Gehan is much faster but does not optimally handle censored observations
BT <- BuyseTest(Treatment ~ TTE(eventtime, censoring = status), data=df.data,
                method.tte = "Gehan", trace = 0)
summary(BT)

#### one time to event endpoint: only differences in survival over 1 unit ####
BT <- BuyseTest(Treatment ~ TTE(eventtime, threshold = 1, censoring = status), data=df.data)
summary(BT)

#### one time to event endpoint with a strata variable
BT <- BuyseTest(Treatment ~ strata + TTE(eventtime, censoring = status), data=df.data)
summary(BT)

#### several endpoints with a strata variable
f <- Treatment ~ strata + T(eventtime, 1, status) + B(toxicity) 
f <- update(f, 
            ~. + T(eventtime, 0.5, status) + C(score, 1) + T(eventtime, 0.25, status))

BT <- BuyseTest(f, data=df.data)
summary(BT)

#### real example : Veteran dataset of the survival package ####
#### Only one endpoint. Type = Time-to-event. Thresold = 0. Stratfication by histological subtype
#### method.tte = "Gehan"

if(require(survival)){
\dontrun{
  data(veteran,package="survival")
 
  ## method.tte = "Gehan"
  BT_Gehan <- BuyseTest(trt ~ celltype + TTE(time,threshold=0,censoring=status), 
                        data=veteran, method.tte="Gehan",
                        method.inference = "permutation", n.resampling = 1e3)
  
  summary_Gehan <- summary(BT_Gehan)
  summary_Gehan <- summary(BT_Gehan, statistic = "winRatio")
  
  ## method.tte = "Peron"
  BT_Peron <- BuyseTest(trt ~ celltype + TTE(time,threshold=0,censoring=status), 
                        data=veteran, method.tte="Peron",
                        method.inference = "permutation", n.resampling = 1e3)

  class(BT_Peron)
  summary(BT_Peron)
}
}
}
\references{
J. Peron, M. Buyse, B. Ozenne, L. Roche and P. Roy (2018). \bold{An extension of generalized pairwise comparisons for prioritized outcomes in the presence of censoring}. \emph{Statistical Methods in Medical Research} 27: 1230-1239  \cr 
D. Wang, S. Pocock (2016). \bold{A win ratio approach to comparing continuous non-normal outcomes in clinical trials}. \emph{Pharmaceutical Statistics} 15:238-245 \cr
I. Bebu, J. M. Lachin. \bold{Large sample inference for a win ratio analysis of a composite outcome based on prioritized components}. \emph{Biostatistics} 17(1):178-187 \cr
Marc Buyse (2010). \bold{Generalized pairwise comparisons of prioritized endpoints in the two-sample problem}. \emph{Statistics in Medicine} 29:3245-3257 \cr
Efron B (1967). \bold{The two sample problem with censored data}. \emph{Proceedings of the Fifth Berkeley Symposium on Mathematical Statistics and Probability} 4:831-583 \cr
Gehan EA (1965). \bold{A generalized two-sample Wilcoxon test for doubly censored data}. \emph{Biometrika}  52(3):650-653 \cr
}
\seealso{
\code{\link{BuyseRes-summary}} for a summary of the results of generalized pairwise comparison. \cr
\code{\link{BuyseRes-class}} for a presentation of the \code{BuyseRes} object. \cr
\code{\link{constStrata}} to create a strata variable from several clinical variables. \cr
}
\keyword{BuyseTest}
\keyword{function}
