% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mix_mode.R
\name{mix_mode}
\alias{mix_mode}
\title{Mode estimation}
\usage{
mix_mode(
  mixture,
  tol_mixp = 0,
  tol_x = 1e-06,
  tol_conv = 1e-08,
  type = "all",
  inside_range = TRUE
)
}
\arguments{
\item{mixture}{An object of class \code{mixture} generated with \code{\link[=mixture]{mixture()}}.}

\item{tol_mixp}{Components with a mixture proportion below \code{tol_mixp} are discarded when estimating modes;
note that this does not apply to the biggest component so that it is not possible to discard all components;
should be between \code{0} and \code{1}; default is \code{0}.}

\item{tol_x}{(for continuous mixtures) Tolerance parameter for distance in-between modes; default is \code{1e-6}; if two modes are closer than \code{tol_x} the first estimated mode is kept.}

\item{tol_conv}{(for continuous mixtures) Tolerance parameter for convergence of the algorithm; default is \code{1e-8}.}

\item{type}{(for discrete mixtures) Type of modes, either \code{"unique"} or \code{"all"} (the latter includes flat modes); default is \code{"all"}.}

\item{inside_range}{Should modes outside of \code{mixture$range} be discarded? Default is \code{TRUE}.
This sometimes occurs with very small components when K is large.}
}
\value{
A list of class \code{mix_mode} containing:
\item{mode_estimates}{estimates of the mixture modes.}
\item{algo}{algorithm used for mode estimation.}
\item{dist}{from \code{mixture}.}
\item{dist_type}{type of mixture distribution, i.e. continuous or discrete.}
\item{pars}{from \code{mixture}.}
\item{pdf_func}{from \code{mixture}.}
\item{K}{from \code{mixture}.}
\item{nb_var}{from \code{mixture}.}
}
\description{
Mode estimation in univariate mixture distributions.
The fixed-point algorithm of \insertCite{carreira-perpinan_mode-finding_2000;textual}{BayesMultiMode} is used for Gaussian mixtures.
The Modal EM algorithm of \insertCite{li_nonparametric_2007;textual}{BayesMultiMode} is used for other continuous mixtures.
A basic algorithm is used for discrete mixtures, see \insertCite{Cross2024;textual}{BayesMultiMode}.
}
\details{
This function finds modes in a univariate mixture defined as:
\deqn{p(.) = \sum_{k=1}^{K}\pi_k p_k(.),}
where \eqn{p_k} is a density or probability mass/density function.

\strong{Fixed-point algorithm}
Following \insertCite{carreira-perpinan_mode-finding_2000;textual}{BayesMultiMode}, a mode \eqn{x} is found by iterating the two steps:
\deqn{(i) \quad p(k|x^{(n)}) = \frac{\pi_k p_k(x^{(n)})}{p(x^{(n)})},}
\deqn{(ii) \quad x^{(n+1)} = f(x^{(n)}),}
with
\deqn{f(x) = (\sum_k p(k|x) \sigma_k)^{-1}\sum_k p(k|x) \sigma_k \mu_k,}
until convergence, that is, until \eqn{abs(x^{(n+1)}-x^{(n)})< \text{tol}_\text{conv}},
where \eqn{\text{tol}_\text{conv}} is an argument with default value \eqn{1e-8}.
Following Carreira-perpinan (2000), the algorithm is started at each component location.
Separately, it is necessary to identify identical modes which diverge only up to
a small value; this tolerance value can be controlled with the argument
\code{tol_x}.

\strong{MEM algorithm}
Following \insertCite{li_nonparametric_2007;textual}{BayesMultiMode}, a mode \eqn{x} is found by iterating the two steps:
\deqn{(i) \quad p(k|x^{(n)}) = \frac{\pi_k p_k(x^{(n)})}{p(x^{(n)})},}
\deqn{(ii) \quad x^{(n+1)} = \text{argmax}_x  \sum_k p(k|x) \text{log} p_k(x^{(n)}),}
until convergence, that is, until \eqn{abs(x^{(n+1)}-x^{(n)})< \text{tol}_\text{conv}},
where \eqn{\text{tol}_\text{conv}} is an argument with default value \eqn{1e-8}.
The algorithm is started at each component location.
Separately, it is necessary to identify identical modes which diverge only up to
a small value. Modes which are closer then \code{tol_x} are merged.

\strong{Discrete method}
By definition, modes must satisfy either:
\deqn{p(y_{m}-1) < p(y_{m}) > p(y_{m}+1);}
\deqn{p(y_{m}-1) < p(y_{m}) = p(y_{m}+1) = \ldots = p(y_{m}+l-1) > p(y_{m}+l).}

The algorithm evaluate each location point with these two conditions.
}
\examples{

# Example with a normal distribution ====================================
mu = c(0,5)
sigma = c(1,2)
p = c(0.5,0.5)

params = c(eta = p, mu = mu, sigma = sigma)
mix = mixture(params, dist = "normal", range = c(-5,15))
modes = mix_mode(mix)

# summary(modes)
# plot(modes)

# Example with a skew normal =============================================
xi = c(0,6)
omega = c(1,2)
alpha = c(0,0)
p = c(0.8,0.2)
params = c(eta = p, xi = xi, omega = omega, alpha = alpha)
dist = "skew_normal"

mix = mixture(params, dist = dist, range = c(-5,15))
modes = mix_mode(mix)
# summary(modes)
# plot(modes)

# Example with an arbitrary continuous distribution ======================
xi = c(0,6)
omega = c(1,2)
alpha = c(0,0)
nu = c(3,100)
p = c(0.8,0.2)
params = c(eta = p, mu = xi, sigma = omega, xi = alpha, nu = nu)

pdf_func <- function(x, pars) {
  sn::dst(x, pars["mu"], pars["sigma"], pars["xi"], pars["nu"])
}

mix = mixture(params, pdf_func = pdf_func,
dist_type = "continuous", loc = "mu", range = c(-5,15))
modes = mix_mode(mix)

# summary(modes)
# plot(modes, from = -4, to = 4)

# Example with a poisson distribution ====================================
lambda = c(0.1,10)
p = c(0.5,0.5)
params = c(eta = p, lambda = lambda)
dist = "poisson"


mix = mixture(params, range = c(0,50), dist = dist)

modes = mix_mode(mix)

# summary(modes)
# plot(modes)

# Example with an arbitrary discrete distribution =======================
mu = c(20,5)
size = c(20,0.5)
p = c(0.5,0.5)
params = c(eta = p, mu = mu, size = size)


pmf_func <- function(x, pars) {
  dnbinom(x, mu = pars["mu"], size = pars["size"])
}

mix = mixture(params, range = c(0, 50),
pdf_func = pmf_func, dist_type = "discrete")
modes = mix_mode(mix)

# summary(modes)
# plot(modes)

}
\references{
\insertRef{Cross2024}{BayesMultiMode}

\insertAllCited{}
}
