% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mc.backward.vs.R
\name{mc.backward.vs}
\alias{mc.backward.vs}
\title{Backward selection with two filters (using parallel computation)}
\usage{
mc.backward.vs(
  x,
  y,
  split.ratio = 0.8,
  probit = FALSE,
  true.idx = NULL,
  xinfo = matrix(0, 0, 0),
  numcut = 100L,
  usequants = FALSE,
  cont = FALSE,
  rm.const = TRUE,
  k = 2,
  power = 2,
  base = 0.95,
  split.prob = "polynomial",
  ntree = 50L,
  ndpost = 1000,
  nskip = 1000,
  keepevery = 1L,
  printevery = 100L,
  verbose = FALSE,
  mc.cores = 2L,
  nice = 19L,
  seed = 99L
)
}
\arguments{
\item{x}{A matrix or a data frame of predictors values with each row corresponding to an observation and each column 
corresponding to a predictor. If a predictor is a factor with \eqn{q} levels in a data frame, it is replaced with \eqn{q} dummy 
variables.}

\item{y}{A vector of response (continuous or binary) values.}

\item{split.ratio}{A number between \eqn{0} and \eqn{1}; the data set \code{(x, y)} is split into a training set and a testing
set according to the \code{split.ratio}.}

\item{probit}{A Boolean argument indicating whether the response variable is binary or continuous; \code{probit=FALSE} (by default)
means that the response variable is continuous.}

\item{true.idx}{(Optional) A vector of indices of the true relevant predictors; if provided, metrics including precision, recall
and F1 score are returned.}

\item{xinfo}{A matrix of cut-points with each row corresponding to a predictor and each column corresponding to a cut-point.
\code{xinfo=matrix(0.0,0,0)} indicates the cut-points are specified by BART.}

\item{numcut}{The number of possible cut-points; If a single number is given, this is used for all predictors; 
Otherwise a vector with length equal to \code{ncol(x)} is required, where the \eqn{i-}th element gives the number of 
cut-points for the \eqn{i-}th predictor in \code{x}. If \code{usequants=FALSE}, \code{numcut} equally spaced 
cut-points are used to cover the range of values in the corresponding column of \code{x}. 
If \code{usequants=TRUE}, then min(\code{numcut}, the number of unique values in the corresponding column of 
\code{x} - 1) cut-point values are used.}

\item{usequants}{A Boolean argument indicating how the cut-points in \code{xinfo} are generated; 
If \code{usequants=TRUE}, uniform quantiles are used for the cut-points; Otherwise, the cut-points are generated uniformly.}

\item{cont}{A Boolean argument indicating whether to assume all predictors are continuous.}

\item{rm.const}{A Boolean argument indicating whether to remove constant predictors.}

\item{k}{The number of prior standard deviations that \eqn{E(Y|x) = f(x)} is away from \eqn{+/-.5}. The response 
(\code{y}) is internally scaled to the range from \eqn{-.5} to \eqn{.5}. The bigger \code{k} is, the more conservative 
the fitting will be.}

\item{power}{The power parameter of the polynomial splitting probability for the tree prior. Only used if 
\code{split.prob="polynomial"}.}

\item{base}{The base parameter of the polynomial splitting probability for the tree prior if \code{split.prob="polynomial"}; 
if \code{split.prob="exponential"}, the probability of splitting a node at depth \eqn{d} is \code{base}\eqn{^d}.}

\item{split.prob}{A string indicating what kind of splitting probability is used for the tree prior. If 
\code{split.prob="polynomial"}, the splitting probability in Chipman et al. (2010) is used; 
If \code{split.prob="exponential"}, the splitting probability in Rockova and Saha (2019) is used.}

\item{ntree}{The number of trees in the ensemble.}

\item{ndpost}{The number of posterior samples returned.}

\item{nskip}{The number of posterior samples burned in.}

\item{keepevery}{Every \code{keepevery} posterior sample is kept to be returned to the user.}

\item{printevery}{As the MCMC runs, a message is printed every \code{printevery} iterations.}

\item{verbose}{A Boolean argument indicating whether any messages are printed out.}

\item{mc.cores}{The number of cores to employ in parallel.}

\item{nice}{Set the job niceness. The default niceness is \eqn{19} and niceness goes from \eqn{0} (highest) to \eqn{19} 
(lowest).}

\item{seed}{Seed required for reproducible MCMC.}
}
\value{
The function \code{mc.backward.vs()} returns a list with the following components.
\item{best.model.names}{The vector of column names of the predictors selected by the backward selection approach.}
\item{best.model.cols}{The vector of column indices of the predictors selected by the backward selection approach.}
\item{best.model.order}{The step where the best model is located.}
\item{models}{The list of winner models from each step of the backward selection procedure; length equals \code{ncol{x}}.}
\item{model.errors}{The vector of MSEs (or MLLs if the response variable is binary) for the \code{ncol{x}} winner models.}
\item{elpd.loos}{The vector of LOO scores for the \code{ncol{x}} winner models.}
\item{all.models}{The list of all the evaluated models.}
\item{all.model.errors}{The vector of MSEs (or MLLs if the response variable is binary) for all the evaluated models.}
\item{precision}{The precision score for the backward selection approach; only returned if \code{true.idx} is provided.}
\item{recall}{The recall score for the backward selection approach; only returned if \code{true.idx} is provided.}
\item{f1}{The F1 score for the backward selection approach; only returned if \code{true.idx} is provided.}
\item{all.models.idx}{The vector of Boolean arguments indicating whether the corresponding model in \code{all.models} is
acceptable or not; a model containing all the relevant predictors is an acceptable model; only returned if \code{true.idx} 
is provided.}
}
\description{
This function implements the backward variable selection approach for BART (see Algorithm 2 in Luo and Daniels (2021) for 
details). Parallel computation is used within each step of the backward selection approach.
}
\details{
The backward selection starts with the full model with all the predictors, followed by comparing the deletion of each predictor
using mean squared error (MSE) if the response variable is continuous (or mean log loss (MLL) if the response variable is binary)
and then deleting the predictor whose loss gives the smallest MSE (or MLL). This process is repeated until there is only one
predictor in the model and ultimately returns \code{ncol{x}} "winner" models with different model sizes ranging from \eqn{1} to
\code{ncol{x}}.\cr
Given the \code{ncol{x}} "winner" models, the one with the largest expected log pointwise predictive density based on leave-one-out 
(LOO) cross validation is the best model. See Section 3.3 in Luo and Daniels (2021) for details.\cr
If \code{true.idx} is provided, the precision, recall and F1 scores are returned.\cr
}
\examples{
## simulate data (Scenario C.C.1. in Luo and Daniels (2021))
set.seed(123)
data = friedman(100, 5, 1, FALSE)
## parallel::mcparallel/mccollect do not exist on windows
if(.Platform$OS.type=='unix') {
## test mc.backward.vs() function
  res = mc.backward.vs(data$X, data$Y, split.ratio=0.8, probit=FALSE, 
  true.idx=c(1:5), ntree=10, ndpost=100, nskip=100, mc.cores=2)
}
}
\references{
Chipman, H. A., George, E. I. and McCulloch, R. E. (2010). 
  "BART: Bayesian additive regression trees."
   \emph{Ann. Appl. Stat.} \strong{4} 266--298.

Luo, C. and Daniels, M. J. (2021)
  "Variable Selection Using Bayesian Additive Regression Trees."
  \emph{arXiv preprint arXiv:2112.13998}.
  
Rockova V, Saha E (2019). 
  “On theory for BART.” 
  \emph{In The 22nd International Conference on Artificial Intelligence and Statistics} (pp. 2839–2848). PMLR.
  
Vehtari, Aki, Andrew Gelman, and Jonah Gabry (2017).
  "Erratum to: Practical Bayesian model evaluation using leave-one-out cross-validation and WAIC."
  \emph{Stat. Comput.} 27.5, p. 1433.
}
\seealso{
\code{\link{permute.vs}}, \code{\link{medianInclusion.vs}} and \code{\link{abc.vs}}.
}
\author{
Chuji Luo: \email{cjluo@ufl.edu} and Michael J. Daniels: \email{daniels@ufl.edu}.
}
