% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/10_bvar.R, R/90_print.R
\name{bvar}
\alias{bvar}
\alias{print.bvar}
\title{Hierarchical Bayesian Vector Autoregression}
\usage{
bvar(data, lags, n_draw = 10000, n_burn = 5000, n_thin = 1,
  priors = bv_priors(), mh = bv_mh(), fcast = NULL, irf = NULL,
  verbose = TRUE, ...)

\method{print}{bvar}(x, ...)
}
\arguments{
\item{data}{Numeric matrix or dataframe. Note that observations need to be
ordered from earliest to latest one.}

\item{lags}{Integer scalar. Number of lags to apply to the data.}

\item{n_draw}{Integer scalar. Number of total iterations for the model
to cycle through.}

\item{n_burn}{Integer scalar. Number of iterations to discard.}

\item{n_thin}{Integer scalar. Provides the option of reducing the number of
stored iterations to every \emph{n_thin}'th one. The number of saved
iterations thus equals \code{(n_draw - n_burn) / n_thin}.}

\item{priors}{\code{bv_priors} object containing priors and their settings.
See \code{\link{bv_priors}}.}

\item{mh}{\code{bv_metropolis} object with settings regarding the acceptance
rate of the Metropolis-Hastings step. See \code{\link{bv_mh}}.}

\item{fcast}{\code{bv_fcast} object of forecast options set with
\code{\link{bv_fcast}}. May be set to \code{NULL} to skip forecasting.
Forecasts may also be calculated ex-post using \code{\link{predict.bvar}}.}

\item{irf}{\code{bv_irf} object with options regarding impulse responses and
forecast error variance decompositions. Set via\code{\link{bv_irf}}. May be
set to \code{NULL} to skip the calculation. May also be computed ex-post
using \code{\link{irf.bvar}}.}

\item{verbose}{Logical scalar. Whether to print intermediate results and
progress.}

\item{...}{Not used.}

\item{x}{A \code{bvar} object.}
}
\value{
Returns a \code{bvar} object with the following elements:
\itemize{
  \item \code{beta} - Numeric array with saved draws from the posterior
  distribution of the VAR coefficients. See \code{\link{coef.bvar}}.
  \item \code{sigma} - Numeric array with saved draws from the posterior
  distribution of the model's VCOV-matrix. See \code{\link{vcov.bvar}}.
  \item \code{hyper} - Numeric matrix with saved draws from the posterior
  distributions of the hierarchical priors' hyperparameters.
  \item \code{ml} - Numeric vector with the values of the posterior marginal
  likelihood corresponding to each draw of hyperparameters and associated
  VAR coefficients.
  \item \code{optim} - List with outputs from \code{\link[stats]{optim}},
  which is used to find suitable starting values.
  \item \code{prior} - \code{bv_priors} object. See \code{\link{bv_priors}}.
  \item \code{call} - Call to the function. See \code{\link{match.call}}.
  \item \code{meta} - List with meta information such as number of variables,
  accepted draws, number of iterations, et cetera.
  \item \code{variables} - Character vector with column names of \emph{data}.
  \item \code{fcast} - \code{bvar_fcast} object with posterior forecast
  draws, quantiles as well as the forecast's setup from \emph{fcast}.
  \item \code{irf} - \code{bvar_irf} object with posterior impulse response
  and their quantiles, forecast error variance decomposition draws, as well
  as the setup obtained from \emph{irf}.
}
}
\description{
Hierarchical Bayesian estimation of Vector Autoregression (VAR) models in
the fashion of Giannone et al. (2015).
}
\examples{
# Access a subset of the fred_qd dataset and transform it to be stationary
data("fred_qd")
data <- fred_qd[, c("CPIAUCSL", "UNRATE", "FEDFUNDS")]
data[5:nrow(data), 1] <- diff(log(data[, 1]), lag = 4) * 100
data <- data[5:nrow(data), ]

# Compute VAR using 2 lags and a ridiculously low number of draws
x <- bvar(
  data = data, lags = 1,
  n_draw = 500, n_burn = 400, n_thin = 2, verbose = FALSE
)

\donttest{
# Check out some of the outputs generated
plot(x)
predict(x)
plot(predict(x))
irf(x)
plot(irf(x))
}
}
\references{
Giannone, D., Lenza, M., & Primiceri, G. E. (2015). Prior Selection for Vector Autoregressions. Review of Economics and Statistics, 97, 436-451. \url{https://doi.org/10.1162/REST_a_00483}.
}
\seealso{
\code{\link{bv_priors}}; \code{\link{bv_mh}};
\code{\link{bv_fcast}}; \code{\link{bv_irf}}; \code{\link{predict.bvar}};
\code{\link{irf.bvar}}; \code{\link{plot.bvar}}
}
\author{
Nikolas Kuschnig, Lukas Vashold
}
\keyword{BVAR}
\keyword{VAR}
\keyword{autoregression}
\keyword{hierarchical}
\keyword{macroeconomics}
\keyword{prior}
\keyword{vector}
