% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv.BTLLasso.R
\name{cv.BTLLasso}
\alias{cv.BTLLasso}
\title{Cross-validation function for BTLLasso}
\usage{
cv.BTLLasso(Y, X = NULL, Z1 = NULL, Z2 = NULL, folds = 10, lambda,
  control = ctrl.BTLLasso(), cores = folds, trace = TRUE,
  trace.cv = TRUE, cv.crit = c("RPS", "Deviance"))
}
\arguments{
\item{Y}{A \code{response.BTLLasso} object created by
\code{\link{response.BTLLasso}}.}

\item{X}{Matrix containing all \bold{subject-specific covariates} that are
to be included with \bold{object-specific effects}. One row represents one
subject, one column represents one covariate. X has to be standardized.}

\item{Z1}{Matrix containing all \bold{object-subject-specific covariates}
that are to be included with \bold{object-specific effects}. One row
represents one subject, one column represents one combination between
covariate and object. Column names have to follow the scheme
'firstvar.object1',...,'firstvar.objectm',...,'lastvar.objectm'. The object
names 'object1',...,'objectm' have to be identical to the object names used
in the \code{response.BTLLasso} object \code{Y}. The variable names and the
object names have to be separated by '.'.  The rownames of the matrix',
Z.name, 'have to be equal to the subjects specified in the response object.
Z1 has to be standardized.}

\item{Z2}{Matrix containing all \bold{object-subject-specific covariates or
object-specific covariates} that are to be included with \bold{global
effects}. One row represents one subject, one column represents one
combination between covariate and object. Column names have to follow the
scheme 'firstvar.object1',...,'firstvar.objectm',...,'lastvar.objectm'. The
object names 'object1',...,'objectm' have to be identical to the object
names used in the \code{response.BTLLasso} object \code{Y}. The variable
names and the object names have to be separated by '.'.  The rownames of the
matrix', Z.name, 'have to be equal to the subjects specified in the response
object. Z2 has to be standardized.}

\item{folds}{Number of folds for the crossvalidation. Default is 10.}

\item{lambda}{Vector of tuning parameters.}

\item{control}{Function for control arguments, mostly for internal use. See
also \code{\link{ctrl.BTLLasso}}.}

\item{cores}{Number of cores used for (parallelized) cross-validation. By
default, equal to the number of folds.}

\item{trace}{Should the trace of the BTLLasso algorithm be printed?}

\item{trace.cv}{Should the trace fo the cross-validation be printed? If
parallelized, the trace is not working on Windows machines.}

\item{cv.crit}{Which criterion should be used to evaluate cross-validation. Choice is 
between Ranked probability score and deviance. Only \code{RPS} considers the ordinal
structure of the response.}
}
\value{
\item{coefs}{Matrix containing all (original) coefficients, one row
per tuning parameter, one column per coefficient.} 
\item{coefs.repar}{Matrix
containing all reparameterized (for symmetric side constraint) coefficients,
one row per tuning parameter, one column per coefficient.}
\item{logLik}{Vector of log-likelihoods, one value per tuning parameter.}
\item{design}{List containing design matrix and several additional information like, 
e.g., number and names of covariates.} 
\item{Y}{Response object.} 
\item{penalty}{List containing all penalty matrices and some further information on penalties} 
\item{response}{Vector containing 0-1 coded
response.} 
\item{X}{X matrix containing subject-specific covariates.} 
\item{Z1}{Z1 matrix containing subject-object-specific covariates.} 
\item{Z2}{Z2 matrix containing (subject)-object-specific covariates.} 
\item{lambda}{Vector of tuning parameters.} 
\item{control}{Control argument, specified by \code{\link{ctrl.BTLLasso}}.}
\item{criterion}{Vector containing values of the chosen cross-validation criterion, 
one value per tuning parameter.}
\item{folds}{Number of folds in cross validation.} 
\item{cv.crit}{Cross-validation criterion, either \code{RPS} or \code{Deviance}.}
}
\description{
Performs crossvalidation of BTLLasso, including the BTLLasso algorithm for
the whole data set.
}
\details{
Cross validation can be performed parallel, default is 10-fold cross
validation on 10 cores. Output is a cv.BTLLasso object which can then be
used for bootstrap confidence intervalls using \code{\link{boot.BTLLasso}}
and \code{\link{ci.BTLLasso}}.
}
\examples{

\dontrun{
##### Example with simulated data set containing X, Z1 and Z2
data(SimData)

## Specify tuning parameters
lambda <- exp(seq(log(151), log(1.05), length = 30)) - 1

## Specify control argument
## -> allow for object-specific order effects and penalize intercepts
ctrl <- ctrl.BTLLasso(penalize.intercepts = TRUE, object.order.effect = TRUE,
                      penalize.order.effect.diffs = TRUE)

## Simple BTLLasso model for tuning parameters lambda
m.sim <- BTLLasso(Y = SimData$Y, X = SimData$X, Z1 = SimData$Z1, 
                  Z2 = SimData$Z2, lambda = lambda, control = ctrl)
print(m.sim)

singlepaths(m.sim)

## Cross-validate BTLLasso model for tuning parameters lambda
set.seed(5)
m.sim.cv <- cv.BTLLasso(Y = SimData$Y, X = SimData$X, Z1 = SimData$Z1, 
                        Z2 = SimData$Z2, lambda = lambda, control = ctrl)
print(m.sim.cv)

singlepaths(m.sim.cv, plot.order.effect = FALSE, 
            plot.intercepts = FALSE, plot.Z2 = FALSE)
paths(m.sim.cv, y.axis = 'L2')

## Example for bootstrap confidence intervals for illustration only
## Don't calculate bootstrap confidence intervals with B = 10!!!!
set.seed(5)
m.sim.boot <- boot.BTLLasso(m.sim.cv, B = 10, cores = 10)
print(m.sim.boot)
ci.BTLLasso(m.sim.boot)


##### Example with small version from GLES data set
data(GLESsmall)

## extract data and center covariates for better interpretability
Y <- GLESsmall$Y
X <- scale(GLESsmall$X, scale = FALSE)
Z1 <- scale(GLESsmall$Z1, scale = FALSE)

## vector of subtitles, containing the coding of the X covariates
subs.X <- c('', 'female (1); male (0)')

## vector of tuning parameters
lambda <- exp(seq(log(61), log(1), length = 30)) - 1


## compute BTLLasso model 
m.gles <- BTLLasso(Y = Y, X = X, Z1 = Z1, lambda = lambda)
print(m.gles)

singlepaths(m.gles, subs.X = subs.X)
paths(m.gles, y.axis = 'L2')

## Cross-validate BTLLasso model 
m.gles.cv <- cv.BTLLasso(Y = Y, X = X, Z1 = Z1, lambda = lambda)
print(m.gles.cv)

singlepaths(m.gles.cv, subs.X = subs.X)
}

}
\author{
Gunther Schauberger\cr \email{gunther@stat.uni-muenchen.de}\cr
\url{http://www.statistik.lmu.de/~schauberger/}
}
\references{
Schauberger, Gunther and Tutz, Gerhard (2015): Modelling
Heterogeneity in Paired Comparison Data - an L1 Penalty Approach with an
Application to Party Preference Data, \emph{Department of Statistics, LMU
Munich}, Technical Report 183

Schauberger, Gunther, Groll Andreas and Tutz, Gerhard (2016): Modelling 
Football Results in the German Bundesliga Using Match-specific Covariates, 
\emph{Department of Statistics, LMU Munich}, Technical Report 197
}
\seealso{
\code{\link{BTLLasso}}, \code{\link{boot.BTLLasso}}, \code{\link{ctrl.BTLLasso}},
\code{\link{singlepaths}}, \code{\link{paths}}
}
\keyword{BTLLasso}
\keyword{cross}
\keyword{validation}

