% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BayesEstNAM.R
\name{banam}
\alias{banam}
\title{Bayesian estimation of the network autocorrelation model}
\usage{
banam(
  y,
  X,
  W,
  prior = "flat",
  prior.mean = NULL,
  prior.Sigma = NULL,
  postdraws = 5000,
  burnin = 1000
)
}
\arguments{
\item{y}{A numeric vector containing the observations of the outcome variable.}

\item{X}{The design matrix of the predictor variables. If absent a column of ones
is automatically to model the intercept.}

\item{W}{A weight matrix (in the case of a NAM with a single weight matrix) or
a list of weight matrices (in the case of a NAM with multiple weight matrices).}

\item{prior}{A character string specifying which prior to use in the case of a
NAM with a single weight matrix. The options are 'flat', 'IJ', and 'normal',
for a flat prior, the independence Jeffreys prior, and a normal prior, respectively.}

\item{prior.mean}{A scalar (or vector) specifying the prior mean(s) of the network
autocorrelation(s). The default prior mean is 0.}

\item{prior.Sigma}{A scalar (or matrix) specifying the prior variance (or prior covariance
matrix) of the network autocorrelation(s). In the univariate case, the default prior variance
is 1e6. In the multivariate case, the default prior covariance matrix is the identity matrix
times 1e6.}

\item{postdraws}{An integer specifying the number of posterior draws after burn-in.}

\item{burnin}{An integer specifying the number of draws for burn-in.}
}
\value{
The output is an object of class \code{banam}. For users of \code{BANAM}, the following
        are the useful objects:
\itemize{
\item \code{rho.draws} Matrix of posterior draws for the network autocorrelation parameter(s).
\item \code{beta.draws} Matrix of posterior draws for the coefficients.
\item \code{sigma2.draws} Matrix of posterior draws for the error variance.
\item \code{summarystats} Table with summary statistics of the posterior.
\item \code{residuals} Residuals based on all posterior draws.
\item \code{fitted.values} Fitted values based on all posterior draws.
}
}
\description{
The \code{banam} function can be used for Bayesian estimation of the
network autocorrelation model (NAM). In the case of a single weight matrix, a flat prior,
the independence Jeffreys prior, and a normal prior can be specified for the network autocorrelation
parameter. In the case of multiple weight matrices, a multivariate normal prior can be specified.
}
\examples{
\donttest{
#example analyses
#generate example data
set.seed(234)
n <- 50
d1 <- .2
Wadj1 <- sna::rgraph(n, tprob=d1, mode="graph")
W1 <- sna::make.stochastic(Wadj1, mode="row")
d2 <- .4
Wadj2 <- sna::rgraph(n, tprob=d2, mode="graph")
W2 <- sna::make.stochastic(Wadj2, mode="row")
# set rho, beta, sigma2, and generate y
rho1 <- .3
K <- 3
beta <- rnorm(K)
sigma2 <- 1
X <- matrix(c(rep(1, n), rnorm(n*(K-1))), nrow=n, ncol=K)
y <- c((solve(diag(n) - rho1*W1))\%*\%(X\%*\%beta + rnorm(n)))

#Bayesian estimation of NAM with a single weight matrix using a flat prior for rho
best1 <- banam(y,X,W1)
print(best1)

#Bayesian estimation of NAM with two weight matrices using standard normal priors
best2 <- banam(y,X,W=list(W1,W2))
print(best2)

#Bayes factor testing of equality/order hypotheses using environment of package 'BFpack'
BFbest2 <- BF(best2,hypothesis="rho1>rho2>0; rho1=rho2>0; rho1=rho2=0")
}
}
\references{
Dittrich, D., Leenders, R.Th.A.J., & Mulder, J. (2017).
Bayesian estimation of the network autocorrelation model. Social Network, 48, 213–236.
<doi:10.1016/j.socnet.2016.09.002>

Dittrich, D., Leenders, R.Th.A.J., & Mulder, J. (2019). Network autocorrelation
modeling: A Bayes factor approach for testing (multiple) precise and interval hypotheses.
Sociological Methods & Research, 48, 642-676. <doi:10.1177/0049124117729712>

Dittrich, D., Leenders, R.Th.A.J., & Mulder, J. (2020). Network
Autocorrelation Modeling: Bayesian Techniques for Estimating and Testing Multiple
Network Autocorrelations. Sociological Methodology, 50, 168-214.
<doi:10.1177/0081175020913899>
}
