# Setup -------------------------------------------------------------------

  file <- system.file(
    "extdata", "example.gt3x", package = "AGread"
  )

  test <- read_gt3x(file)

  test$RAW$Timestamp <- gsub(
    "\\..+$", "", as.character(test$RAW$Timestamp)
  )
  test$IMU$Timestamp <- gsub(
    "\\..+$", "", as.character(test$IMU$Timestamp)
  )


# Testing -----------------------------------------------------------------

  test_that("gt3x file reading and operations are consistent", {

    ## Helper for tests

      save_gt3x <- function(code) {
        path <- tempfile(fileext = ".rds")
        saveRDS(code, path)
        path
      }


    ## Initial tests

      testthat::expect_snapshot_file(
        save_gt3x(test$RAW),
        "read_3x_raw.rds"
      )

      testthat::expect_snapshot_file(
        save_gt3x(test$IMU),
        "read_3x_imu.rds"
      )


    ## Re-read file (for timestamps)

      test <- read_gt3x(file)

      raw_1s <- suppressMessages(
        collapse_gt3x(test$RAW)[ ,-2]
      )

      imu_1s <- suppressMessages(
        collapse_gt3x(test$IMU)[ ,-2]
      )

    ## Using `expect_equal` because snapshots are a headache

      testthat::expect_equal(
        raw_1s,
        structure(
          list(
            file_source_PrimaryAccel = rep(
              "gt3x file", 179
            ),
            Timestamp = structure(
              1550134680:1550134858,
              class = c("POSIXct", "POSIXt"),
              tzone = "UTC"
            ),
            day_of_year = rep(45, 179),
            minute_of_day = rep(
              538:540, each = 60, length.out = 179
            ),
            ENMO = c(
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
              0.147242996727975, 15.3110794971784, 14.1333025308784,
              13.4322538935885, 14.5722484104966, 15.3601004745579,
              15.0496902263116, 14.5438214420215, 13.9521583224617,
              13.7137689903077, 14.4831049575414, 14.1993604259647,
              14.1452597493579, 14.5362675280551, 14.2705274633095,
              13.9492032705403, 14.1003655619088, 14.5644396561098,
              12.8943195874359, 14.121399089549, 14.3211182498604,
              33.1094976375251, 26.9634776026769, 21.5011820524098,
              24.900696045977, 26.2159441177982, 23.2342679148507,
              18.6370944710913, 19.2797275635475, 20.4201422371807,
              23.7576701427788, 23.4746485326858, 24.8908904391817,
              23.3324100981841, 38.8053673141962, 86.3547997256223,
              27.6449969962373, 157.458183168595, 282.025546685355,
              725.432553062419, 0,                36.4191310806679,
              41.6579779188873, 32.8781457011956, 29.0159466837398,
              59.9331747411744, 31.2835062639247, 47.4058843032853,
              29.4155641929172, 44.360982294117, 42.9155020408078,
              49.4595603931035, 38.779642768657, 23.1786595938994,
              43.6057748035425, 109.131198928127, 61.000095090718,
              22.834526160409, 5.67086474334152, 29.2703430950095,
              62.173139206109, 11.6476850439268, 10.0279320517214,
              54.1481057628295, 68.6572754505878, 39.6354149210835,
              52.4882743913887, 52.3787187758147, 35.6823757926388,
              39.3632957166498, 69.824597186359, 64.0823265516044,
              89.0424480813452, 51.9761076971956, 51.0519064097246,
              54.7160734275887, 48.039377080363, 38.2411951809622,
              34.7768829589067, 49.7403336223745, 60.4160008292846,
              62.4863101776441, 49.6117933175008, 64.0565371812013,
              49.840928792583, 13.5682166432938, 8.03806080589374,
              9.31217771757563, 48.3868389455476, 58.741411056744,
              35.2642894642044, 20.7116890400862, 7.79191716830875,
              62.078058794254, 39.1988483194559, 60.3439258096418,
              60.5179644746455, 22.5644766968477, 15.0123570466298,
              22.0393483855661, 50.3897258150327, 57.2609945253008,
              39.0417942507554, 49.7082612375607
            )
          ),
          class = "data.frame",
          row.names = c(NA,-179L)
        )
      )

      testthat::expect_equal(
        imu_1s,
        structure(
          list(
            file_source_IMU = rep(
              "gt3x file", 179
            ),
            Timestamp = structure(
              1550134680:1550134858,
              class = c("POSIXct", "POSIXt"),
              tzone = "UTC"
            ),
            mean_Accelerometer_x = c(
              0.0126708984375, 0.0123828125, 0.0123291015625,
              0.0121337890625, 0.0122265625, 0.0120849609375,
              0.012724609375, 0.01275791015625, 0.012646484375,
              0.012783203125, 0.012529296875, 0.0127978515625,
              0.0124609375, 0.0127099609375, 0.0126318359375,
              0.0127490234375, 0.0125830078125, 0.012822265625,
              0.0129541015625, 0.012734375, 0.01287109375,
              0.01279296875, 0.012835400390625, 0.0126123046875,
              0.01263671875, 0.0128125, 0.0126220703125,
              0.0128369140625, 0.01267578125, 0.0129638671875,
              0.012626953125, 0.0127197265625, 0.0125341796875,
              0.0129150390625, 0.0127001953125, 0.0126806640625,
              0.012705029296875, 0.012802734375, 0.01287109375,
              0.0128515625, 0.0128515625, 0.01291015625,
              0.0127783203125, 0.012919921875, 0.0129296875,
              0.0128564453125, 0.0129150390625, 0.0128857421875,
              0.01243427734375, 0.012607421875, 0.01263671875,
              0.0127734375, 0.0128076171875, 0.0126416015625,
              0.0123974609375, 0.01294921875, 0.012587890625,
              0.01263671875, 0.0125634765625, 0.0128662109375,
              0.012622802734375, 0.0125830078125, 0.0126904296875,
              0.012607421875, 0.0126708984375, 0.0129052734375,
              0.0126806640625, 0.012705078125, 0.012744140625,
              0.0125390625, 0.012734375, 0.012724609375,
              0.013072412109375, 0.0130126953125, 0.013017578125,
              0.0429443359375, 0.9440625, 0.9866015625,
              0.9868017578125, 0.98681640625, 0.9869775390625,
              0.9868017578125, 0.986689453125, 0.9868501953125,
              0.986787109375, 0.987158203125, 0.986796875,
              0.9865478515625, 0.9867724609375, 0.9866650390625,
              0.9866552734375, 0.9867333984375, 0.986787109375,
              0.9868603515625, 0.986719775390625,
              0.9868310546875, 0.8653369140625, 0.0048583984375,
              -0.0012646484375, -0.0011572265625, -0.0015087890625,
              -0.000537109375, 0.00169921875, 0.0026806640625,
              0.001610009765625, 0.001015625, 0.000888671875,
              0.0005712890625, 0.0008544921875, 0.00611328125,
              -0.192314453125, 0.038623046875, 0.1175341796875,
              0.10236328125, 0.5249560546875, -7.25585937499998e-05,
              -0.0227978515625, -0.0088720703125, 0.0033349609375,
              -0.0005126953125, 0.1281298828125, 0.1914111328125,
              -0.2051318359375, 0.1349267578125, -0.2451904296875,
              0.09891484375, -0.0060107421875, 0.2760888671875,
              0.04236328125, -0.02439453125, -0.1561083984375,
              0.010556640625, -0.1452294921875, -0.0311083984375,
              0.0663593750000001, -0.0007421875, -0.1098974609375,
              -0.0660693359375, 0.4657177734375, -0.302666015625,
              -0.0977783203125, 0.025234375, -0.3703662109375,
              0.404052734375, 0.01143134765625, 0.132099609375,
              -0.1521533203125, 0.0192578125, 0.025810546875,
              0.0253515625, 0.1034326171875, -0.000654296875,
              -0.1054052734375, 0.266640625, 0.069270068359375,
              -0.2778759765625, 0.0497265625, -0.16998046875,
              -0.103310546875, -0.030107421875, -0.1972998046875,
              -0.2126953125, -0.203720703125, 0.00168564453125007,
              0.262060546875, 0.1159814453125, -0.300498046875,
              -0.0006787109375, 0.090302734375, -0.039521484375,
              -0.0391162109375, -0.00443359375, -0.06266826171875,
              -0.0614892578125, -0.0377490234375, 0.4689501953125,
              -0.56171875, 0.0076611328125, -0.182294921875
            ),
            mean_Accelerometer_y = c(
             -0.00400390625,
             -0.0037939453125, -0.00390625, -0.003828125, -0.00376953125,
             -0.003740234375, -0.003515625, -0.003921044921875, -0.003857421875,
             -0.00376953125, -0.003701171875, -0.003759765625, -0.0037255859375,
             -0.003759765625, -0.003955078125, -0.0037646484375, -0.0039501953125,
             -0.0039404296875, -0.0039990234375, -0.003896484375, -0.0035986328125,
             -0.0034326171875, -0.003761474609375, -0.0036376953125, -0.0037744140625,
             -0.003779296875, -0.0036767578125, -0.003662109375, -0.0038818359375,
             -0.0036181640625, -0.003798828125, -0.003837890625, -0.00357421875,
             -0.0037353515625, -0.0039501953125, -0.003955078125, -0.0036306640625,
             -0.003779296875, -0.00376953125, -0.0036962890625, -0.003720703125,
             -0.003671875, -0.0038330078125, -0.003818359375, -0.0036279296875,
             -0.0039208984375, -0.0035205078125, -0.003525390625, -0.004142919921875,
             -0.0038330078125, -0.00373046875, -0.0037939453125, -0.003818359375,
             -0.0036865234375, -0.0039013671875, -0.0040185546875, -0.0037451171875,
             -0.003994140625, -0.0039892578125, -0.00388671875, -0.003770751953125,
             -0.003935546875, -0.0039892578125, -0.0036669921875, -0.003779296875,
             -0.0036962890625, -0.0038427734375, -0.0039306640625, -0.004130859375,
             -0.0043017578125, -0.0033447265625, -0.003505859375, -0.003558251953125,
             -0.0037890625, -0.0035546875, -0.00541015625, -0.0148876953125,
             -0.0157666015625, -0.0157666015625, -0.0159375, -0.0159521484375,
             -0.01591796875, -0.016220703125, -0.01619541015625, -0.0164501953125,
             -0.01640625, -0.0162109375, -0.01609375, -0.016064453125, -0.0162744140625,
             -0.0161328125, -0.0161328125, -0.016376953125, -0.016142578125,
             -0.016272607421875, -0.0159814453125, -0.256787109375, -1.0171484375,
             -1.0248876953125, -1.025068359375, -1.0238330078125, -1.023837890625,
             -1.0186572265625, -1.020244140625, -1.02535146484375, -1.027294921875,
             -1.0272705078125, -1.02712890625, -1.0267822265625, -1.0434228515625,
             -0.93568359375, -0.300908203125, -0.243642578125, -0.2654833984375,
             -0.7529931640625, 0.004645751953125, -0.4049755859375, -1.02921875,
             -1.0101953125, -1.0119775390625, -0.9533154296875, -0.8085107421875,
             -0.877880859375, -0.749609375, -0.805810546875, -0.88136728515625,
             -0.85478515625, -0.47833984375, 0.5755615234375, -0.65599609375,
             -0.1308642578125, 0.1020263671875, -0.8915478515625, -0.8536181640625,
             -0.0838115234374998, -0.396376953125, -0.9266015625, -0.93388671875,
             0.1118603515625, -0.683271484375, 0.6274951171875, -0.3172265625,
             -0.05013671875, -0.8170166015625, -0.80203583984375, -0.8394677734375,
             -0.908828125, -0.9239599609375, -0.67697265625, 0.008330078125,
             0.0155126953125, -0.1004150390625, -0.8565185546875, -0.7684130859375,
             -0.81588994140625, -0.8460107421875, -0.8561328125, -0.853359375,
             -0.0738232421875, 0.2047216796875, -0.90296875, -0.8575439453125,
             -0.860615234375, -0.42644736328125, -0.64328125, -0.8547900390625,
             -0.843759765625, -0.8730419921875, -0.8750048828125, -0.74134765625,
             -0.144453125, 0.156953125, -0.99539150390625, -0.9927294921875,
             -0.98486328125, 0.4163818359375, -0.562431640625, -0.3019091796875,
             -0.6309033203125
            ),
            mean_Accelerometer_z = c(
             0.9949267578125,
             0.9954296875, 0.9950634765625, 0.995029296875, 0.9952880859375,
             0.995263671875, 0.9954248046875, 0.9955572265625, 0.9956982421875,
             0.995615234375, 0.9958642578125, 0.996064453125, 0.995654296875,
             0.9961181640625, 0.9961279296875, 0.9966845703125, 0.9966552734375,
             0.996484375, 0.9969921875, 0.9967431640625, 0.9970068359375,
             0.996806640625, 0.997210595703125, 0.99705078125, 0.997177734375,
             0.997255859375, 0.996953125, 0.9969775390625, 0.9972021484375,
             0.9980419921875, 0.99798828125, 0.9982421875, 0.99783203125,
             0.9983984375, 0.99818359375, 0.998466796875, 0.997955517578125,
             0.99783203125, 0.998515625, 0.9981689453125, 0.9983642578125,
             0.9984765625, 0.998681640625, 0.998486328125, 0.9990966796875,
             0.9993212890625, 0.9989892578125, 0.998818359375, 0.99905732421875,
             0.9994384765625, 0.9989501953125, 0.9989794921875, 0.9993017578125,
             0.9994921875, 0.9996435546875, 0.9995556640625, 0.9997802734375,
             1.000068359375, 1.0001171875, 1.000185546875, 0.999908203125,
             0.9997607421875, 1.000009765625, 1.000078125, 1.0004736328125,
             1.0000048828125, 1.000185546875, 0.999814453125, 1.000068359375,
             1.0004296875, 1.000263671875, 1.00033203125, 1.00043037109375,
             1.0004052734375, 1.000712890625, 0.99962890625, 0.0171142578125,
             -0.0631787109375, -0.063369140625, -0.0630908203125, -0.0633447265625,
             -0.06236328125, -0.0625146484375, -0.06278525390625, -0.062900390625,
             -0.0630810546875, -0.0625927734375, -0.062744140625, -0.0626513671875,
             -0.0630126953125, -0.0625, -0.0624560546875, -0.0624072265625,
             -0.062578125, -0.062181298828125, -0.0625439453125, -0.0771337890625,
             -0.080771484375, 0.0718115234375, -0.05591796875, -0.0749560546875,
             0.0575634765625, 0.1358544921875, 0.12390625, 0.07163173828125,
             0.0240869140625, -0.0034375, 0.0065625, 0.0266650390625, 0.005751953125,
             0.1234912109375, 0.9497607421875, 1.007177734375, 1.0054150390625,
             1.011396484375, 1.00236879882812, 0.7797216796875, 0.0054443359375,
             -0.1256005859375, -0.023642578125, -0.27962890625, 0.066630859375,
             -0.204912109375, 0.2903564453125, -0.22654296875, 0.18567333984375,
             -0.401142578125, 0.64931640625, -0.557099609375, 0.2054638671875,
             -0.21267578125, -0.09171875, 0.4250830078125, 0.565517578125,
             0.796573291015625, -0.1943994140625, 0.43154296875, 0.418994140625,
             0.28908203125, 0.249052734375, 0.188369140625, 0.2591015625,
             0.262568359375, 0.139130859375, 0.3099021484375, -0.274521484375,
             0.1611767578125, 0.036220703125, 0.45736328125, -0.326279296875,
             -0.161826171875, -0.012470703125, 0.5320947265625, 0.1233056640625,
             0.153694140625, -0.3446484375, -0.0166796875, 0.2988232421875,
             -0.2824951171875, -0.2032861328125, 0.387724609375, 0.52662109375,
             0.52517578125, 0.48250732421875, -0.016201171875, -0.0164697265625,
             0.4165576171875, 0.5382958984375, -0.174404296875, 0.6044140625,
             -0.334921875, -0.2041259765625, 0.2395123046875, 0.2625732421875,
             0.28240234375, 0.2543896484375, 0.319052734375, 0.3612353515625,
             0.0427587890625
            ),
            mean_Temperature_C = c(
              54.6947318860003, 54.5913982519112,
              54.460538933843, 54.3163810263442, 54.1691979892677, 54.0095550125445,
              53.8512598658311, 53.7071124414546, 53.5759431224841, 53.4502055584534,
              53.3132660294518, 53.1746192491044, 53.0310903289435, 52.8948995943918,
              52.7625127356458, 52.6191036225969, 52.4675176741548, 52.3250370662238,
              52.190164209904, 52.0565193764821, 51.9121518065373, 51.7793156711213,
              51.6532489370503, 51.5194839969986, 51.3947947451978, 51.2682484831611,
              51.1390964164387, 51.0083569054825, 50.8717468460388, 50.7393899390708,
              50.6188040808555, 50.5043882889075, 50.3740381510652, 50.2453054092348,
              50.129212317719, 50.0118912033054, 49.8928562481547, 49.7721571722186,
              49.6498940144355, 49.5392820982931, 49.426573557691, 49.3064070243677,
              49.1838143970266, 49.0666430415029, 48.9542040669027, 48.8425438385305,
              48.7344179199619, 48.6223683184757, 48.5118699195719, 48.4081232504642,
              48.3027229436933, 48.2019352107339, 48.1010276706626, 47.9976940365734,
              47.8934318973664, 47.7864441463617, 47.6781085653471, 47.5809449975253,
              47.4861775719433, 47.3813763007323, 47.2750693536415, 47.1732713472102,
              47.0746700940446, 46.9797828613506, 46.8892985400221, 46.7966576906779,
              46.6967086075024, 46.5981373061148, 46.5032800251988, 46.4129754145383,
              46.3219819129839, 46.2283227031877, 46.140162340314, 46.0414236084874,
              45.9377305530623, 45.8428433203683, 45.7490942552382, 45.655644707888,
              45.561056992974, 45.4794084461546, 45.3965019246593, 45.3226707918971,
              45.2467430346751, 45.1713852597884, 45.0878189006238, 45.010064084944,
              44.9327285941562, 44.8508104813348, 44.7737745083269, 44.7004825075686,
              44.629916118608, 44.5563545518478, 44.4829127921995, 44.4083927685434,
              44.3337433532063, 44.2618686705828, 44.1873486469266, 44.1167223544101,
              44.0458264958915, 43.974511312481, 43.9055922713103, 43.8353853036857,
              43.7628720491553, 43.6891008199491, 43.6174807274386, 43.5451525748962,
              43.471471201024, 43.4103695739104, 43.3435172054215, 43.2787614613923,
              43.2084346866557, 43.1413427039428, 43.0751193227917, 43.0110824214345,
              42.9390783471301, 42.8605322054756, 42.7682034536581, 42.6776292769956,
              42.5926560828185, 42.5115766197811, 42.4391532205847, 42.3743076212215,
              42.3001769706793, 42.2266753074751, 42.1552103651746, 42.0862913240039,
              42.0170128614971, 41.9513286123501, 41.8901970334585, 41.8261301803232,
              41.7695812234652, 41.714350144839, 41.6559741295232, 41.6025102057988,
              41.5437478125468, 41.4880045585168, 41.4356788023563, 41.3789201830523,
              41.3241383810961, 41.2713333964877, 41.2170607747574, 41.1594036021134,
              41.1054304981631, 41.0567588589182, 41.0065737563312, 40.9578880397507,
              40.9018782148966, 40.8527872507597, 40.8039059490689, 40.7532574924762,
              40.7087191985949, 40.6542369144186, 40.6026300009301, 40.5489564147597,
              40.4953768771723, 40.4424778439809, 40.3935665905121, 40.3432176516994,
              40.2954445657944, 40.2463835534356, 40.1975921070787, 40.1529939096414,
              40.1054005344044, 40.0604533987534, 40.0189896548835, 39.9726243025444,
              39.9279961533291, 39.8803728263141, 39.8351156897609, 39.7926141167833,
              39.7476564980101, 39.7042264199147, 39.6667205039892, 39.6239856066611,
              39.5833709956974, 39.5427563847337, 39.4990866924143, 39.4564353605467,
              39.415940556695
            ),
            Gyroscope_VM_DegPerS = c(
               0.817274639254085,
               0.796904751863188, 0.832365473500669, 0.826904758514041, 0.819911933809771,
               0.825183289300233, 0.812165316333012, 0.8347044485616, 0.822579354054801,
               0.811547116039663, 0.814413188393496, 0.819363615980502, 0.823804174050602,
               0.824845865098866, 0.833602262825933, 0.819944502707948, 0.820997818029185,
               0.817825664090288, 0.82518657787507, 0.817809329543203, 0.815492119435778,
               0.818969960908174, 0.818641331013528, 0.827144188466406, 0.82377168759624,
               0.828659250684145, 0.851671394688444, 0.838598214642062, 0.844206674812718,
               0.829444805901021, 0.84589024127711, 0.829153443584877, 0.834320321791147,
               0.82404514882212, 0.815664218750307, 0.815198557943928, 0.826655964466531,
               0.806064177341197, 0.800216051344058, 0.794012307917641, 0.795698849862832,
               0.806843729075136, 0.823962663850995, 0.827090232445969, 0.828083703413252,
               0.827233955756051, 0.830345038296146, 0.837363032242026, 0.841989575751059,
               0.842271123225737, 0.842913077953762, 0.844175318388724, 0.839211398889193,
               0.833705237885878, 0.848087474354899, 0.856139412008613, 0.836478093603678,
               0.848739142332625, 0.839034621179045, 0.837215895022729, 0.843989797295681,
               0.836147409803587, 0.836457872602774, 0.844727757230007, 0.842052447601728,
               0.852837365108285, 0.848827441316024, 0.84320493827141, 0.850999025223474,
               0.828134710900663, 0.812961605897341, 0.817949358580734, 0.820659855302952,
               0.804952144215402, 0.816782831800298, 30.5997533443439, 112.7541521175,
               0.825565697339374, 0.811285806888717, 0.818957267808625, 0.827157733728779,
               0.820272979347604, 0.817292308726213, 0.811492610377456, 0.81168726338404,
               0.824147679530814, 0.808915991244314, 0.822580907603523, 0.8166355884576,
               0.811617736534795, 0.815404320332093, 0.821221656799306, 0.827346534943819,
               0.831072496931152, 0.816686837490265, 0.813638149179426, 127.758653376581,
               62.8490930614281, 16.9601948434161, 20.5970658727366, 20.0360474690198,
               18.8383915428799, 4.8406221423416, 11.2884661765777, 5.84135373090127,
               9.37246801245486, 14.1496427496351, 12.1677135287042, 11.3225609867602,
               25.1893106283564, 111.39980877125, 68.513404155643, 508.845408457362,
               175.911890219883, 1062.36753111166, 0.844859598120649, 108.779311699837,
               199.887190283855, 147.890926458837, 148.445704596686, 102.462205989124,
               229.857903593524, 172.632513697311, 203.433546259829, 159.782786117227,
               208.753572416623, 269.807370331005, 176.626320585122, 140.817467924303,
               205.469655534349, 361.550721261718, 259.268142453107, 85.8811273869776,
               21.8819452216342, 165.349182475458, 212.401354537887, 9.84477683482413,
               17.3197432476166, 239.432589893499, 205.89717062719, 208.25076156371,
               266.044014375381, 297.26340819913, 161.968663305421, 243.690092199267,
               199.770555135978, 178.125224036011, 223.287448854442, 275.480800865399,
               298.417645112695, 330.478027145934, 364.781019186801, 89.2832372154391,
               191.254299968622, 303.320747814945, 157.158572347657, 322.47126693313,
               254.650600174788, 344.360645934949, 302.228107755301, 58.3384830215487,
               3.10273743996308, 5.06463705011715, 208.010593170641, 215.663468125402,
               241.88907745449, 132.184340303575, 27.8807016150274, 265.833881927498,
               222.920920284826, 292.816578956907, 298.306305059353, 53.9843653584289,
               5.44948173133307, 38.0318393544599, 230.082394467896, 201.207683709701,
               245.512270179076, 185.911516561798
              ),
            mean_abs_Gyroscope_x_DegPerS = c(
               0.75981210769972,
               0.742886739811455, 0.772629715170346, 0.771960593503254, 0.765294932576504,
               0.772244335006708, 0.76483538181584, 0.781997287692304, 0.7721173460717,
               0.76053787746448, 0.765326779717259, 0.770870675197439, 0.778840760881985,
               0.776915007675226, 0.786132468609354, 0.77858620723454, 0.779723762736865,
               0.77244921242181, 0.781379216563698, 0.778403032197318, 0.774126421966772,
               0.777779105673611, 0.777179554836867, 0.786655126037, 0.780833176005245,
               0.782983261003635, 0.813035732475364, 0.796886435727044, 0.805813129625647,
               0.789832527560893, 0.809929724868661, 0.794050752798397, 0.797777081875645,
               0.790989684885682, 0.780714453958518, 0.780607494026967, 0.793398792179998,
               0.772783666002022, 0.767764159328455, 0.765254132961176, 0.762603626069256,
               0.773603493818115, 0.788716409361125, 0.795277290664805, 0.792131058029057,
               0.795565523451733, 0.797527283209869, 0.805782023370942, 0.807107795243618,
               0.810174504081959, 0.809931312349354, 0.812201658381775, 0.810661195552947,
               0.800778837017625, 0.814232538631191, 0.827521983840394, 0.806474823337166,
               0.82248228722818, 0.811366481209763, 0.80981685771101, 0.817004669467924,
               0.808782123644054, 0.809375481013676, 0.816655568704536, 0.812334475159527,
               0.82140724086239, 0.822293542230966, 0.817057642777942, 0.828374425379666,
               0.801204390716781, 0.779695733359779, 0.78722630248558, 0.791186882789872,
               0.774615862815339, 0.783072820844043, 1.47087827011305, 39.1164688380338,
               0.788655479256873, 0.782248493823747, 0.785032885772044, 0.797086719474055,
               0.788462567082668, 0.786263812099108, 0.778280055366044, 0.776271010560663,
               0.791098464899613, 0.778852361250127, 0.790904937050041, 0.786840373114795,
               0.782377906325783, 0.78378250798525, 0.792266558423204, 0.798966790653479,
               0.793403366087885, 0.783778250632497, 0.784451666788944, 51.1415978072743,
               45.3414601487362, 16.0179003667453, 17.2682923190695, 18.5956544775138,
               13.576889536215, 4.21530738920235, 5.70828319373321, 5.71608436443179,
               8.322480824285, 13.205498387401, 12.1592188161218, 11.3178907657117,
               18.9204581388939, 93.7494586876209, 41.2253134173373, 5.38055719777149,
               3.15721085156633, 10.8615692385351, 0.827133713215522, 87.5995125078204,
               69.7161986282041, 60.5734574666662, 52.9903096446521, 52.6918030072919,
               14.456329153268, 18.2306019914278, 30.8060016594801, 20.5293012292082,
               51.0095518589113, 39.0350784624765, 126.680592204615, 116.859543874854,
               192.030935782093, 324.015446419999, 238.684614608993, 69.550841434414,
               15.196483101547, 156.376510357569, 185.681135437459, 3.55972128011699,
               5.17481852871825, 19.7601780467902, 31.036713621413, 38.2075647470931,
               40.6255803311971, 64.4396898265549, 26.7445903427328, 40.0878031582683,
               50.6183350133766, 58.0251513816767, 35.7562288806685, 119.125058781928,
               283.166787973502, 291.266364390931, 341.195659258549, 70.6619887268711,
               23.6267512876661, 39.1031015356526, 25.7716777146645, 36.4244211188939,
               105.308457701957, 307.392606100992, 276.273089525105, 53.9671488830308,
               1.62446038554162, 3.16589599845899, 196.808011996745, 176.639160033042,
               42.7600600277579, 20.56310825253, 4.77411062933938, 24.6158006078437,
               116.114057629504, 275.372229018007, 283.665191118051, 36.2332937471634,
               1.73796579236035, 9.59760826104504, 43.6523096765731, 48.0385899521583,
               216.258677946218, 167.782163648837
              ),
            mean_abs_Gyroscope_y_DegPerS = c(
              0.268329694262594,
              0.263595072854986, 0.278537673085527, 0.266116385160448, 0.265303847924428,
              0.260304381534684, 0.235984238354317, 0.262481710089573, 0.251423855675247,
              0.252670195763983, 0.251257369621439, 0.246525778193441, 0.232770118160194,
              0.240576693445385, 0.241973976798669, 0.226688406054277, 0.221493319870938,
              0.23567885583157, 0.233275809335335, 0.2213568657099, 0.227636213025504,
              0.230086121972681, 0.22843503275978, 0.225925784741689, 0.225864022074074,
              0.235363414546265, 0.218577888404925, 0.221679922279167, 0.218961246130275,
              0.219922369656346, 0.210423021769906, 0.208090019478122, 0.210872606025263,
              0.193263812243576, 0.205199593664956, 0.202731548640612, 0.198226237937105,
              0.193955106534325, 0.182412879228106, 0.162826256640084, 0.178823300150203,
              0.191306326821778, 0.196406505274776, 0.188067638263376, 0.205077181980599,
               0.193445624579018, 0.198570743497266, 0.194261846812406, 0.207044157175978,
               0.189609423104717, 0.191702378731519, 0.192177559097695, 0.181476713235114,
               0.194027784013463, 0.200179706718845, 0.184131250493017, 0.187148059297445,
               0.170093327206757, 0.172576640408774, 0.17142849179367, 0.171210509633908,
               0.175266825046856, 0.178539448350141, 0.183297530664574, 0.179988260785274,
               0.189456949097769, 0.168449442456919, 0.169472245339963, 0.156980987739318,
               0.159741009534041, 0.192393434626155, 0.183185104723854, 0.183262945862352,
               0.18484364782403, 0.197853263566926, 24.2660940420227, 92.5425257213213,
               0.21293050652165, 0.181916377799167, 0.207576728988578, 0.191684903885024,
               0.197325237445212, 0.193040594600961, 0.204938476255653, 0.213088225197018,
               0.207130518027665, 0.193969956301053, 0.197819188836235, 0.191574021643488,
               0.188361649036137, 0.200945515201322, 0.191129945975644, 0.189668274442045,
               0.216204823546794, 0.194440363568208, 0.188000042388181, 27.8327311737666,
               25.8798805330205, 1.82019090684725, 4.53925724807656, 2.4652260290996,
               6.30282987279155, 1.22514846755256, 7.02193377581181, 0.341993326634691,
               1.46247385102774, 1.70176087365108, 0.175754378191942, 0.158607605784771,
               9.64957304344759, 29.1799404301678, 23.081926013577, 8.5314822522917,
               4.67114811854655, 20.2068643456479, 0.125339430795293, 21.627018859762,
               174.354573717964, 126.837369498721, 127.965072848055, 59.3323268118175,
               225.054379393871, 167.380244625932, 190.544116071728, 152.114641694906,
               193.203179246339, 261.547968376433, 76.8867659828262, 39.3438777798031,
               50.4200823469192, 111.519653694684, 67.7108666550488, 32.7766089359556,
               9.92065729415368, 30.8161243309039, 64.2664505870036, 7.77905916495073,
               8.70181128589493, 15.2783421853377, 24.2812276220387, 33.2459872032365,
               74.6514899696842, 124.653730158129, 134.268557991693, 233.242392830958,
               177.356548203902, 156.434389081044, 207.077705083092, 189.052515918887,
               60.6499237760748, 96.7782272467804, 74.7880066303965, 25.3060577703537,
               180.099786225049, 295.579114642314, 145.29938865315, 315.478815661473,
               174.735904208625, 119.465435230861, 96.3289129884367, 11.0722170567145,
               2.05061396369785, 3.23497513032764, 47.9920127773772, 57.7909183871524,
               227.205690685301, 127.324201771323, 26.1377861039035, 256.999318581328,
               132.661489816869, 54.1982220041114, 62.9107723645602, 25.3290921420468,
               3.8723178727349, 16.6583658625875, 35.2885141717374, 79.6580156247823,
               70.7689699720557, 37.4478597529742
            ),
            mean_abs_Gyroscope_z_DegPerS = c(
              0.110515421016894,
               0.101239605089746, 0.0979828479057211, 0.102426227052763, 0.109939905020029,
               0.102613303156313, 0.104245619063607, 0.112557264924237, 0.114440810258187,
               0.114954982002404, 0.108595720040794, 0.1118270929181, 0.11826003896658,
               0.121622218615796, 0.120224307974598, 0.106196484561323, 0.113038307625163,
               0.109772195869976, 0.11002443959562, 0.0969752118659663, 0.101266879278782,
               0.0971063087456421, 0.101630219757211, 0.105415480653577, 0.117743788760772,
               0.122374053735431, 0.115248455758584, 0.122850453210866, 0.107749545932967,
               0.111841298011031, 0.108928261332879, 0.0989279428894108, 0.110387719560607,
               0.106683758664703, 0.102571802633939, 0.106848057505069, 0.110057275975392,
               0.104638154771382, 0.120550964633413, 0.120320021708351, 0.127535667823401,
               0.114029020263054, 0.121398096882498, 0.111882208977363, 0.113520800578213,
               0.102651761394368, 0.103120814847192, 0.103206187560639, 0.107475496651657,
               0.106738363629684, 0.11312884490032, 0.10969405608213, 0.102072794293974,
               0.107307996803024, 0.111112191638853, 0.106398672609295, 0.105524179250641,
               0.108530313738042, 0.109750628898196, 0.110751792882821, 0.112977240553386,
               0.107985188772158, 0.0981443314582239, 0.100310416146015, 0.112871252451278,
               0.1128015554624, 0.110520293801771, 0.105670484723798, 0.0989024837547816,
               0.113533246690887, 0.104349700720623, 0.109880522477072, 0.10803699213382,
               0.10627686368338, 0.101994531968083, 10.7193504326357, 6.56701237386481,
               0.103214520626945, 0.089204465863778, 0.086232133352401, 0.0868909384331619,
               0.0889820560413641, 0.0908619658322952, 0.0911911555568878, 0.0850313331116407,
               0.0836901488493204, 0.0867692277299233, 0.0915481789200068, 0.0865768770690454,
               0.0890986724809528, 0.0850898859320979, 0.0812542133569488, 0.0811235482333314,
               0.0864889200187133, 0.0798277977021007, 0.0802211140814032, 84.8254989646595,
               9.86585645529622, 0.321770976089029, 0.545656578743503, 0.362689636176952,
               1.21821172799028, 0.26574705851175, 1.18963059950045, 0.128650632067397,
               0.193608954114888, 0.166921475083429, 0.122866832452583, 0.108541750039544,
               1.32157358764783, 35.6869028410964, 29.7545632505333, 508.446828862949,
               175.356026498701, 1061.95577179041, 0.103807849856288, 39.1285141379051,
               32.0477277741476, 18.9671766369848, 21.9006364949719, 35.1578932818053,
               15.3499797401105, 21.4590209790646, 40.0933105250392, 20.428470316885,
               19.0270384969101, 19.0689050220441, 32.106066794063, 43.1813400998584,
               33.4175353299093, 63.5028380903444, 45.2117462656204, 25.6536184502544,
               4.36813988297761, 22.8352720080152, 33.035331292747, 2.82151896672089,
               11.1201627934286, 233.497308317196, 197.100412955139, 192.686742761626,
               231.131688914835, 224.735608640718, 44.9040055393206, 38.1965852464009,
               42.7425626396463, 25.4398413417096, 24.7221047526954, 38.099694547553,
               41.7444929622397, 84.9328184567375, 74.0202239432893, 24.6386466407219,
               21.3549189033728, 31.9629413131025, 27.5178690690405, 21.4069494777798,
               26.5591248910898, 58.9714519398685, 39.2890238645865, 12.0114798839395,
               0.701422532843144, 0.975459693567736, 22.237118405261, 69.375227130129,
               30.3007687696152, 18.8835181299884, 2.67460999592127, 34.1461794712612,
               33.6420441217819, 51.5676455766723, 27.4932068728163, 9.74319532464094,
               2.27625229984199, 25.5231988525248, 206.075279565199, 156.593074147075,
               41.5093892970447, 38.3314537495471
            ),
            mean_magnetometer_direction = c(
              "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE",
              "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE",
              "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE",
              "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE",
              "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE",
              "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE",
              "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "SE", "N",
              "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N",
              "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N",
              "N", "N", "N", "N", "N", "N", "N", "N", "WNW", "WNW", "WNW", "WNW",
              "WNW", "NNW", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N",
              "NW", "N", "N", "N", "N", "NNW", "NW", "WNW", "N", "NNW", "NNW",
              "N", "N", "N", "N", "N", "NNW", "N", "N", "N", "N", "N", "N",
              "N", "N", "NNW", "N", "N", "N", "N", "N", "N", "N", "N", "NNW",
              "NW", "NNW", "N", "N", "N", "NW", "N", "NNW", "N", "N", "N",
              "N", "NNE", "NNW", "NNW", "N", "N"
            )
          ),
          class = "data.frame",
          row.names = c(NA, -179L)
        )
      )


  })
