% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rank_regression.R
\encoding{UTF-8}
\name{rank_regression}
\alias{rank_regression}
\alias{rank_regression.wt_cdf_estimation}
\title{Rank Regression for Parametric Lifetime Distributions}
\usage{
rank_regression(x, ...)

\method{rank_regression}{wt_cdf_estimation}(
  x,
  distribution = c("weibull", "lognormal", "loglogistic", "normal", "logistic", "sev",
    "weibull3", "lognormal3", "loglogistic3"),
  conf_level = 0.95,
  ...
)
}
\arguments{
\item{x}{Object of class \code{wt_cdf_estimation} returned from
\code{\link{estimate_cdf}}.}

\item{...}{Further arguments passed to or from other methods. Currently not used.}

\item{distribution}{Supposed distribution of the random variable.}

\item{conf_level}{Confidence level of the interval. If \code{distribution} is
\code{"weibull"} this must be one of \code{0.9}, \code{0.95} or \code{0.99}.}
}
\value{
Returns a list with classes \code{wt_model}, \code{wt_rank_regression} and
\code{wt_model_estimation} containing the following elements:
\itemize{
  \item \code{coefficients} : A named vector of estimated coefficients
    (parameters of the assumed distribution). \strong{Note}: The parameters
    are given in location-scale-parameterization.
  \item \code{confint} : Confidence intervals for parameters. If
    \code{distribution} is \code{"lognormal3"} or \code{"loglogistic3"} no
    confidence interval for the threshold parameter is computed.
  \item \code{varcov} : Provided, if \code{distribution} is not
    \code{"weibull"} or \code{"weibull3"}. Estimated heteroscedasticity-consistent
    variance-covariance matrix for the (log-)location-scale parameters.
  \item \code{shape_scale_coefficients} : Only included if
    \code{distribution} is \code{"weibull"} or \code{"weibull3"}
    (parameterization used in \code{\link[stats:Weibull]{stats::Weibull}}).
  \item \code{shape_scale_confint} : Only included if \code{distribution} is
    \code{"weibull"} or \code{"weibull3"}. Approximated confidence intervals
    for scale \eqn{\eta} and shape \eqn{\beta} (and threshold \eqn{\gamma})
    if \code{distribution} is \code{"weibull3"}.
  \item \code{r_squared} : Coefficient of determination.
  \item \code{data} : A tibble with class \code{wt_cdf_estimation} returned from \code{\link{estimate_cdf}}.
  \item \code{distribution} : Specified distribution.
}

If more than one method was specified in \code{\link{estimate_cdf}},
  the resulting output is a list with class \code{wt_model_estimation_list}.
  In this case each list element has classes \code{wt_rank_regression} and
  \code{wt_model_estimation} and the items listed above, are included.
}
\description{
This function fits an \strong{x on y} regression to a linearized two- or
three-parameter lifetime distribution for complete and (multiple) right
censored data. The parameters are determined in the frequently used
(log-)location-scale parameterization.

For the Weibull, estimates are additionally transformed such that they are in
line with the parameterization provided by the \emph{stats} package
(see \link[stats]{Weibull}).
}
\details{
If \code{distribution} is \code{"weibull"} or \code{"weibull3"}, the approximated
confidence intervals for the parameters can only be estimated on the following
confidence levels (see 'References' \emph{(Mock, 1995)}):
\itemize{
  \item \code{conf_level} = 0.90,
  \item \code{conf_level} = 0.95,
  \item \code{conf_level} = 0.99.
}

If the distribution is not the Weibull, the confidence intervals of the
parameters are computed on the basis of a heteroscedasticity-consistent
covariance matrix. Here it should be said that there is no statistical foundation
to determine the standard errors of the parameters using \emph{Least Squares}
in context of \emph{Rank Regression}. For an accepted statistical method use
\link[=ml_estimation]{maximum likelihood}.
}
\examples{
# Reliability data preparation:
## Data for two-parametric model:
data_2p <- reliability_data(
  shock,
  x = distance,
  status = status
)

## Data for three-parametric model:
data_3p <- reliability_data(
  alloy,
  x = cycles,
  status = status
)

# Probability estimation:
prob_tbl_2p <- estimate_cdf(
  data_2p,
  methods = "johnson"
)

prob_tbl_3p <- estimate_cdf(
  data_3p,
  methods = "johnson"
)

prob_tbl_mult <- estimate_cdf(
  data_3p,
  methods = c("johnson", "kaplan")
)

# Example 1 - Fitting a two-parametric weibull distribution:
rr_2p <- rank_regression(
  x = prob_tbl_2p,
  distribution = "weibull"
)

# Example 2 - Fitting a three-parametric lognormal distribution:
rr_3p <- rank_regression(
  x = prob_tbl_3p,
  distribution = "lognormal3",
  conf_level = 0.99
)

# Example 3 - Fitting a three-parametric loglogistic distribution if multiple
# methods in estimate_cdf were specified:
rr_lists <- rank_regression(
  x = prob_tbl_mult,
  distribution = "loglogistic3",
  conf_level = 0.90
)

}
\references{
\itemize{
    \item Mock, R., Methoden zur Datenhandhabung in
  Zuverlässigkeitsanalysen, vdf Hochschulverlag AG an der ETH Zürich, 1995
    \item Meeker, William Q; Escobar, Luis A., Statistical methods for
  reliability data, New York: Wiley series in probability and statistics, 1998}
}
