% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trajectoryDistances.R
\encoding{UTF-8}
\name{trajectories}
\alias{trajectories}
\alias{segmentDistances}
\alias{trajectoryDistances}
\alias{trajectoryLengths}
\alias{trajectoryAngles}
\alias{trajectoryPCoA}
\alias{trajectoryProjection}
\alias{trajectoryConvergence}
\alias{trajectoryDirectionality}
\alias{centerTrajectories}
\alias{trajectorySelection}
\alias{trajectoryPlot}
\alias{is.metric}
\title{Community trajectory analysis}
\usage{
segmentDistances(d, sites, surveys = NULL,
  distance.type = "directed-segment", add = TRUE, verbose = FALSE)

trajectoryDistances(d, sites, surveys = NULL, distance.type = "DSPD",
  symmetrization = "mean", add = TRUE, verbose = FALSE)

trajectoryLengths(d, sites, surveys = NULL, verbose = FALSE)

trajectoryAngles(d, sites, surveys = NULL, all = FALSE, stats = TRUE,
  add = TRUE, verbose = FALSE)

trajectorySelection(d, sites, selection)

trajectoryPCoA(d, sites, surveys = NULL, selection = NULL,
  traj.colors = NULL, axes = c(1, 2), ...)

trajectoryPlot(x, sites, surveys = NULL, selection = NULL,
  traj.colors = NULL, axes = c(1, 2), ...)

trajectoryProjection(d, target, trajectory, tol = 1e-06, add = TRUE)

trajectoryConvergence(d, sites, surveys = NULL, symmetric = FALSE,
  add = TRUE, verbose = FALSE)

trajectoryDirectionality(d, sites, surveys = NULL, add = TRUE,
  verbose = FALSE)

centerTrajectories(d, sites, verbose = FALSE)

is.metric(d, tol = 1e-04)
}
\arguments{
\item{d}{A symmetric \code{\link{matrix}} or an object of class \code{\link{dist}} containing the distance values between pairs of community states (see details).}

\item{sites}{A vector indicating the site corresponding to each community state.}

\item{surveys}{A vector indicating the survey corresponding to each community state (only necessary when surveys are not in order).}

\item{distance.type}{The type of distance index to be calculated (Besse et al. 2016; De Cáceres et al. submitted). For \code{segmentDistances} the available indices are:
  \itemize{
    \item{\code{Hausdorff}: Hausdorff distance between two segments.}
    \item{\code{directed-segment}: Directed segment distance (default).}
    \item{\code{PPA}: Perpendicular-parallel-angle distance.}
  }
whereas for \code{trajectoryDistances} the available indices are:
  \itemize{
    \item{\code{Hausdorff}: Hausdorff distance between two trajectories.}
    \item{\code{SPD}: Segment path distance.}
    \item{\code{DSPD}: Directed segment path distance (default).}
  }}

\item{add}{Flag to indicate that constant values should be added (local transformation) to correct triplets of distance values that do not fulfill the triangle inequality.}

\item{verbose}{Provides console output informing about process (useful for large dataset).}

\item{symmetrization}{Function used to obtain a symmetric distance, so that DSPD(T1,T2) = DSPD(T2,T1) (e.g., \code{mean} or \code{min}). If \code{symmetrization = NULL} then the symmetrization is not conducted and the output dissimilarity matrix is not symmetric.}

\item{all}{A flag to indicate that angles are desired for all triangles (i.e. all pairs of segments) in the trajectory. If FALSE, angles are calculated for consecutive segments only.}

\item{stats}{A flag to indicate that circular statistics are desired (mean, standard deviation and mean resultant length, i.e. rho)}

\item{selection}{A character vector of sites, a numeric vector of site indices or logical vector of the same length as \code{sites}, indicating a subset of site trajectories to be selected.}

\item{traj.colors}{A vector of colors (one per site). If \code{selection != NULL} the length of the color vector should be equal to the number of sites selected.}

\item{axes}{The pair of principal coordinates to be plotted.}

\item{...}{Additional parameters for function \code{\link{arrows}}.}

\item{x}{A data.frame or matrix where rows are community states and columns are coordinates in an arbitrary space}

\item{target}{An integer vector of the community states to be projected.}

\item{trajectory}{An integer vector of the trajectory onto which target states are to be projected.}

\item{tol}{Numerical tolerance value to determine that projection of a point lies within the trajectory.}

\item{symmetric}{A logical flag to indicate a symmetric convergence comparison of trajectories.}
}
\value{
Function \code{trajectoryDistances} returns an object of class \code{\link{dist}} containing the distances between trajectories (if \code{symmetrization = NULL} then the object returned is of class \code{matrix}). 

Function \code{trajectorySegments} returns a list with the following elements:
\itemize{
  \item{\code{Dseg}: Distance matrix between segments.}
  \item{\code{Dini}: Distance matrix between initial points of segments.}
  \item{\code{Dfin}: Distance matrix between final points of segments.}
  \item{\code{Dinifin}: Distance matrix between initial points of one segment and the final point of the other.}
  \item{\code{Dfinini}: Distance matrix between final points of one segment and the initial point of the other.}
}

Function \code{trajectoryLengths} returns a data frame with the length of each segment on each trajectory and the total length of all trajectories. Function \code{trajectoryPCoA} returns the result of calling \code{\link{cmdscale}}.

Function \code{trajectoryAngles} returns a data frame with angle values on each trajectory. If \code{stats=TRUE}, then the mean, standard deviation and mean resultant length of those angles are also returned. 

Function \code{trajectoryPCoA} returns the result of calling \code{\link{cmdscale}}.

Function \code{trajectoryProjection} returns a data frame with the following columns:
\itemize{
  \item{\code{distanceToTrajectory}: Distances to the trajectory, i.e. rejection (\code{NA} for target points whose projection is outside the trajectory).}
  \item{\code{segment}: Segment that includes the projected point (\code{NA} for target points whose projection is outside the trajectory).}
  \item{\code{relativePosition}: Relative position of the projected point within the trajectory, i.e. values from 0 to 1 with 0 representing the start of the trajectory and 1 representing the end (\code{NA} for target points whose projection is outside the trajectory).}
}

Function \code{trajectoryConvergence} returns a list with two elements:
\itemize{
  \item{\code{tau}: A matrix with the statistic (Mann-Kendall's tau) of the convergence/divergence test between trajectories. If \code{symmetric=TRUE} then the matrix is square. Otherwise the statistic of the test of the row trajectory approaching the column trajectory.}
  \item{\code{p.value}: A matrix with the p-value of the convergence/divergence test between trajectories. If \code{symmetric=TRUE} then the matrix is square. Otherwise the p-value indicates the test of the row trajectory approaching the column trajectory.}
}

Function \code{trajectoryDirectionality} returns a vector with directionality values (one per trajectory).

Function \code{centerTrajectory} returns an object of class \code{\link{dist}}.
}
\description{
Community trajectory analysis (CTA) is a framework to analyze community dynamics described as trajectories in a chosen space of community resemblance (De \enc{Cáceres}{Caceres} et al. 2019).
CTA takes trajectories as objects to be analyzed and compared geometrically. Given a distance matrix between community states, the set of functions for CTA are:
\itemize{
\item{Functions \code{segmentDistances} and \code{trajectoryDistances} calculate the distance between pairs of directed segments and community trajectories, respectively.}
\item{Function \code{trajectoryLengths} calculates lengths of directed segments and total path lengths of trajectories.}
\item{Function \code{trajectoryAngles} calculates the angle between consecutive pairs of directed segments or between segments of ordered triplets of points.}
\item{Function \code{trajectoryPCoA} performs principal coordinates analysis (\code{\link{cmdscale}}) and draws trajectories in the ordination scatterplot.}
\item{Function \code{trajectoryPlot} Draws trajectories in a scatterplot corresponding to the input coordinates.}
\item{Function \code{trajectoryProjection} projects a set of target points onto a specified trajectory and returns the distance to the trajectory (i.e. rejection) and the relative position of the projection point within the trajectory.}
\item{Function \code{trajectoryConvergence} performs the Mann-Kendall trend test on the distances between trajectories (symmetric test) or the distance between points of one trajectory to the other.}
\item{Function \code{trajectorySelection} allows selecting the submatrix of distances corresponding to a given subset of trajectories.}
\item{Function \code{trajectoryDirectionality} returns (for each trajectory) a statistic that measures directionality of the whole trajectory.}
\item{Function \code{centerTrajectories} shifts all trajectories to the center of the compositional space and returns a modified distance matrix.}
\item{Function \code{is.metric} checks whether the input dissimilarity matrix is metric (i.e. all triplets fulfill the triangle inequality).}
}
}
\details{
Details of calculations are given in De \enc{Cáceres}{Caceres} et al (submitted). 
The input distance matrix \code{d} should ideally be metric. That is, all subsets of distance triplets should fulfill the triangle inequality (see function \code{is.metric}). 
All CTA functions that require metricity include a parameter '\code{add}', which by default is TRUE, meaning that whenever the triangle inequality is broken the minimum constant required to fulfill it is added to the three distances.
If such local (an hence, inconsistent across triplets) corrections are not desired, users should find another way modify \code{d} to achieve metricity, such as PCoA, metric MDS or non-metric MDS (see CTA vignette). 
If parameter '\code{add}' is set to FALSE and problems of triangle inequality exist, CTA functions may provide missing values in some cases where they should not.

The resemblance between trajectories is done by adapting concepts and procedures used for the analysis of trajectories in space (i.e. movement data) (Besse et al. 2016).   

Function \code{trajectoryAngles} calculates angles between consecutive segments (or between the segments corresponding to all ordered triplets) in degrees. For each pair of segments, the angle between the two is defined on the plane that contains the two segments, and measures the change in direction (in degrees) from one segment to the other. 
Angles are always positive, with zero values indicating segments that are in a straight line, and values equal to 180 degrees for segments that are in opposite directions.

Function \code{centerTrajectories} performs centering of trajectories using matrix algebra as explained in Anderson (2017).
}
\examples{
  #Description of sites and surveys
  sites = c(1,1,1,2,2,2)
  surveys=c(1,2,3,1,2,3)
  
  #Raw data table
  xy<-matrix(0, nrow=6, ncol=2)
  xy[2,2]<-1
  xy[3,2]<-2
  xy[4:6,1] <- 0.5
  xy[4:6,2] <- xy[1:3,2]
  xy[6,1]<-1
  
  #Distance matrix
  d = dist(xy)
  d
  
  trajectoryLengths(d, sites, surveys)
  trajectoryAngles(d, sites, surveys)
  segmentDistances(d, sites, surveys)$Dseg
  trajectoryDistances(d, sites, surveys, distance.type = "Hausdorff")
  trajectoryDistances(d, sites, surveys, distance.type = "DSPD")
  
  #Draw trajectories
  trajectoryPCoA(d, sites, traj.colors = c("black","red"), lwd = 2)
  
  
  #Should give the same results if surveys are not in order 
  #(here we switch surveys for site 2)
  temp = xy[5,]
  xy[5,] = xy[6,]
  xy[6,] = temp
  surveys[5] = 3
  surveys[6] = 2
  trajectoryLengths(dist(xy), sites, surveys)
  segmentDistances(dist(xy), sites, surveys)$Dseg
  trajectoryDistances(dist(xy), sites, surveys, distance.type = "Hausdorff")
  trajectoryDistances(dist(xy), sites, surveys, distance.type = "DSPD")
 
}
\references{
Besse, P., Guillouet, B., Loubes, J.-M. & François, R. (2016). Review and perspective for distance based trajectory clustering. IEEE Trans. Intell. Transp. Syst., 17, 3306–3317.

De \enc{Cáceres}{Caceres} M, Coll L, Legendre P, Allen RB, Wiser SK, Fortin MJ, Condit R & Hubbell S. (2019). Trajectory analysis in community ecology. Ecological Monographs.

Anderson (2017). Permutational Multivariate Analysis of Variance (PERMANOVA). Wiley StatsRef: Statistics Reference Online. 1-15. Article ID: stat07841.
}
\seealso{
\code{\link{cmdscale}}
}
\author{
Miquel De \enc{Cáceres}{Caceres}, Forest Sciences Center of Catalonia
}
