% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pb_regression.R
\name{pb_regression}
\alias{pb_regression}
\title{Passing-Bablok Regression for Method Comparison}
\usage{
pb_regression(
  x,
  y = NULL,
  data = NULL,
  conf_level = 0.95,
  ci_method = c("analytical", "bootstrap"),
  boot_n = 1999,
  na_action = c("omit", "fail")
)
}
\arguments{
\item{x}{Numeric vector of measurements from method 1 (reference method),
or a formula of the form \code{method1 ~ method2}.}

\item{y}{Numeric vector of measurements from method 2 (test method).
Ignored if \code{x} is a formula.}

\item{data}{Optional data frame containing the variables specified in
\code{x} and \code{y} (or in the formula).}

\item{conf_level}{Confidence level for intervals (default: 0.95).}

\item{ci_method}{Method for calculating confidence intervals:
\code{"analytical"} (default) uses the original Passing-Bablok (1983) method,
\code{"bootstrap"} uses BCa bootstrap resampling.}

\item{boot_n}{Number of bootstrap resamples when \code{ci_method = "bootstrap"}
(default: 1999).}

\item{na_action}{How to handle missing values: \code{"omit"} (default) removes
pairs with any NA, \code{"fail"} stops with an error.}
}
\value{
An object of class \code{c("pb_regression", "valytics_comparison", "valytics_result")},
which is a list containing:

\describe{
\item{input}{List with original data and metadata:
\itemize{
\item \code{x}, \code{y}: Numeric vectors (after NA handling)
\item \code{n}: Number of paired observations
\item \code{n_excluded}: Number of pairs excluded due to NAs
\item \code{var_names}: Named character vector with variable names
}
}
\item{results}{List with statistical results:
\itemize{
\item \code{intercept}: Intercept point estimate
\item \code{slope}: Slope point estimate
\item \code{intercept_ci}: Named numeric vector with lower and upper CI
\item \code{slope_ci}: Named numeric vector with lower and upper CI
\item \code{residuals}: Perpendicular residuals
\item \code{fitted_x}: Fitted x values
\item \code{fitted_y}: Fitted y values
}
}
\item{cusum}{List with CUSUM linearity test results (if calculable):
\itemize{
\item \code{statistic}: CUSUM test statistic
\item \code{critical_value}: Critical value at alpha = 0.05
\item \code{p_value}: Approximate p-value
\item \code{linear}: Logical; TRUE if linearity assumption holds
}
}
\item{settings}{List with analysis settings:
\itemize{
\item \code{conf_level}: Confidence level used
\item \code{ci_method}: CI method used
\item \code{boot_n}: Number of bootstrap samples (if applicable
}
}
\item{call}{The matched function call.}
}
}
\description{
Performs Passing-Bablok regression to assess agreement between two measurement
methods. This non-parametric regression method is robust to outliers and does
not assume normally distributed errors. The implementation uses a fast
O(n log n) algorithm from the \code{robslopes} package for point estimation.
}
\details{
Passing-Bablok regression is a non-parametric method for fitting a linear
relationship between two measurement methods. Unlike ordinary least squares,
it:

\itemize{
\item Makes no assumptions about error distribution
\item Accounts for measurement error in both variables
\item Is robust to outliers
\item Produces results independent of which variable is assigned to X or Y
(when using the equivariant form)
}

The slope is estimated as the median of all pairwise slopes (in absolute
value for the equivariant version), and the intercept is the median of
\code{y - slope * x}.
}
\section{Interpretation}{

\itemize{
\item \strong{Slope = 1}: No proportional difference between methods
\item \strong{Slope != 1}: Proportional (multiplicative) difference exists
\item \strong{Intercept = 0}: No constant difference between methods
\item \strong{Intercept != 0}: Constant (additive) difference exists
}

Use the confidence intervals to test these hypotheses: if 1 is within the
slope CI and 0 is within the intercept CI, the methods are considered
equivalent.
}

\section{Assumptions}{

\itemize{
\item Linear relationship between X and Y (test with CUSUM)
\item Measurement range covers the intended clinical range
\item Data are continuously distributed
}
}

\section{CUSUM Test for Linearity}{

The CUSUM (cumulative sum) test checks the linearity assumption. A
significant result (p < 0.05) suggests non-linearity, and Passing-Bablok
regression may not be appropriate.
}

\examples{
# Simulated method comparison data
set.seed(42)
method_a <- rnorm(50, mean = 100, sd = 15)
method_b <- 1.05 * method_a + 3 + rnorm(50, sd = 5)  # slope=1.05, intercept=3

# Basic analysis
pb <- pb_regression(method_a, method_b)
pb

# Using formula interface with data frame
df <- data.frame(reference = method_a, test = method_b)
pb <- pb_regression(reference ~ test, data = df)

# With bootstrap confidence intervals
pb_boot <- pb_regression(method_a, method_b, ci_method = "bootstrap")

# Using package example data
data(glucose_methods)
pb <- pb_regression(reference ~ poc_meter, data = glucose_methods)
summary(pb)
plot(pb)

}
\references{
Passing H, Bablok W (1983). A new biometrical procedure for testing the
equality of measurements from two different analytical methods. Application
of linear regression procedures for method comparison studies in clinical
chemistry, Part I. \emph{Journal of Clinical Chemistry and Clinical
Biochemistry}, 21(11):709-720. \doi{10.1515/cclm.1983.21.11.709}

Passing H, Bablok W (1984). Comparison of several regression procedures for
method comparison studies and determination of sample sizes. Application of
linear regression procedures for method comparison studies in Clinical
Chemistry, Part II. \emph{Journal of Clinical Chemistry and Clinical
Biochemistry}, 22(6):431-445. \doi{10.1515/cclm.1984.22.6.431}

Bablok W, Passing H, Bender R, Schneider B (1988). A general regression
procedure for method transformation. Application of linear regression
procedures for method comparison studies in clinical chemistry, Part III.
\emph{Journal of Clinical Chemistry and Clinical Biochemistry},
26(11):783-790. \doi{10.1515/cclm.1988.26.11.783}

Raymaekers J, Dufey F (2022). Equivariant Passing-Bablok regression in
quasilinear time. \emph{arXiv preprint}. \doi{10.48550/arXiv.2202.08060}
}
\seealso{
\code{\link[=plot.pb_regression]{plot.pb_regression()}} for visualization,
\code{\link[=summary.pb_regression]{summary.pb_regression()}} for detailed summary,
\code{\link[=ba_analysis]{ba_analysis()}} for Bland-Altman analysis
}
