% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gum.R
\name{gum}
\alias{gum}
\alias{ges}
\title{Generalised Univariate Model}
\usage{
gum(y, orders = c(1, 1), lags = c(1, frequency(y)),
  type = c("additive", "multiplicative"), persistence = NULL,
  transition = NULL, measurement = NULL, initial = c("optimal",
  "backcasting"), ic = c("AICc", "AIC", "BIC", "BICc"), loss = c("MSE",
  "MAE", "HAM", "MSEh", "TMSE", "GTMSE", "MSCE"), h = 10,
  holdout = FALSE, cumulative = FALSE, interval = c("none",
  "parametric", "likelihood", "semiparametric", "nonparametric"),
  level = 0.95, occurrence = c("none", "auto", "fixed", "general",
  "odds-ratio", "inverse-odds-ratio", "direct"), oesmodel = "MNN",
  bounds = c("restricted", "admissible", "none"), silent = c("all",
  "graph", "legend", "output", "none"), xreg = NULL, xregDo = c("use",
  "select"), initialX = NULL, updateX = FALSE, persistenceX = NULL,
  transitionX = NULL, ...)

ges(...)
}
\arguments{
\item{y}{Vector or ts object, containing data needed to be forecasted.}

\item{orders}{Order of the model. Specified as vector of number of states
with different lags. For example, \code{orders=c(1,1)} means that there are
two states: one of the first lag type, the second of the second type.}

\item{lags}{Defines lags for the corresponding orders. If, for example,
\code{orders=c(1,1)} and lags are defined as \code{lags=c(1,12)}, then the
model will have two states: the first will have lag 1 and the second will
have lag 12. The length of \code{lags} must correspond to the length of
\code{orders}.}

\item{type}{Type of model. Can either be \code{"A"} - additive - or
\code{"M"} - multiplicative. The latter means that the GUM is fitted on
log-transformed data.}

\item{persistence}{Persistence vector \eqn{g}, containing smoothing
parameters. If \code{NULL}, then estimated.}

\item{transition}{Transition matrix \eqn{F}. Can be provided as a vector.
Matrix will be formed using the default \code{matrix(transition,nc,nc)},
where \code{nc} is the number of components in state vector. If \code{NULL},
then estimated.}

\item{measurement}{Measurement vector \eqn{w}. If \code{NULL}, then
estimated.}

\item{initial}{Can be either character or a vector of initial states. If it
is character, then it can be \code{"optimal"}, meaning that the initial
states are optimised, or \code{"backcasting"}, meaning that the initials are
produced using backcasting procedure.}

\item{ic}{The information criterion used in the model selection procedure.}

\item{loss}{The type of Loss Function used in optimization. \code{loss} can
be: \code{MSE} (Mean Squared Error), \code{MAE} (Mean Absolute Error),
\code{HAM} (Half Absolute Moment), \code{TMSE} - Trace Mean Squared Error,
\code{GTMSE} - Geometric Trace Mean Squared Error, \code{MSEh} - optimisation
using only h-steps ahead error, \code{MSCE} - Mean Squared Cumulative Error.
If \code{loss!="MSE"}, then likelihood and model selection is done based
on equivalent \code{MSE}. Model selection in this cases becomes not optimal.

There are also available analytical approximations for multistep functions:
\code{aMSEh}, \code{aTMSE} and \code{aGTMSE}. These can be useful in cases
of small samples.

Finally, just for fun the absolute and half analogues of multistep estimators
are available: \code{MAEh}, \code{TMAE}, \code{GTMAE}, \code{MACE}, \code{TMAE},
\code{HAMh}, \code{THAM}, \code{GTHAM}, \code{CHAM}.}

\item{h}{Length of forecasting horizon.}

\item{holdout}{If \code{TRUE}, holdout sample of size \code{h} is taken from
the end of the data.}

\item{cumulative}{If \code{TRUE}, then the cumulative forecast and prediction
interval are produced instead of the normal ones. This is useful for
inventory control systems.}

\item{interval}{Type of interval to construct. This can be:

\itemize{
\item \code{"none"}, aka \code{"n"} - do not produce prediction
interval.
\item \code{"parametric"}, \code{"p"} - use state-space structure of ETS. In
case of mixed models this is done using simulations, which may take longer
time than for the pure additive and pure multiplicative models. This type
of interval relies on unbiased estimate of in-sample error variance, which
divides the sume of squared errors by T-k rather than just T.
\item \code{"likelihood"}, \code{"l"} - these are the same as \code{"p"}, but
relies on the biased estimate of variance from the likelihood (division by
T, not by T-k).
\item \code{"semiparametric"}, \code{"sp"} - interval based on covariance
matrix of 1 to h steps ahead errors and assumption of normal / log-normal
distribution (depending on error type).
\item \code{"nonparametric"}, \code{"np"} - interval based on values from a
quantile regression on error matrix (see Taylor and Bunn, 1999). The model
used in this process is e[j] = a j^b, where j=1,..,h.
}
The parameter also accepts \code{TRUE} and \code{FALSE}. The former means that
parametric interval are constructed, while the latter is equivalent to
\code{none}.
If the forecasts of the models were combined, then the interval are combined
quantile-wise (Lichtendahl et al., 2013).}

\item{level}{Confidence level. Defines width of prediction interval.}

\item{occurrence}{The type of model used in probability estimation. Can be
\code{"none"} - none,
\code{"fixed"} - constant probability,
\code{"general"} - the general Beta model with two parameters,
\code{"odds-ratio"} - the Odds-ratio model with b=1 in Beta distribution,
\code{"inverse-odds-ratio"} - the model with a=1 in Beta distribution,
\code{"direct"} - the TSB-like (Teunter et al., 2011) probability update
mechanism a+b=1,
\code{"auto"} - the automatically selected type of occurrence model.}

\item{oesmodel}{The type of ETS model used for the modelling of the time varying
probability. Object of the class "oes" can be provided here, and its parameters
would be used in iETS model.}

\item{bounds}{What type of bounds to use in the model estimation. The first
letter can be used instead of the whole word.}

\item{silent}{If \code{silent="none"}, then nothing is silent, everything is
printed out and drawn. \code{silent="all"} means that nothing is produced or
drawn (except for warnings). In case of \code{silent="graph"}, no graph is
produced. If \code{silent="legend"}, then legend of the graph is skipped.
And finally \code{silent="output"} means that nothing is printed out in the
console, but the graph is produced. \code{silent} also accepts \code{TRUE}
and \code{FALSE}. In this case \code{silent=TRUE} is equivalent to
\code{silent="all"}, while \code{silent=FALSE} is equivalent to
\code{silent="none"}. The parameter also accepts first letter of words ("n",
"a", "g", "l", "o").}

\item{xreg}{The vector (either numeric or time series) or the matrix (or
data.frame) of exogenous variables that should be included in the model. If
matrix included than columns should contain variables and rows - observations.
Note that \code{xreg} should have number of observations equal either to
in-sample or to the whole series. If the number of observations in
\code{xreg} is equal to in-sample, then values for the holdout sample are
produced using \link[smooth]{es} function.}

\item{xregDo}{The variable defines what to do with the provided xreg:
\code{"use"} means that all of the data should be used, while
\code{"select"} means that a selection using \code{ic} should be done.
\code{"combine"} will be available at some point in future...}

\item{initialX}{The vector of initial parameters for exogenous variables.
Ignored if \code{xreg} is NULL.}

\item{updateX}{If \code{TRUE}, transition matrix for exogenous variables is
estimated, introducing non-linear interactions between parameters.
Prerequisite - non-NULL \code{xreg}.}

\item{persistenceX}{The persistence vector \eqn{g_X}, containing smoothing
parameters for exogenous variables. If \code{NULL}, then estimated.
Prerequisite - non-NULL \code{xreg}.}

\item{transitionX}{The transition matrix \eqn{F_x} for exogenous variables. Can
be provided as a vector. Matrix will be formed using the default
\code{matrix(transition,nc,nc)}, where \code{nc} is number of components in
state vector. If \code{NULL}, then estimated. Prerequisite - non-NULL
\code{xreg}.}

\item{...}{Other non-documented parameters.  For example parameter
\code{model} can accept a previously estimated GUM model and use all its
parameters.  \code{FI=TRUE} will make the function produce Fisher
Information matrix, which then can be used to calculated variances of
parameters of the model.
You can also pass two parameters to the optimiser: 1. \code{maxeval} - maximum
number of evaluations to carry on; 2. \code{xtol_rel} - the precision of the
optimiser. The default values used in es() are \code{maxeval=5000} and
\code{xtol_rel=1e-8}. You can read more about these parameters in the
documentation of \link[nloptr]{nloptr} function.}
}
\value{
Object of class "smooth" is returned. It contains:

\itemize{
\item \code{model} - name of the estimated model.
\item \code{timeElapsed} - time elapsed for the construction of the model.
\item \code{states} - matrix of fuzzy components of GUM, where \code{rows}
correspond to time and \code{cols} to states.
\item \code{initialType} - Type of the initial values used.
\item \code{initial} - initial values of state vector (extracted from
\code{states}).
\item \code{nParam} - table with the number of estimated / provided parameters.
If a previous model was reused, then its initials are reused and the number of
provided parameters will take this into account.
\item \code{measurement} - matrix w.
\item \code{transition} - matrix F.
\item \code{persistence} - persistence vector. This is the place, where
smoothing parameters live.
\item \code{fitted} - fitted values.
\item \code{forecast} - point forecast.
\item \code{lower} - lower bound of prediction interval. When
\code{interval="none"} then NA is returned.
\item \code{upper} - higher bound of prediction interval. When
\code{interval="none"} then NA is returned.
\item \code{residuals} - the residuals of the estimated model.
\item \code{errors} - matrix of 1 to h steps ahead errors.
\item \code{s2} - variance of the residuals (taking degrees of freedom
into account).
\item \code{interval} - type of interval asked by user.
\item \code{level} - confidence level for interval.
\item \code{cumulative} - whether the produced forecast was cumulative or not.
\item \code{y} - original data.
\item \code{holdout} - holdout part of the original data.
\item \code{occurrence} - model of the class "oes" if the occurrence model was estimated.
If the model is non-intermittent, then occurrence is \code{NULL}.
\item \code{xreg} - provided vector or matrix of exogenous variables. If
\code{xregDo="s"}, then this value will contain only selected exogenous variables.
\item \code{updateX} - boolean, defining, if the states of exogenous variables
were estimated as well.
\item \code{initialX} - initial values for parameters of exogenous variables.
\item \code{persistenceX} - persistence vector g for exogenous variables.
\item \code{transitionX} - transition matrix F for exogenous variables.
\item \code{ICs} - values of information criteria of the model. Includes
AIC, AICc, BIC and BICc.
\item \code{logLik} - log-likelihood of the function.
\item \code{lossValue} - Cost function value.
\item \code{loss} - Type of loss function used in the estimation.
\item \code{FI} - Fisher Information. Equal to NULL if \code{FI=FALSE} or
when \code{FI} variable is not provided at all.
\item \code{accuracy} - vector of accuracy measures for the holdout sample.
In case of non-intermittent data includes: MPE, MAPE, SMAPE, MASE, sMAE,
RelMAE, sMSE and Bias coefficient (based on complex numbers). In case of
intermittent data the set of errors will be: sMSE, sPIS, sCE (scaled
cumulative error) and Bias coefficient. This is available only when
\code{holdout=TRUE}.
}
}
\description{
Function constructs Generalised Univariate Model, estimating matrices F, w,
vector g and initial parameters.
}
\details{
The function estimates the Single Source of Error state space model of the
following type:

\deqn{y_{t} = o_{t} (w' v_{t-l} + x_t a_{t-1} + \epsilon_{t})}

\deqn{v_{t} = F v_{t-l} + g \epsilon_{t}}

\deqn{a_{t} = F_{X} a_{t-1} + g_{X} \epsilon_{t} / x_{t}}

Where \eqn{o_{t}} is the Bernoulli distributed random variable (in case of
normal data equal to 1), \eqn{v_{t}} is the state vector (defined using
\code{orders}) and \eqn{l} is the vector of \code{lags}, \eqn{x_t} is the
vector of exogenous parameters. \eqn{w} is the \code{measurement} vector,
\eqn{F} is the \code{transition} matrix, \eqn{g} is the \code{persistence}
vector, \eqn{a_t} is the vector of parameters for exogenous variables,
\eqn{F_{X}} is the \code{transitionX} matrix and \eqn{g_{X}} is the
\code{persistenceX} matrix. Finally, \eqn{\epsilon_{t}} is the error term.

For some more information about the model and its implementation, see the
vignette: \code{vignette("gum","smooth")}
}
\examples{

# Something simple:
gum(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,bounds="a",interval="p")

# A more complicated model with seasonality
\dontrun{ourModel <- gum(rnorm(118,100,3),orders=c(2,1),lags=c(1,4),h=18,holdout=TRUE)}

# Redo previous model on a new data and produce prediction interval
\dontrun{gum(rnorm(118,100,3),model=ourModel,h=18,interval="sp")}

# Produce something crazy with optimal initials (not recommended)
\dontrun{gum(rnorm(118,100,3),orders=c(1,1,1),lags=c(1,3,5),h=18,holdout=TRUE,initial="o")}

# Simpler model estiamted using trace forecast error loss function and its analytical analogue
\dontrun{gum(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,bounds="n",loss="TMSE")
gum(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,bounds="n",loss="aTMSE")}

# Introduce exogenous variables
\dontrun{gum(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,xreg=c(1:118))}

# Ask for their update
\dontrun{gum(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,xreg=c(1:118),updateX=TRUE)}

# Do the same but now let's shrink parameters...
\dontrun{gum(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,xreg=c(1:118),updateX=TRUE,loss="TMSE")
ourModel <- gum(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,loss="aTMSE")}

# Or select the most appropriate one
\dontrun{gum(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,xreg=c(1:118),xregDo="s")

summary(ourModel)
forecast(ourModel)
plot(forecast(ourModel))}

}
\references{
\itemize{
\item Svetunkov I. (2015 - Inf) "smooth" package for R - series of posts about the underlying
models and how to use them: \url{https://forecasting.svetunkov.ru/en/tag/smooth/}.
\item Svetunkov I. (2017). Statistical models underlying functions of 'smooth'
package for R. Working Paper of Department of Management Science, Lancaster
University 2017:1, 1-52.
}

\itemize{
\item Taylor, J.W. and Bunn, D.W. (1999) A Quantile Regression Approach to
Generating Prediction Intervals. Management Science, Vol 45, No 2, pp
225-237.
\item Lichtendahl Kenneth C., Jr., Grushka-Cockayne Yael, Winkler
Robert L., (2013) Is It Better to Average Probabilities or
Quantiles? Management Science 59(7):1594-1611. DOI:
[10.1287/mnsc.1120.1667](https://doi.org/10.1287/mnsc.1120.1667)
}
}
\seealso{
\code{\link[forecast]{ets}, \link[smooth]{es}, \link[smooth]{ces},
\link[smooth]{sim.es}}
}
\author{
Ivan Svetunkov, \email{ivan@svetunkov.ru}
}
\keyword{models}
\keyword{nonlinear}
\keyword{regression}
\keyword{smooth}
\keyword{ts}
\keyword{univar}
